import {
  HttpClient,
  HttpHeaders,
  HttpParams,
} from "@angular/common/http";
import {Injectable} from "@angular/core";
import {ArchiveStatisticsDto} from "@app/features/home/models/archive-statistics-dto.model";
import {HomeAction} from "@app/features/home/stores/home.action";
import {AccessResourceApiEnum} from "@app/shared/enums/api.enum";
import {LocalStateEnum} from "@app/shared/enums/local-state.enum";
import {urlSeparator} from "@app/shared/enums/routes.enum";
import {environment} from "@environments/environment";
import {HomeHelper} from "@home/helpers/home.helper";
import {Archive} from "@home/models/archive.model";
import {HomeArchiveDataFileState} from "@home/stores/archive/data-file/home-archive-data-file.state";
import {HomeArchivePackageState} from "@home/stores/archive/package/home-archive-package.state";
import {HomeArchiveRatingState} from "@home/stores/archive/rating/home-archive-rating.state";
import {Metadata} from "@models";
import {
  Action,
  Actions,
  Selector,
  State,
  StateContext,
  Store,
} from "@ngxs/store";
import {ApiActionEnum} from "@shared/enums/api-action.enum";
import {ViewModeTableEnum} from "@shared/enums/view-mode-table.enum";
import {ArchiveHelper} from "@shared/helpers/archive.helper";
import {Observable} from "rxjs";
import {
  catchError,
  tap,
} from "rxjs/operators";
import {
  ApiService,
  BasicState,
  CollectionTyped,
  defaultResourceStateInitValue,
  Facet,
  isEmptyString,
  isNullOrUndefined,
  MappingObject,
  MappingObjectUtil,
  NotificationService,
  QueryParameters,
  QueryParametersUtil,
  ResourceStateModel,
  SolidifyStateError,
  StoreUtil,
  StringUtil,
} from "solidify-frontend";

export const HOME_SEARCH_ALL: string = "*";
const DEFAULT_VIEW_MODE_TABLE: ViewModeTableEnum = environment.defaultHomeViewModeTableEnum;

export interface HomeStateModel extends ResourceStateModel<Archive> {
  search: string;
  facets: Facet[];
  viewModeTableEnum: ViewModeTableEnum;
  listRelativeArchive: Archive[];
  facetsSelected: MappingObject<string[]>;
  archiveStatisticDto: ArchiveStatisticsDto;
}

@Injectable()
@State<HomeStateModel>({
  name: LocalStateEnum.home,
  defaults: {
    ...defaultResourceStateInitValue(),
    search: StringUtil.stringEmpty,
    viewModeTableEnum: DEFAULT_VIEW_MODE_TABLE,
    queryParameters: new QueryParameters(environment.defaultPageSizeHomePage),
    listRelativeArchive: undefined,
    facets: undefined,
    facetsSelected: {},
    archiveStatisticDto: undefined,
  },
  children: [
    HomeArchiveDataFileState,
    HomeArchivePackageState,
    HomeArchiveRatingState,
    // HomeArchiveCollectionState,
  ],
})
export class HomeState extends BasicState<HomeStateModel> {
  protected get _urlResource(): string {
    return AccessResourceApiEnum.publicMetadata;
  }

  constructor(protected apiService: ApiService,
              protected store: Store,
              protected httpClient: HttpClient,
              protected notificationService: NotificationService,
              protected actions$: Actions) {
    super();
  }

  private readonly PATH_FILTER: string = "query";
  private readonly DEFAULT_SEARCH: string = HOME_SEARCH_ALL;

  @Selector()
  static current(state: HomeStateModel): Archive {
    return state.current;
  }

  @Selector()
  static search(state: HomeStateModel): string {
    return state.search;
  }

  @Selector()
  static isLoading(state: HomeStateModel): boolean {
    return StoreUtil.isLoadingState(state);
  }

  @Selector()
  static currentTitle(state: HomeStateModel): string | undefined {
    if (isNullOrUndefined(state.current)) {
      return undefined;
    }
    return state.current.title;
  }

  @Action(HomeAction.ChangeQueryParameters)
  changeQueryParameters(ctx: StateContext<HomeStateModel>, action: HomeAction.ChangeQueryParameters): void {
    ctx.patchState({
      queryParameters: action.queryParameters,
    });
    ctx.dispatch(new HomeAction.Search(false, ctx.getState().search, ctx.getState().facetsSelected));
  }

  @Action(HomeAction.Search)
  search(ctx: StateContext<HomeStateModel>, action: HomeAction.Search): Observable<CollectionTyped<Metadata>> {
    const searchString = this.getSearchStringToApply(ctx, action);
    const viewMode = this.getViewModeToApply(ctx, action);
    let queryParameters = StoreUtil.getQueryParametersToApply(action.queryParameters, ctx);
    if (action.resetPagination) {
      queryParameters = QueryParametersUtil.resetToFirstPage(queryParameters);
    }
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
      search: searchString,
      facetsSelected: action.facetsSelected,
      viewModeTableEnum: viewMode,
      queryParameters: queryParameters,
    });

    queryParameters = QueryParametersUtil.clone(queryParameters);
    MappingObjectUtil.set(QueryParametersUtil.getSearchItems(queryParameters), this.PATH_FILTER, isEmptyString(searchString) ? HOME_SEARCH_ALL : searchString);

    const facetsSelectedForBackend = HomeHelper.getFacetsSelectedForBackend(action.facetsSelected);
    const json = JSON.stringify(facetsSelectedForBackend);
    let headers: HttpHeaders = new HttpHeaders();
    headers = headers.set("Content-Type", "application/x-www-form-urlencoded");
    headers = headers.set("Accept", "application/json");
    const queryParametersHttp = ApiService.getQueryParameters(undefined, queryParameters);
    let body = new HttpParams();
    body = body.set("conditions", json);
    return this.httpClient.post<CollectionTyped<Metadata>>(this._urlResource + urlSeparator + ApiActionEnum.SEARCH,
      body,
      {
        headers,
        params: queryParametersHttp,
      },
    ).pipe(
      StoreUtil.cancelUncompleted(ctx, this.actions$, [HomeAction.Search]),
      tap((collection: CollectionTyped<Metadata>) => {
        const collectionArchive = {
          _links: collection._links,
          _page: collection._page,
          _facets: collection._facets,
        } as CollectionTyped<Archive>;
        collectionArchive._data = ArchiveHelper.adaptListArchivesMetadataInArchive(collection._data);
        ctx.dispatch(new HomeAction.SearchSuccess(collectionArchive));
      }),
      catchError(error => {
        ctx.dispatch(new HomeAction.SearchFail());
        throw error;
      }),
    );
  }

  private getSearchStringToApply(ctx: StateContext<HomeStateModel>, action: HomeAction.Search): string {
    if (isNullOrUndefined(action.search)) {
      return this.getDefaultSearchStringIfNotDefined(ctx);
    }
    return action.search;
  }

  private getDefaultSearchStringIfNotDefined(ctx: StateContext<HomeStateModel>): string {
    if (ctx.getState().search === null || ctx.getState().search === undefined) {
      return this.DEFAULT_SEARCH;
    }
    return ctx.getState().search;
  }

  @Action(HomeAction.SearchSuccess)
  searchSuccess(ctx: StateContext<HomeStateModel>, action: HomeAction.SearchSuccess): void {
    const queryParameters = StoreUtil.updateQueryParameters(ctx, action.collection);

    ctx.patchState({
      list: isNullOrUndefined(action.collection) ? [] : action.collection._data,
      total: isNullOrUndefined(action.collection) ? 0 : action.collection._page.totalItems,
      facets: isNullOrUndefined(action.collection) ? [] : action.collection._facets,
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
      queryParameters,
    });
  }

  @Action(HomeAction.SearchFail)
  searchFail(ctx: StateContext<HomeStateModel>): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
  }

  @Action(HomeAction.SearchRelativeArchive)
  searchRelativeArchive(ctx: StateContext<HomeStateModel>, action: HomeAction.SearchRelativeArchive): Observable<CollectionTyped<Metadata>> {
    const searchString = `organizationalunitid=${action.archive.organizationalUnitId}`;

    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
    });

    const queryParameters = new QueryParameters(5);
    MappingObjectUtil.set(QueryParametersUtil.getSearchItems(queryParameters), this.PATH_FILTER, searchString);

    return this.apiService.get<Metadata>(this._urlResource + urlSeparator + "search", queryParameters)
      .pipe(
        StoreUtil.cancelUncompleted(ctx, this.actions$, [HomeAction.Search]),
        tap((collection: any) => {
          collection._data = ArchiveHelper.adaptListArchivesMetadataInArchive(collection._data as Metadata[]);
          ctx.dispatch(new HomeAction.SearchRelativeArchiveSuccess(action, collection));
        }),
        catchError(error => {
          ctx.dispatch(new HomeAction.SearchRelativeArchiveFail(action));
          throw error;
        }),
      );
  }

  @Action(HomeAction.SearchRelativeArchiveSuccess)
  searchRelativeArchiveSuccess(ctx: StateContext<HomeStateModel>, action: HomeAction.SearchRelativeArchiveSuccess): void {
    ctx.patchState({
      listRelativeArchive: isNullOrUndefined(action.list) ? [] : action.list._data.filter(archive => archive.resId !== action.parentAction.archive.resId),
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
  }

  @Action(HomeAction.SearchRelativeArchiveFail)
  searchRelativeArchiveFail(ctx: StateContext<HomeStateModel>): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
  }

  @Action(HomeAction.SearchDetail)
  detail(ctx: StateContext<HomeStateModel>, action: HomeAction.SearchDetail): Observable<Metadata> {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
      current: undefined,
      listRelativeArchive: undefined,
    });

    return this.apiService.getById<Metadata>(this._urlResource, action.resId)
      .pipe(
        tap(model => ctx.dispatch(new HomeAction.SearchDetailSuccess(action, ArchiveHelper.adaptArchiveMetadataInArchive(model)))),
        catchError(error => {
          ctx.dispatch(new HomeAction.SearchDetailFail(action));
          throw error;
        }),
      );
  }

  @Action(HomeAction.SearchDetailSuccess)
  detailSuccess(ctx: StateContext<HomeStateModel>, action: HomeAction.SearchDetailSuccess): void {
    ctx.patchState({
      current: action.model,
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
    ctx.dispatch(new HomeAction.SearchRelativeArchive(action.model));
  }

  @Action(HomeAction.SearchDetailFail)
  detailFail(ctx: StateContext<HomeStateModel>, action: HomeAction.SearchDetailFail): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
  }

  @Action(HomeAction.CleanCurrent)
  cleanCurrent(ctx: StateContext<HomeStateModel>, action: HomeAction.CleanCurrent): void {
    ctx.patchState({
      current: undefined,
      archiveStatisticDto: undefined,
    });
  }

  private getViewModeToApply(ctx: StateContext<HomeStateModel>, action: HomeAction.Search): ViewModeTableEnum {
    const currentViewModeInState = ctx.getState().viewModeTableEnum;
    const viewMode = isNullOrUndefined(action.viewMode) ? (isNullOrUndefined(currentViewModeInState) ? DEFAULT_VIEW_MODE_TABLE : currentViewModeInState) : action.viewMode;
    return viewMode;
  }

  @Action(HomeAction.GetStatistics)
  getStatistics(ctx: StateContext<HomeStateModel>, action: HomeAction.GetStatistics): Observable<ArchiveStatisticsDto> {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
      archiveStatisticDto: undefined,
    });

    return this.apiService.getByIdInPath<ArchiveStatisticsDto>(`${this._urlResource}/${action.archiveId}/${ApiActionEnum.STATISTICS}`)
      .pipe(
        tap((model: ArchiveStatisticsDto) => {
          ctx.dispatch(new HomeAction.GetStatisticsSuccess(action, model));
        }),
        catchError(error => {
          ctx.dispatch(new HomeAction.GetStatisticsFail(action));
          throw new SolidifyStateError(error);
        }),
      );
  }

  @Action(HomeAction.GetStatisticsSuccess)
  getStatisticsSuccess(ctx: StateContext<HomeStateModel>, action: HomeAction.GetStatisticsSuccess): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
      archiveStatisticDto: action.archiveStatisticDto,
    });
  }

  @Action(HomeAction.GetStatisticsFail)
  getStatisticsFail(ctx: StateContext<HomeStateModel>, action: HomeAction.GetStatisticsFail): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
  }
}

