import {Injectable} from "@angular/core";
import {AccessResourceApiEnum} from "@app/shared/enums/api.enum";
import {LocalStateEnum} from "@app/shared/enums/local-state.enum";
import {
  HomeArchiveRatingAction,
  homeArchiveRatingActionNameSpace,
} from "@home/stores/archive/rating/home-archive-rating.action";
import {HomeAction} from "@home/stores/home.action";
import {ArchiveUserRating} from "@models";
import {
  Action,
  Actions,
  State,
  StateContext,
  Store,
} from "@ngxs/store";
import {ApiActionEnum} from "@shared/enums/api-action.enum";
import {LabelTranslateEnum} from "@shared/enums/label-translate.enum";
import {Observable} from "rxjs";
import {
  catchError,
  tap,
} from "rxjs/operators";
import {
  ApiService,
  CollectionTyped,
  CompositionState,
  CompositionStateModel,
  defaultCompositionStateInitValue,
  NotificationService,
  OverrideDefaultAction,
  SolidifyStateError,
  StoreUtil,
  urlSeparator,
} from "solidify-frontend";

export const defaultHomeArchiveRatingValue: () => HomeArchiveRatingStateModel = () =>
  ({
    ...defaultCompositionStateInitValue(),
  });

export interface HomeArchiveRatingStateModel extends CompositionStateModel<ArchiveUserRating> {
}

@Injectable()
@State<HomeArchiveRatingStateModel>({
  name: LocalStateEnum.home_archive_rating,
  defaults: {
    ...defaultHomeArchiveRatingValue(),
  },
  children: [],
})
export class HomeArchiveRatingState extends CompositionState<HomeArchiveRatingStateModel, ArchiveUserRating> {
  constructor(protected apiService: ApiService,
              protected store: Store,
              protected notificationService: NotificationService,
              protected actions$: Actions) {
    super(apiService, store, notificationService, actions$, {
      nameSpace: homeArchiveRatingActionNameSpace,
      resourceName: ApiActionEnum.RATING,
    });
  }

  protected get _urlResource(): string {
    return AccessResourceApiEnum.publicMetadata;
  }

  private _evaluateSubResourceUrl(parentId: string): string {
    return this._urlResource + urlSeparator + parentId + urlSeparator + this._resourceName;
  }

  @Action(HomeArchiveRatingAction.GetAllByCurrentUser)
  getAllByCurrentUser(ctx: StateContext<HomeArchiveRatingStateModel>, action: HomeArchiveRatingAction.GetAllByCurrentUser): Observable<CollectionTyped<ArchiveUserRating>> {
    let reset = {};
    if (!action.keepCurrentContext) {
      reset = {
        total: 0,
        list: undefined,
      };
    }
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
      queryParameters: StoreUtil.getQueryParametersToApply(action.queryParameters, ctx),
      ...reset,
    });
    const url = this._evaluateSubResourceUrl(action.parentId) + urlSeparator + ApiActionEnum.RATING_BY_USER;
    return this.apiService.get<ArchiveUserRating>(url, ctx.getState().queryParameters)
      .pipe(
        tap((collection: CollectionTyped<ArchiveUserRating>) => {
          ctx.dispatch(new HomeArchiveRatingAction.GetAllSuccess(action, collection));
        }),
        catchError(error => {
          ctx.dispatch(new HomeArchiveRatingAction.GetAllFail(action));
          throw new SolidifyStateError(this, error);
        }),
      );
  }

  @OverrideDefaultAction()
  @Action(HomeArchiveRatingAction.GetAllSuccess)
  getAllSuccess(ctx: StateContext<HomeArchiveRatingStateModel>, action: HomeArchiveRatingAction.GetAllSuccess): void {
    const queryParameters = StoreUtil.updateQueryParameters(ctx, action.list);

    ctx.patchState({
      list: action.list._data,
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
      queryParameters: queryParameters,
    });
  }

  @OverrideDefaultAction()
  @Action(HomeArchiveRatingAction.GetAllFail)
  getAllFail(ctx: StateContext<HomeArchiveRatingStateModel>, action: HomeArchiveRatingAction.GetAllFail): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
  }

  @OverrideDefaultAction()
  @Action(HomeArchiveRatingAction.Create)
  create(ctx: StateContext<HomeArchiveRatingStateModel>, action: HomeArchiveRatingAction.Create): Observable<ArchiveUserRating> {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
    });
    const url = this._evaluateSubResourceUrl(action.parentId);
    return this.apiService.post<ArchiveUserRating>(url, undefined, {
      "ratingType": action.model.ratingType.resId,
      "grade": action.model.grade,
    })
      .pipe(
        tap(() => {
          ctx.dispatch(new HomeArchiveRatingAction.CreateSuccess(action, action.parentId));
        }),
        catchError(error => {
          ctx.dispatch(new HomeArchiveRatingAction.CreateFail(action));
          throw new SolidifyStateError(this, error);
        }),
      );
  }

  @OverrideDefaultAction()
  @Action(HomeArchiveRatingAction.Update)
  update(ctx: StateContext<HomeArchiveRatingStateModel>, action: HomeArchiveRatingAction.Update): Observable<ArchiveUserRating> {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
    });
    const url = this._evaluateSubResourceUrl(action.parentId);
    return this.apiService.patchByIdWithCustomUrl<ArchiveUserRating>(url, action.model)
      .pipe(
        tap(() => {
          ctx.dispatch(new HomeArchiveRatingAction.UpdateSuccess(action, action.parentId));
        }),
        catchError(error => {
          ctx.dispatch(new HomeArchiveRatingAction.UpdateFail(action, action.parentId));
          throw new SolidifyStateError(this, error);
        }),
      );
  }

  @OverrideDefaultAction()
  @Action(HomeArchiveRatingAction.CreateSuccess)
  createSuccess(ctx: StateContext<HomeArchiveRatingStateModel>, action: HomeArchiveRatingAction.CreateSuccess): void {
    super.createSuccess(ctx, action);
    this.notificationService.showSuccess(LabelTranslateEnum.evaluationTakenIntoAccount);
    ctx.dispatch(new HomeAction.GetStatistics(action.parentId));
  }

  @OverrideDefaultAction()
  @Action(HomeArchiveRatingAction.UpdateSuccess)
  updateSuccess(ctx: StateContext<HomeArchiveRatingStateModel>, action: HomeArchiveRatingAction.UpdateSuccess): void {
    super.updateSuccess(ctx, action);
    this.notificationService.showSuccess(LabelTranslateEnum.evaluationTakenIntoAccount);
    ctx.dispatch(new HomeAction.GetStatistics(action.parentId));
  }
}
