import {HttpClient} from "@angular/common/http";
import {Injectable} from "@angular/core";
import {AccessResourceApiEnum} from "@app/shared/enums/api.enum";
import {LocalStateEnum} from "@app/shared/enums/local-state.enum";
import {HomeArchivePackageAction} from "@home/stores/archive/package/home-archive-package.action";
import {
  Action,
  Actions,
  State,
  StateContext,
  Store,
} from "@ngxs/store";
import {ApiResourceNameEnum} from "@shared/enums/api-resource-name.enum";
import {ArchiveMetadataPackages} from "@shared/models/business/archive-metadata-packages.model";
import {Observable} from "rxjs";
import {
  catchError,
  tap,
} from "rxjs/operators";
import {
  ApiService,
  BaseStateModel,
  BasicState,
  isFalse,
  NotificationService,
  urlSeparator,
} from "solidify-frontend";

export const defaultHomeArchiveDataFileValue: () => HomeArchivePackageStateModel = () =>
  ({
    isLoadingCounter: 0,
    package: undefined,
  });

export interface HomeArchivePackageStateModel extends BaseStateModel {
  package: ArchiveMetadataPackages;
}

@Injectable()
@State<HomeArchivePackageStateModel>({
  name: LocalStateEnum.home_archive_package,
  defaults: {
    ...defaultHomeArchiveDataFileValue(),
  },
  children: [],
})
export class HomeArchivePackageState extends BasicState<HomeArchivePackageStateModel> {
  private readonly _resourceName: ApiResourceNameEnum = ApiResourceNameEnum.PACKAGES;

  protected get _urlResource(): string {
    return AccessResourceApiEnum.publicMetadata;
  }

  constructor(protected apiService: ApiService,
              protected store: Store,
              protected httpClient: HttpClient,
              protected notificationService: NotificationService,
              protected actions$: Actions) {
    super();
  }

  @Action(HomeArchivePackageAction.GetById)
  getById(ctx: StateContext<HomeArchivePackageStateModel>, action: HomeArchivePackageAction.GetById): Observable<ArchiveMetadataPackages> {
    const objectToPatch: HomeArchivePackageStateModel | any = {
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
    };

    if (isFalse(action.keepCurrentContext)) {
      objectToPatch.package = undefined;
    }
    ctx.patchState(objectToPatch);

    return this.apiService.getByIdInPath<ArchiveMetadataPackages>(this._urlResource + urlSeparator + action.resId + urlSeparator + this._resourceName)
      .pipe(
        tap(model => ctx.dispatch(new HomeArchivePackageAction.GetByIdSuccess(action, model))),
        catchError(error => {
          ctx.dispatch(new HomeArchivePackageAction.GetByIdFail(action));
          throw error;
        }),
      );
  }

  @Action(HomeArchivePackageAction.GetByIdSuccess)
  getByIdSuccess(ctx: StateContext<HomeArchivePackageStateModel>, action: HomeArchivePackageAction.GetByIdSuccess): void {
    ctx.patchState({
      package: action.model,
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
  }

  @Action(HomeArchivePackageAction.GetByIdFail)
  getByIdFail(ctx: StateContext<HomeArchivePackageStateModel>, action: HomeArchivePackageAction.GetByIdFail): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
  }
}
