import {HttpClient} from "@angular/common/http";
import {Injectable} from "@angular/core";
import {AccessResourceApiEnum} from "@app/shared/enums/api.enum";
import {LocalStateEnum} from "@app/shared/enums/local-state.enum";
import {DepositDataFileHelper} from "@deposit/helpers/deposit-data-file.helper";
import {ArchiveDataFile} from "@home/models/archive-data-file.model";
import {HomeArchiveDataFileAction} from "@home/stores/archive/data-file/home-archive-data-file.action";
import {HomeState} from "@home/stores/home.state";
import {
  Action,
  Actions,
  ofActionCompleted,
  State,
  StateContext,
  Store,
} from "@ngxs/store";
import {ApiActionEnum} from "@shared/enums/api-action.enum";
import {ApiResourceNameEnum} from "@shared/enums/api-resource-name.enum";
import {ArchiveDataFileHelper} from "@shared/helpers/archive-data-file.helper";
import {ArchiveMetadataDataFile} from "@shared/models/business/archive-metadata-data-file.model";
import {Observable} from "rxjs";
import {
  catchError,
  tap,
} from "rxjs/operators";
import {
  ApiService,
  BaseResourceState,
  BasicState,
  CollectionTyped,
  isNullOrUndefined,
  isTrue,
  MappingObjectUtil,
  MemoizedUtil,
  NotificationService,
  QueryParameters,
  QueryParametersUtil,
  SolidifyStateError,
  StoreUtil,
  urlSeparator,
} from "solidify-frontend";

export const defaultHomeArchiveDataFileValue: () => HomeArchiveDataFileStateModel = () =>
  ({
    isLoadingCounter: 0,
    currentFolder: "" + DepositDataFileHelper.ROOT,
    folders: [],
    foldersWithIntermediateFolders: [],
    intermediateFolders: [],
    numberFiles: undefined,
    queryParameters: new QueryParameters(),
    total: 0,
    list: [],
    current: undefined,
  });

export interface HomeArchiveDataFileStateModel extends BaseResourceState {
  currentFolder: string;
  folders: string[];
  foldersWithIntermediateFolders: string[];
  intermediateFolders: string[];
  numberFiles: number | undefined;
  list: ArchiveDataFile[];
  current: ArchiveDataFile;
}

@Injectable()
@State<HomeArchiveDataFileStateModel>({
  name: LocalStateEnum.home_archive_dataFile,
  defaults: {
    ...defaultHomeArchiveDataFileValue(),
  },
  children: [],
})
export class HomeArchiveDataFileState extends BasicState<HomeArchiveDataFileStateModel> {
  private readonly _resourceName: ApiResourceNameEnum = ApiResourceNameEnum.DATAFILE;
  private readonly KEY_PATH: string = "path";

  protected get _urlResource(): string {
    return AccessResourceApiEnum.publicMetadata;
  }

  private evaluateSubResourceUrl(parentId: string): string {
    return this._urlResource + urlSeparator + parentId + urlSeparator + this._resourceName;
  }

  constructor(protected apiService: ApiService,
              protected store: Store,
              protected httpClient: HttpClient,
              protected notificationService: NotificationService,
              protected actions$: Actions) {
    super();
  }

  @Action(HomeArchiveDataFileAction.ChangeQueryParameters)
  changeQueryParameters(ctx: StateContext<HomeArchiveDataFileStateModel>, action: HomeArchiveDataFileAction.ChangeQueryParameters): void {
    ctx.patchState({
      queryParameters: action.queryParameters,
    });
    if (isTrue(action.getAllAfterChange)) {
      ctx.dispatch(new HomeArchiveDataFileAction.GetAll(action.parentId, undefined, action.keepCurrentContext));
    }
  }

  @Action(HomeArchiveDataFileAction.GetAll)
  getAll(ctx: StateContext<HomeArchiveDataFileStateModel>, action: HomeArchiveDataFileAction.GetAll): Observable<CollectionTyped<ArchiveMetadataDataFile>> {
    let queryParameters = action.queryParameters;
    if (isNullOrUndefined(queryParameters)) {
      queryParameters = QueryParametersUtil.clone(ctx.getState().queryParameters);
    }
    const currentFolder = ctx.getState().currentFolder;
    if (isNullOrUndefined(currentFolder)) {
      MappingObjectUtil.delete(QueryParametersUtil.getSearchItems(queryParameters), this.KEY_PATH);

    } else {
      MappingObjectUtil.set(QueryParametersUtil.getSearchItems(queryParameters), this.KEY_PATH, ctx.getState().currentFolder);
    }
    ctx.patchState({
      queryParameters: queryParameters,
    });

    let reset = {};
    if (!action.keepCurrentContext) {
      reset = {
        total: 0,
        list: undefined,
      };
    }
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
      queryParameters: StoreUtil.getQueryParametersToApply(action.queryParameters, ctx),
      ...reset,
    });
    const url = this.evaluateSubResourceUrl(action.parentId);
    return this.apiService.get<ArchiveMetadataDataFile>(url, ctx.getState().queryParameters)
      .pipe(
        tap((collection: CollectionTyped<ArchiveMetadataDataFile>) => {
          collection._data = ArchiveDataFileHelper.adaptListArchivesMetadataInArchive(collection._data) as ArchiveDataFile[] | any;
          ctx.dispatch(new HomeArchiveDataFileAction.GetAllSuccess(action, collection as CollectionTyped<ArchiveDataFile>));
        }),
        catchError(error => {
          ctx.dispatch(new HomeArchiveDataFileAction.GetAllFail(action));
          throw new SolidifyStateError(error);
        }),
      );
  }

  @Action(HomeArchiveDataFileAction.GetAllSuccess)
  getAllSuccess(ctx: StateContext<HomeArchiveDataFileStateModel>, action: HomeArchiveDataFileAction.GetAllSuccess): void {
    const queryParameters = StoreUtil.updateQueryParameters(ctx as any, action.list);

    ctx.patchState({
      list: action.list._data,
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
      queryParameters: queryParameters,
    });
  }

  @Action(HomeArchiveDataFileAction.GetAllFail)
  getAllFail(ctx: StateContext<HomeArchiveDataFileStateModel>, action: HomeArchiveDataFileAction.GetAllFail): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
  }

  @Action(HomeArchiveDataFileAction.GetById)
  getById(ctx: StateContext<HomeArchiveDataFileStateModel>, action: HomeArchiveDataFileAction.GetById): Observable<ArchiveDataFile> {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
      current: undefined,
    });

    return this.apiService.getById<ArchiveMetadataDataFile>(this.evaluateSubResourceUrl(action.parentId), action.resId)
      .pipe(
        tap(model => ctx.dispatch(new HomeArchiveDataFileAction.GetByIdSuccess(action, ArchiveDataFileHelper.adaptArchiveMetadataInArchiveDataFile(model)))),
        catchError(error => {
          ctx.dispatch(new HomeArchiveDataFileAction.GetByIdFail(action));
          throw error;
        }),
      );
  }

  @Action(HomeArchiveDataFileAction.GetByIdSuccess)
  getByIdSuccess(ctx: StateContext<HomeArchiveDataFileStateModel>, action: HomeArchiveDataFileAction.GetByIdSuccess): void {
    ctx.patchState({
      current: action.model,
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
  }

  @Action(HomeArchiveDataFileAction.GetByIdFail)
  getByIdFail(ctx: StateContext<HomeArchiveDataFileStateModel>, action: HomeArchiveDataFileAction.GetByIdFail): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
  }

  @Action(HomeArchiveDataFileAction.Refresh)
  refresh(ctx: StateContext<HomeArchiveDataFileStateModel>, action: HomeArchiveDataFileAction.Refresh): Observable<any> {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
    });

    return StoreUtil.dispatchParallelActionAndWaitForSubActionsCompletion(ctx, [
      {
        action: new HomeArchiveDataFileAction.GetAll(action.parentId, undefined, true),
        subActionCompletions: [
          this.actions$.pipe(ofActionCompleted(HomeArchiveDataFileAction.GetAllSuccess)),
          this.actions$.pipe(ofActionCompleted(HomeArchiveDataFileAction.GetAllFail)),
        ],
      },
      {
        action: new HomeArchiveDataFileAction.GetListFolder(action.parentId),
        subActionCompletions: [
          this.actions$.pipe(ofActionCompleted(HomeArchiveDataFileAction.GetListFolderSuccess)),
          this.actions$.pipe(ofActionCompleted(HomeArchiveDataFileAction.GetListFolderFail)),
        ],
      },
    ]).pipe(
      tap(success => {
        if (success) {
          ctx.dispatch(new HomeArchiveDataFileAction.RefreshSuccess(action));
        } else {
          ctx.dispatch(new HomeArchiveDataFileAction.RefreshFail(action));
        }
      }),
    );
  }

  @Action(HomeArchiveDataFileAction.RefreshSuccess)
  refreshSuccess(ctx: StateContext<HomeArchiveDataFileStateModel>, action: HomeArchiveDataFileAction.RefreshSuccess): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
  }

  @Action(HomeArchiveDataFileAction.RefreshFail)
  refreshFail(ctx: StateContext<HomeArchiveDataFileStateModel>, action: HomeArchiveDataFileAction.RefreshFail): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
  }

  @Action(HomeArchiveDataFileAction.GetListFolder)
  getListFolder(ctx: StateContext<HomeArchiveDataFileStateModel>, action: HomeArchiveDataFileAction.GetListFolder): Observable<string[]> {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
    });

    return this.httpClient.get<string[]>(`${this._urlResource}/${action.parentId}/${ApiActionEnum.LIST_FOLDERS}`)
      .pipe(
        tap((listFolder: string[]) => {
          ctx.dispatch(new HomeArchiveDataFileAction.GetListFolderSuccess(action, listFolder));
        }),
        catchError(error => {
          ctx.dispatch(new HomeArchiveDataFileAction.GetListFolderFail(action));
          throw new SolidifyStateError(error);
        }),
      );
  }

  @Action(HomeArchiveDataFileAction.GetListFolderSuccess)
  getListFolderSuccess(ctx: StateContext<HomeArchiveDataFileStateModel>, action: HomeArchiveDataFileAction.GetListFolderSuccess): void {
    const folders = [...action.folder];
    const foldersWithIntermediateFolders = action.folder;
    const intermediateFolders = DepositDataFileHelper.createIntermediateFolders(foldersWithIntermediateFolders);
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
      folders,
      foldersWithIntermediateFolders,
      intermediateFolders,
    });
  }

  @Action(HomeArchiveDataFileAction.GetListFolderFail)
  getListFolderFail(ctx: StateContext<HomeArchiveDataFileStateModel>, action: HomeArchiveDataFileAction.GetListFolderFail): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
  }

  @Action(HomeArchiveDataFileAction.ChangeCurrentFolder)
  changeCurrentFolder(ctx: StateContext<HomeArchiveDataFileStateModel>, action: HomeArchiveDataFileAction.ChangeCurrentFolder): void {
    ctx.patchState({
      currentFolder: action.newCurrentFolder,
    });
    const archiveId = MemoizedUtil.selectSnapshot(this.store, HomeState, state => state.current?.resId);
    if (!isNullOrUndefined(action.newCurrentFolder)) {
      ctx.dispatch(new HomeArchiveDataFileAction.GetListFolder(archiveId));
    }
    if (action.refreshNewFolderContent) {
      let queryParameters = QueryParametersUtil.clone(ctx.getState().queryParameters);
      queryParameters = QueryParametersUtil.resetToFirstPage(queryParameters);
      ctx.dispatch(new HomeArchiveDataFileAction.GetAll(archiveId, queryParameters));
    }
  }
}
