import {HttpClient} from "@angular/common/http";
import {Injectable} from "@angular/core";
import {AccessResourceApiEnum} from "@app/shared/enums/api.enum";
import {LocalStateEnum} from "@app/shared/enums/local-state.enum";
import {ArchiveDataFile} from "@home/models/archive-data-file.model";
import {HomeArchiveCollectionAction} from "@home/stores/archive/collection/home-archive-collection.action";
import {
  Action,
  Actions,
  ofActionCompleted,
  State,
  StateContext,
  Store,
} from "@ngxs/store";
import {ApiResourceNameEnum} from "@shared/enums/api-resource-name.enum";
import {ArchiveDataFileHelper} from "@shared/helpers/archive-data-file.helper";
import {ArchiveMetadataDataFile} from "@shared/models/business/archive-metadata-data-file.model";
import {Observable} from "rxjs";
import {
  catchError,
  tap,
} from "rxjs/operators";
import {
  ApiService,
  BaseResourceState,
  BasicState,
  CollectionTyped,
  isTrue,
  NotificationService,
  QueryParameters,
  SolidifyStateError,
  StoreUtil,
  urlSeparator,
} from "solidify-frontend";

export const defaultHomeArchiveCollectionValue: () => HomeArchiveCollectionStateModel = () =>
  ({
    isLoadingCounter: 0,
    numberFiles: undefined,
    queryParameters: new QueryParameters(),
    total: 0,
    list: [],
    current: undefined,
  });

export interface HomeArchiveCollectionStateModel extends BaseResourceState {
  numberFiles: number | undefined;
  list: ArchiveDataFile[];
  current: ArchiveDataFile;
}

@Injectable()
@State<HomeArchiveCollectionStateModel>({
  name: LocalStateEnum.home_archive_collection,
  defaults: {
    ...defaultHomeArchiveCollectionValue(),
  },
  children: [],
})
export class HomeArchiveCollectionState extends BasicState<HomeArchiveCollectionStateModel> {
  private readonly _resourceName: ApiResourceNameEnum = ApiResourceNameEnum.AIP;

  protected get _urlResource(): string {
    return AccessResourceApiEnum.publicMetadata;
  }

  private evaluateSubResourceUrl(parentId: string): string {
    return this._urlResource + urlSeparator + parentId + urlSeparator + this._resourceName;
  }

  constructor(protected apiService: ApiService,
              protected store: Store,
              protected httpClient: HttpClient,
              protected notificationService: NotificationService,
              protected actions$: Actions) {
    super();
  }

  @Action(HomeArchiveCollectionAction.ChangeQueryParameters)
  changeQueryParameters(ctx: StateContext<HomeArchiveCollectionStateModel>, action: HomeArchiveCollectionAction.ChangeQueryParameters): void {
    ctx.patchState({
      queryParameters: action.queryParameters,
    });
    if (isTrue(action.getAllAfterChange)) {
      ctx.dispatch(new HomeArchiveCollectionAction.GetAll(action.parentId, undefined, action.keepCurrentContext));
    }
  }

  @Action(HomeArchiveCollectionAction.GetAll)
  getAll(ctx: StateContext<HomeArchiveCollectionStateModel>, action: HomeArchiveCollectionAction.GetAll): Observable<CollectionTyped<ArchiveMetadataDataFile>> {
    let reset = {};
    if (!action.keepCurrentContext) {
      reset = {
        total: 0,
        list: undefined,
      };
    }
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
      queryParameters: StoreUtil.getQueryParametersToApply(action.queryParameters, ctx),
      ...reset,
    });
    const url = this.evaluateSubResourceUrl(action.parentId);
    return this.apiService.get<ArchiveMetadataDataFile>(url, ctx.getState().queryParameters)
      .pipe(
        tap((collection: CollectionTyped<ArchiveMetadataDataFile>) => {
          collection._data = ArchiveDataFileHelper.adaptListArchivesMetadataInArchive(collection._data) as ArchiveDataFile[] | any;
          ctx.dispatch(new HomeArchiveCollectionAction.GetAllSuccess(action, collection as CollectionTyped<ArchiveDataFile>));
        }),
        catchError(error => {
          ctx.dispatch(new HomeArchiveCollectionAction.GetAllFail(action));
          throw new SolidifyStateError(error);
        }),
      );
  }

  @Action(HomeArchiveCollectionAction.GetAllSuccess)
  getAllSuccess(ctx: StateContext<HomeArchiveCollectionStateModel>, action: HomeArchiveCollectionAction.GetAllSuccess): void {
    const queryParameters = StoreUtil.updateQueryParameters(ctx as any, action.list);

    ctx.patchState({
      list: action.list._data,
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
      queryParameters: queryParameters,
    });
  }

  @Action(HomeArchiveCollectionAction.GetAllFail)
  getAllFail(ctx: StateContext<HomeArchiveCollectionStateModel>, action: HomeArchiveCollectionAction.GetAllFail): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
  }

  @Action(HomeArchiveCollectionAction.GetById)
  getById(ctx: StateContext<HomeArchiveCollectionStateModel>, action: HomeArchiveCollectionAction.GetById): Observable<ArchiveDataFile> {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
      current: undefined,
    });

    return this.apiService.getById<ArchiveMetadataDataFile>(this.evaluateSubResourceUrl(action.parentId), action.resId)
      .pipe(
        tap(model => ctx.dispatch(new HomeArchiveCollectionAction.GetByIdSuccess(action, ArchiveDataFileHelper.adaptArchiveMetadataInArchiveDataFile(model)))),
        catchError(error => {
          ctx.dispatch(new HomeArchiveCollectionAction.GetByIdFail(action));
          throw error;
        }),
      );
  }

  @Action(HomeArchiveCollectionAction.GetByIdSuccess)
  getByIdSuccess(ctx: StateContext<HomeArchiveCollectionStateModel>, action: HomeArchiveCollectionAction.GetByIdSuccess): void {
    ctx.patchState({
      current: action.model,
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
  }

  @Action(HomeArchiveCollectionAction.GetByIdFail)
  getByIdFail(ctx: StateContext<HomeArchiveCollectionStateModel>, action: HomeArchiveCollectionAction.GetByIdFail): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
  }

  @Action(HomeArchiveCollectionAction.Refresh)
  refresh(ctx: StateContext<HomeArchiveCollectionStateModel>, action: HomeArchiveCollectionAction.Refresh): Observable<any> {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
    });

    return StoreUtil.dispatchParallelActionAndWaitForSubActionsCompletion(ctx, [
      {
        action: new HomeArchiveCollectionAction.GetAll(action.parentId, undefined, true),
        subActionCompletions: [
          this.actions$.pipe(ofActionCompleted(HomeArchiveCollectionAction.GetAllSuccess)),
          this.actions$.pipe(ofActionCompleted(HomeArchiveCollectionAction.GetAllFail)),
        ],
      },
    ]).pipe(
      tap(success => {
        if (success) {
          ctx.dispatch(new HomeArchiveCollectionAction.RefreshSuccess(action));
        } else {
          ctx.dispatch(new HomeArchiveCollectionAction.RefreshFail(action));
        }
      }),
    );
  }

  @Action(HomeArchiveCollectionAction.RefreshSuccess)
  refreshSuccess(ctx: StateContext<HomeArchiveCollectionStateModel>, action: HomeArchiveCollectionAction.RefreshSuccess): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
  }

  @Action(HomeArchiveCollectionAction.RefreshFail)
  refreshFail(ctx: StateContext<HomeArchiveCollectionStateModel>, action: HomeArchiveCollectionAction.RefreshFail): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
  }
}
