import {DOCUMENT} from "@angular/common";
import {
  Inject,
  Injectable,
  Renderer2,
} from "@angular/core";
import {WINDOW} from "@app/app.module";
import {TwitterMetadata} from "@app/features/home/models/twitter-metadata.model";
import {Observable} from "rxjs";

@Injectable({
  providedIn: "root",
})
export class TwitterService {
  private readonly _TWITTER_SCRIPT_ID: string = "twitter-wjs";
  private readonly _TWITTER_WIDGET_URL: string = "https://platform.twitter.com/widgets.js";
  private readonly _SCRIPT: string = "script";
  private readonly _ATTRIBUTE_ID: string = "id";
  private readonly _ATTRIBUTE_SRC: string = "src";
  readonly TWITTER_WINDOW_ID: string = "twttr";

  constructor(@Inject(DOCUMENT) private _document: Document,
              @Inject(WINDOW) private _window: Window) {
  }

  loadScript(renderer: Renderer2): Observable<TwitterMetadata> {
    return new Observable(observer => {
      const twitterMetadata = this.initTwitterComponent(renderer);
      twitterMetadata.ready((metadata: TwitterMetadata) => {
        observer.next(metadata);
        observer.complete();
      });
    });
  }

  private initTwitterComponent(renderer: Renderer2): TwitterMetadata {
    if (this.getTwitterScript()) {
      return this._window[this.TWITTER_WINDOW_ID] || {};
    }

    const scriptTwitter = this.createScript(renderer);
    const firstScriptTag = this.getFirstScriptTag();
    renderer.insertBefore(firstScriptTag.parentNode, scriptTwitter, firstScriptTag);

    return this.initTwitterMetadata();
  }

  private createScript(renderer: Renderer2): Element {
    const script = renderer.createElement(this._SCRIPT);
    renderer.setAttribute(script, this._ATTRIBUTE_ID, this._TWITTER_SCRIPT_ID);
    renderer.setAttribute(script, this._ATTRIBUTE_SRC, this._TWITTER_WIDGET_URL);
    return script;
  }

  private getFirstScriptTag(): Element {
    return this._document.getElementsByTagName(this._SCRIPT)[0];
  }

  private getTwitterScript(): HTMLElement {
    return this._document.getElementById(this._TWITTER_SCRIPT_ID);
  }

  private initTwitterMetadata(): TwitterMetadata {
    const twitterMetadata: TwitterMetadata = {};
    twitterMetadata._e = [];
    twitterMetadata.ready = (metadata: TwitterMetadata) => twitterMetadata._e.push(metadata);
    this._window[this.TWITTER_WINDOW_ID] = twitterMetadata;
    return twitterMetadata;
  }
}


