import {Injectable} from "@angular/core";
import {MatDialog} from "@angular/material/dialog";
import {LocalStateModel} from "@app/shared/models/local-state.model";
import {AppUserState} from "@app/stores/user/app-user.state";
import {Enums} from "@enums";
import {environment} from "@environments/environment";
import {HomeArchiveRestrictedRequestAccessDialog} from "@home/components/dialogs/archive-restricted-request-access/home-archive-restricted-request-access-dialog.component";
import {Archive} from "@home/models/archive.model";
import {Store} from "@ngxs/store";
import {
  RoutesEnum,
  urlSeparator,
} from "@shared/enums/routes.enum";
import {SecurityService} from "@shared/services/security.service";
import {SharedNotificationAction} from "@shared/stores/notification/shared-notification.action";
import {
  take,
  tap,
} from "rxjs/operators";
import {
  isNullOrUndefined,
  MARK_AS_TRANSLATABLE,
  MemoizedUtil,
  OAuth2Service,
} from "solidify-frontend";

@Injectable({
  providedIn: "root",
})
export class ArchiveAccessRightService {
  constructor(private readonly _dialog: MatDialog,
              private readonly _store: Store,
              private readonly _securityService: SecurityService,
              private readonly _oauthService: OAuth2Service) {
  }

  isAccessControlled(archive: Archive): boolean {
    return !this.isDownloadAuthorized(archive);
  }

  isDownloadAuthorized(archive: Archive): boolean {
    if (this.isPublicMetadata(archive)) {
      return true;
    }
    if (!this._securityService.isLoggedIn()) {
      return false;
    }

    if (this._securityService.isRootOrAdmin()) {
      return true;
    }

    if (this.isRestrictedMetadata(archive)) {
      if (this._securityService.isMemberOfOrgUnit(archive.organizationalUnitId)) {
        return true;
      }
      return this._securityService.isActiveArchiveAcl(archive.resId);
    }

    if (this.isClosedMetadata(archive)) {
      if (this._securityService.isStewardOfOrgUnit(archive.organizationalUnitId)) {
        return true;
      }
      return this._securityService.isActiveArchiveAcl(archive.resId);
    }
  }

  isAskAccessAvailable(archive: Archive): boolean {
    return !this.isDownloadAuthorized(archive);
  }

  requestAccessDataset(archive: Archive): void {
    const isLogged = this._store.selectSnapshot((state: LocalStateModel) => state.application.isLoggedIn);

    if (!isLogged) {
      const accessRequestRouteParam = environment.nonceStateSeparator + environment.accessRequestRoute + "=true";
      this._oauthService.initAuthorizationCodeFlow(RoutesEnum.homeDetail + urlSeparator + archive.resId + accessRequestRouteParam);
    } else {
      this.openRequestArchiveDialog(archive.resId, archive.organizationalUnitId);
    }
  }

  private sendRequestAccessDataset(archiveId: string, archiveOrgUnitId: string, message: string): void {
    this._store.dispatch(new SharedNotificationAction.Create({
      model: {
        emitter: MemoizedUtil.currentSnapshot(this._store, AppUserState),
        notifiedOrgUnit: {
          resId: archiveOrgUnitId,
        },
        objectId: archiveId,
        message: message,
        notificationType: {
          resId: Enums.Notification.TypeEnum.ACCESS_DATASET_REQUEST,
        },
      },
    }));
  }

  openRequestArchiveDialog(archiveId: string, archiveOrgUnitId: string): void {
    this._dialog.open(HomeArchiveRestrictedRequestAccessDialog, {
      minWidth: "500px",
    }).afterClosed().pipe(
      take(1),
      tap((message: string | undefined) => {
        if (isNullOrUndefined(message)) {
          return;
        }
        this.sendRequestAccessDataset(archiveId, archiveOrgUnitId, message);
      }),
    ).subscribe();
  }

  isPublicMetadata(archive: Archive): boolean {
    return this._checkCurrentAccessLevelIs(archive, Enums.Deposit.AccessEnum.PUBLIC);
  }

  isRestrictedMetadata(archive: Archive): boolean {
    return this._checkCurrentAccessLevelIs(archive, Enums.Deposit.AccessEnum.RESTRICTED);
  }

  isClosedMetadata(archive: Archive): boolean {
    return this._checkCurrentAccessLevelIs(archive, Enums.Deposit.AccessEnum.CLOSED);
  }

  private _checkCurrentAccessLevelIs(archive: Archive, accessLevel: Enums.Deposit.AccessEnum): boolean {
    if (isNullOrUndefined(archive)) {
      return false;
    }
    return archive.accessLevel === accessLevel;
  }

  getTooltipDownload(archive: Archive, isDisabled: boolean): undefined | string {
    if (this.isPublicMetadata(archive) || !isDisabled) {
      return undefined;
    }
    if (this.isRestrictedMetadata(archive)) {
      return MARK_AS_TRANSLATABLE("homePage.archive.detail.buttonDisabledReason.needToBeMember");
    }
    if (this.isClosedMetadata(archive)) {
      return MARK_AS_TRANSLATABLE("homePage.archive.detail.buttonDisabledReason.needToBeSteward");
    }
  }
}
