import {
  NavigationExtras,
  Params,
} from "@angular/router";
import {Enums} from "@enums";
import {SearchFacet} from "@models";
import {Navigate} from "@ngxs/router-plugin";
import {Store} from "@ngxs/store";
import {RoutesEnum} from "@shared/enums/routes.enum";
import {ViewModeTableEnum} from "@shared/enums/view-mode-table.enum";
import {Observable} from "rxjs";
import {
  isNonEmptyString,
  isNotNullNorUndefined,
  isNullOrUndefined,
  MappingObject,
  MappingObjectUtil,
  StringUtil,
} from "solidify-frontend";

export class HomeHelper {
  static SEARCH_QUERY_PARAM: string = "search";
  static VIEW_QUERY_PARAM: string = "view";
  static VALUES_SEPARATOR: string = ";";
  static QUERY_PARAM_SEPARATOR: string = "&";
  static QUERY_PARAM_VALUE_AFFECTION: string = "=";

  static LIST_FACET_USE_OPERATOR_AND: Enums.Facet.Name[] = [Enums.Facet.Name.CREATORS];

  static getDefaultSearchQueryParams(): any {
    return {[this.SEARCH_QUERY_PARAM]: "*"};
  }

  static navigateToSearch(store: Store, searchTerm?: string | undefined,
                          facetsSelected?: MappingObject<string[]> | undefined,
                          viewModeTable?: ViewModeTableEnum | undefined): Observable<any> {
    let queryParams = {};

    if (isNonEmptyString(searchTerm)) {
      queryParams = {[this.SEARCH_QUERY_PARAM]: encodeURIComponent(searchTerm)};
    }
    if (isNotNullNorUndefined(facetsSelected)) {
      MappingObjectUtil.forEach(facetsSelected, ((value, key) => {
        const valueString = value.map(v => encodeURIComponent(v)).join(this.VALUES_SEPARATOR);
        MappingObjectUtil.set(queryParams, key, valueString);
      }));
    }

    let completeSearchString = undefined;
    MappingObjectUtil.forEach(queryParams, ((value: string, key: string) => {
      if (isNotNullNorUndefined(completeSearchString)) {
        completeSearchString = completeSearchString + this.QUERY_PARAM_SEPARATOR;
      } else {
        completeSearchString = StringUtil.stringEmpty;
      }
      completeSearchString = completeSearchString + key + this.QUERY_PARAM_VALUE_AFFECTION + value;
    }));

    const navigationExtras: NavigationExtras = {
      [this.VIEW_QUERY_PARAM]: viewModeTable,
    };
    if (isNotNullNorUndefined(completeSearchString)) {
      navigationExtras[this.SEARCH_QUERY_PARAM] = completeSearchString;
    }
    return store.dispatch(new Navigate([RoutesEnum.homeSearch], navigationExtras));
  }

  static extractSearchInfosFromUrl(params: Params): { search: string, facetsSelected: MappingObject<string[]>, viewTabMode: ViewModeTableEnum } {
    let searchEncoded = undefined;

    const viewTabMode = params[this.VIEW_QUERY_PARAM];
    const queryParamSearch = params[this.SEARCH_QUERY_PARAM];
    if (!isNullOrUndefined(queryParamSearch)) {
      searchEncoded = queryParamSearch;
    }

    let searchString = StringUtil.stringEmpty;
    const facetsSelected: MappingObject<string[]> = {};
    if (isNotNullNorUndefined(searchEncoded)) {
      const listParams = searchEncoded.split(this.QUERY_PARAM_SEPARATOR);
      listParams.forEach(p => {
        const splitParam = p.split(this.QUERY_PARAM_VALUE_AFFECTION);
        const decodedValue = decodeURIComponent(splitParam[1]);
        if (splitParam[0] === this.SEARCH_QUERY_PARAM) {
          searchString = decodedValue;
        } else {
          if (splitParam.length > 1) {
            MappingObjectUtil.set(facetsSelected, splitParam[0], decodedValue.split(this.VALUES_SEPARATOR));
          }
        }
      });
    }

    return {
      search: searchString,
      facetsSelected: facetsSelected,
      viewTabMode: viewTabMode,
    };
  }

  static getFacetsSelectedForBackend(facetsSelected: MappingObject<string[]>): SearchFacet[] {
    const facetsSelectedForBackend: SearchFacet[] = [];
    MappingObjectUtil.forEach(facetsSelected, (value, key) => {
      if (this.LIST_FACET_USE_OPERATOR_AND.indexOf(key as Enums.Facet.Name) === -1) {
        // FOR OR, equal value =  [ "orgUnitName1", "orgUnitName2" ]
        facetsSelectedForBackend.push({
          field: key,
          type: undefined,
          equalValue: value.length === 1 ? value[0] : value,
        });
      } else {
        // FOR AND, equal value = "orgUnitName1", need to create a second search facet with "orgUnitName2"
        value.forEach(v => {
          facetsSelectedForBackend.push({
            field: key,
            type: undefined,
            equalValue: v,
          });
        });
      }
    });
    return facetsSelectedForBackend;
  }
}
