import {
  ChangeDetectionStrategy,
  Component,
  OnDestroy,
  OnInit,
} from "@angular/core";
import {MatDialog} from "@angular/material/dialog";
import {
  ActivatedRoute,
  Params,
} from "@angular/router";
import {Archive} from "@app/features/home/models/archive.model";
import {HomeAction} from "@app/features/home/stores/home.action";
import {HomeState} from "@app/features/home/stores/home.state";
import {SharedAbstractPresentational} from "@app/shared/components/presentationals/shared-abstract/shared-abstract.presentational";
import {FieldTypeEnum} from "@app/shared/enums/field-type.enum";
import {
  AppRoutesEnum,
  RoutesEnum,
} from "@app/shared/enums/routes.enum";
import {DataTableColumns} from "@app/shared/models/data-table-columns.model";
import {LocalStateModel} from "@app/shared/models/local-state.model";
import {AppState} from "@app/stores/app.state";
import {AppCartAction} from "@app/stores/cart/app-cart.action";
import {AppSystemPropertyState} from "@app/stores/system-property/app-system-property.state";
import {Enums} from "@enums";
import {HomeSearchHelpDialog} from "@home/components/dialogs/home-search-help/home-search-help.dialog";
import {HomeHelper} from "@home/helpers/home.helper";
import {ArchiveAccessRightService} from "@home/services/archive-access-right.service";
import {SystemProperty} from "@models";
import {TranslateService} from "@ngx-translate/core";
import {Navigate} from "@ngxs/router-plugin";
import {
  Actions,
  Store,
} from "@ngxs/store";
import {DataTableComponentEnum} from "@shared/enums/data-table-component.enum";
import {IconNameEnum} from "@shared/enums/icon-name.enum";
import {LabelTranslateEnum} from "@shared/enums/label-translate.enum";
import {TourEnum} from "@shared/enums/tour.enum";
import {ViewModeTableEnum} from "@shared/enums/view-mode-table.enum";
import {DataTableActions} from "@shared/models/data-table-actions.model";
import {SharedArchiveAction} from "@shared/stores/archive/shared-archive.action";
import {Observable} from "rxjs";
import {tap} from "rxjs/operators";
import {
  Facet,
  MappingObject,
  MemoizedUtil,
  OAuth2Service,
  OrderEnum,
  Paging,
  QueryParameters,
  QueryParametersUtil,
} from "solidify-frontend";

@Component({
  selector: "dlcm-home-search-routable",
  templateUrl: "./home-search.routable.html",
  styleUrls: ["./home-search.routable.scss"],
  changeDetection: ChangeDetectionStrategy.OnPush,
})
export class HomeSearchRoutable extends SharedAbstractPresentational implements OnInit, OnDestroy {
  searchObs: Observable<string> = MemoizedUtil.select(this.store, HomeState, state => state.search);
  facetsSelectedObs: Observable<MappingObject<string[]>> = MemoizedUtil.select(this.store, HomeState, state => state.facetsSelected);
  queryParametersObs: Observable<QueryParameters> = MemoizedUtil.select(this.store, HomeState, state => state.queryParameters);
  viewModeTableEnumObs: Observable<ViewModeTableEnum> = MemoizedUtil.select(this.store, HomeState, state => state.viewModeTableEnum);
  totalObs: Observable<number> = MemoizedUtil.select(this.store, HomeState, state => state.total);
  listObs: Observable<Archive[]> = MemoizedUtil.select(this.store, HomeState, state => state.list);
  facetsObs: Observable<Facet[]> = MemoizedUtil.select(this.store, HomeState, state => state.facets);
  isLoadingObs: Observable<boolean> = MemoizedUtil.isLoading(this.store, HomeState);
  systemPropertyObs: Observable<SystemProperty> = MemoizedUtil.select(this.store, AppSystemPropertyState, state => state.current);
  isInTourModeObs: Observable<boolean> = MemoizedUtil.select(this.store, AppState, state => state.isInTourMode);
  isLoggedInObs: Observable<boolean> = MemoizedUtil.select(this.store, AppState, state => state.isLoggedIn);
  isFacetClosed: boolean = true;

  get viewModeTableEnum(): typeof ViewModeTableEnum {
    return ViewModeTableEnum;
  }

  get tourEnum(): typeof TourEnum {
    return TourEnum;
  }

  columns: DataTableColumns[] = [
    {
      field: "organizationalUnitId",
      header: LabelTranslateEnum.organizationalUnit,
      type: FieldTypeEnum.string,
      order: OrderEnum.none,
      component: DataTableComponentEnum.organizationalUnitName,
      isSortable: false,
      isFilterable: false,
    },
    {
      field: "title",
      header: LabelTranslateEnum.title,
      type: FieldTypeEnum.string,
      order: OrderEnum.none,
      isSortable: false,
      isFilterable: false,
    },
    {
      field: "publicationDate",
      header: LabelTranslateEnum.publicationDate,
      type: FieldTypeEnum.date,
      order: OrderEnum.none,
      isSortable: false,
      isFilterable: false,
      component: DataTableComponentEnum.archivePublicationDate,
      width: "100px",
    },
    {
      field: "accessLevel",
      header: LabelTranslateEnum.accessLevel,
      type: FieldTypeEnum.string,
      order: OrderEnum.none,
      filterEnum: Enums.Deposit.AccessEnumTranslate,
      component: DataTableComponentEnum.accessLevel,
      translate: true,
      isSortable: false,
      isFilterable: false,
      width: "100px",
    },
    {
      field: "size",
      header: LabelTranslateEnum.size,
      type: FieldTypeEnum.size,
      order: OrderEnum.none,
      isSortable: false,
      isFilterable: false,
      width: "100px",
    },
    {
      field: "files",
      header: LabelTranslateEnum.files,
      type: FieldTypeEnum.number,
      order: OrderEnum.none,
      isSortable: false,
      isFilterable: false,
      width: "100px",
    },
  ];

  actions: DataTableActions<Archive>[] = [
    {
      logo: IconNameEnum.download,
      callback: current => this.download(current),
      placeholder: current => LabelTranslateEnum.download,
      disableCondition: current => this.archiveAccessRightService.isAccessControlled(current),
      displayOnCondition: current => this.store.selectSnapshot((state: LocalStateModel) => state.application.isLoggedIn) || current.accessLevel === Enums.Deposit.AccessEnum.PUBLIC,
    },
    {
      logo: IconNameEnum.addToCart,
      callback: current => this.addToCart(current),
      placeholder: current => LabelTranslateEnum.addToDownloadOrder,
      displayOnCondition: current => MemoizedUtil.selectSnapshot(this.store, AppState, state => state.isLoggedIn),
    },
    {
      logo: IconNameEnum.sendRequest,
      callback: current => this.requestAccessDataset(current),
      placeholder: current => LabelTranslateEnum.requestAccess,
      displayOnCondition: current => !this.archiveAccessRightService.isDownloadAuthorized(current),
    },
  ];

  constructor(private store: Store,
              private translate: TranslateService,
              private readonly _route: ActivatedRoute,
              private readonly _actions$: Actions,
              private readonly _archiveAccessRightService: ArchiveAccessRightService,
              private readonly _oauthService: OAuth2Service,
              protected readonly _dialog: MatDialog,
              readonly archiveAccessRightService: ArchiveAccessRightService) {
    super();
  }

  ngOnInit(): void {
    super.ngOnInit();
    this.subscribe(this.observeSearchQueryParam());
  }

  private observeSearchQueryParam(): Observable<Params> {
    return this._route.queryParams.pipe(tap(params => {
      const searchInfos = HomeHelper.extractSearchInfosFromUrl(params);
      this.store.dispatch(new HomeAction.Search(true, searchInfos.search, searchInfos.facetsSelected, searchInfos.viewTabMode));
    }));
  }

  private requestAccessDataset(archive: Archive): void {
    this._archiveAccessRightService.requestAccessDataset(archive);
  }

  searchText(search: string): void {
    const facetsSelected = MemoizedUtil.selectSnapshot(this.store, HomeState, state => state.facetsSelected);
    const viewMode = MemoizedUtil.selectSnapshot(this.store, HomeState, state => state.viewModeTableEnum);
    this._search(search, facetsSelected, viewMode);
  }

  searchFacet(facetsSelected: MappingObject<string[]>): void {
    const search = MemoizedUtil.selectSnapshot(this.store, HomeState, state => state.search);
    const viewMode = MemoizedUtil.selectSnapshot(this.store, HomeState, state => state.viewModeTableEnum);
    this._search(search, facetsSelected, viewMode);
  }

  switchViewMode(viewMode: ViewModeTableEnum): void {
    const search = MemoizedUtil.selectSnapshot(this.store, HomeState, state => state.search);
    const facetsSelected = MemoizedUtil.selectSnapshot(this.store, HomeState, state => state.facetsSelected);
    this._search(search, facetsSelected, viewMode);
  }

  private _search(searchTerm: string, facetsSelected: MappingObject<string[]>, viewModeTableEnum: ViewModeTableEnum = ViewModeTableEnum.list): void {
    HomeHelper.navigateToSearch(this.store, searchTerm, facetsSelected, viewModeTableEnum);
  }

  onQueryParametersEvent($event: QueryParameters): void {
    this.store.dispatch(new HomeAction.ChangeQueryParameters($event));
  }

  showDetail(archive: Archive): void {
    this.store.dispatch(new Navigate([RoutesEnum.homeDetail, archive.resId]));
  }

  ngOnDestroy(): void {
    super.ngOnDestroy();
  }

  download(archive: Archive): void {
    this.store.dispatch(new SharedArchiveAction.Download(archive));
  }

  addToCart(archive: Archive): void {
    this.store.dispatch(new AppCartAction.AddToCart(archive));
  }

  navigate($event: RoutesEnum): void {
    this.store.dispatch(new Navigate([$event]));
  }

  goToDeposit(): void {
    const isLogged = this.store.selectSnapshot((state: LocalStateModel) => state.application.isLoggedIn) as boolean;
    if (isLogged) {
      this.store.dispatch(new Navigate([RoutesEnum.deposit]));
    } else {
      this._oauthService.initAuthorizationCodeFlow(AppRoutesEnum.deposit);
    }
  }

  pageChange(paging: Paging): void {
    const queryParameters = QueryParametersUtil.clone(MemoizedUtil.selectSnapshot(this.store, HomeState, state => state.queryParameters));
    queryParameters.paging = paging;
    this.store.dispatch(new HomeAction.ChangeQueryParameters(queryParameters));
  }

  openHelp(): void {
    this._dialog.open(HomeSearchHelpDialog, {});
  }
}
