import {
  ChangeDetectionStrategy,
  Component,
  OnDestroy,
  OnInit,
} from "@angular/core";
import {ActivatedRoute} from "@angular/router";
import {HomeState} from "@app/features/home/stores/home.state";
import {SharedAbstractPresentational} from "@app/shared/components/presentationals/shared-abstract/shared-abstract.presentational";
import {
  AppRoutesEnum,
  RoutesEnum,
} from "@app/shared/enums/routes.enum";
import {LocalStateModel} from "@app/shared/models/local-state.model";
import {AppState} from "@app/stores/app.state";
import {AppCarouselState} from "@app/stores/carousel/app-carousel.state";
import {environment} from "@environments/environment";
import {HomeHelper} from "@home/helpers/home.helper";
import {ArchiveAccessRightService} from "@home/services/archive-access-right.service";
import {TranslateService} from "@ngx-translate/core";
import {Navigate} from "@ngxs/router-plugin";
import {
  Actions,
  Store,
} from "@ngxs/store";
import {ThemeEnum} from "@shared/enums/theme.enum";
import {TourEnum} from "@shared/enums/tour.enum";
import {Observable} from "rxjs";
import {
  MemoizedUtil,
  OAuth2Service,
  QueryParameters,
  StringUtil,
} from "solidify-frontend";

@Component({
  selector: "dlcm-home-page-routable",
  templateUrl: "./home-page.routable.html",
  styleUrls: ["./home-page.routable.scss"],
  changeDetection: ChangeDetectionStrategy.OnPush,
})
export class HomePageRoutable extends SharedAbstractPresentational implements OnInit, OnDestroy {
  queryParametersObs: Observable<QueryParameters> = MemoizedUtil.select(this.store, HomeState, state => state.queryParameters);
  isLoadingObs: Observable<boolean> = MemoizedUtil.isLoading(this.store, HomeState);
  themeObs: Observable<ThemeEnum> = MemoizedUtil.select(this.store, AppState, state => state.theme);
  carouselTilesObs: Observable<Node[]> = MemoizedUtil.select(this.store, AppCarouselState, state => state.tiles);

  searchValueInUrl: string = StringUtil.stringEmpty;

  get backgroundImage(): string {
    return `url("./assets/themes/${environment.theme}/home-image.jpg")`;
  }

  get currentThemeName(): string {
    return StringUtil.convertToPascalCase(environment.theme);
  }

  constructor(private store: Store,
              private translate: TranslateService,
              private readonly _route: ActivatedRoute,
              private readonly _actions$: Actions,
              private readonly _archiveAccessRightService: ArchiveAccessRightService,
              private readonly _oauthService: OAuth2Service) {
    super();
  }

  get tourEnum(): typeof TourEnum {
    return TourEnum;
  }

  ngOnInit(): void {
    super.ngOnInit();
  }

  search(searchTerm: string): void {
    HomeHelper.navigateToSearch(this.store, searchTerm);
  }

  ngOnDestroy(): void {
    super.ngOnDestroy();
  }

  navigate($event: RoutesEnum): void {
    this.store.dispatch(new Navigate([$event]));
  }

  goToDeposit(): void {
    const isLogged = this.store.selectSnapshot((state: LocalStateModel) => state.application.isLoggedIn) as boolean;
    if (isLogged) {
      this.store.dispatch(new Navigate([RoutesEnum.deposit]));
    } else {
      this._oauthService.initAuthorizationCodeFlow(AppRoutesEnum.deposit);
    }
  }

  getToSearch(): void {
    this.store.dispatch(new Navigate([RoutesEnum.homeSearch], HomeHelper.getDefaultSearchQueryParams()));
  }
}
