import {
  ChangeDetectionStrategy,
  ChangeDetectorRef,
  Component,
  Inject,
  OnDestroy,
  OnInit,
  ViewChild,
} from "@angular/core";
import {
  MatDialog,
  MatDialogRef,
} from "@angular/material/dialog";
import {ActivatedRoute} from "@angular/router";
import {DepositDataFileHelper} from "@deposit/helpers/deposit-data-file.helper";
import {DepositAction} from "@deposit/stores/deposit.action";
import {Enums} from "@enums";
import {ArchiveDataFile} from "@home/models/archive-data-file.model";
import {HomeArchiveDataFileAction} from "@home/stores/archive/data-file/home-archive-data-file.action";
import {HomeArchiveDataFileState} from "@home/stores/archive/data-file/home-archive-data-file.state";
import {Deposit} from "@models";
import {Navigate} from "@ngxs/router-plugin";
import {
  Actions,
  Store,
} from "@ngxs/store";
import {SharedFileAndAipInformationContainer} from "@shared/components/containers/shared-file-and-aip-information/shared-file-and-aip-information.container";
import {SharedDataTablePresentational} from "@shared/components/presentationals/shared-data-table/shared-data-table.presentational";
import {SharedAbstractRoutable} from "@shared/components/routables/shared-abstract/shared-abstract.routable";
import {DataTableComponentEnum} from "@shared/enums/data-table-component.enum";
import {FieldTypeEnum} from "@shared/enums/field-type.enum";
import {FileViewModeEnum} from "@shared/enums/file-view-mode.enum";
import {LabelTranslateEnum} from "@shared/enums/label-translate.enum";
import {
  AppRoutesEnum,
  HomePageRoutesEnum,
  RoutesEnum,
} from "@shared/enums/routes.enum";
import {
  AbstractFileVisualizer,
  FILE_VISUALIZERS,
} from "@shared/filevisualizer/services/abstract-file-visualizer.service";
import {DataTableActions} from "@shared/models/data-table-actions.model";
import {DataTableBulkActions} from "@shared/models/data-table-bulk-actions.model";
import {DataTableColumns} from "@shared/models/data-table-columns.model";
import {LocalStateModel} from "@shared/models/local-state.model";
import {SecurityService} from "@shared/services/security.service";
import {Observable} from "rxjs";
import {
  distinctUntilChanged,
  tap,
} from "rxjs/operators";
import {
  isUndefined,
  MemoizedUtil,
  OrderEnum,
  QueryParameters,
} from "solidify-frontend";

@Component({
  selector: "dlcm-home-archive-file-list-routable",
  templateUrl: "./home-archive-file-list.routable.html",
  styleUrls: ["./home-archive-file-list.routable.scss"],
  changeDetection: ChangeDetectionStrategy.OnPush,
})
export class HomeArchiveFileListRoutable extends SharedAbstractRoutable implements OnInit, OnDestroy {
  private _archiveId: string;
  private readonly _KEY_QUERY_PARAMETERS: keyof ArchiveDataFile = "status";
  isLoadingDataFileObs: Observable<boolean> = MemoizedUtil.isLoading(this._store, HomeArchiveDataFileState);
  listDataFileObs: Observable<ArchiveDataFile[]> = MemoizedUtil.select(this._store, HomeArchiveDataFileState, state => state.list);
  queryParametersObs: Observable<QueryParameters> = MemoizedUtil.select(this._store, HomeArchiveDataFileState, state => state.queryParameters);
  intermediateFoldersObs: Observable<string[]> = MemoizedUtil.select(this._store, HomeArchiveDataFileState, state => state.intermediateFolders);
  foldersWithIntermediateFoldersObs: Observable<string[]> = MemoizedUtil.select(this._store, HomeArchiveDataFileState, state => state.foldersWithIntermediateFolders);
  currentFolderObs: Observable<string> = MemoizedUtil.select(this._store, HomeArchiveDataFileState, state => state.currentFolder);

  @ViewChild("dataTablePresentational")
  readonly dataTablePresentational: SharedDataTablePresentational<ArchiveDataFile>;

  currentFileViewMode: FileViewModeEnum = FileViewModeEnum.FolderView;

  columns: DataTableColumns<ArchiveDataFile>[];

  actions: DataTableActions<ArchiveDataFile>[] = [
    // {
    //   logo: IconNameEnum.download,
    //   callback: (archiveDataFile: ArchiveDataFile) => this.downloadDataFile(this._resId, archiveDataFile),
    //   placeholder: current => LabelTranslateEnum.download,
    //   displayOnCondition: current => current.status === Enums.DataFile.StatusEnum.PROCESSED || current.status === Enums.DataFile.StatusEnum.READY || current.status === Enums.DataFile.StatusEnum.VIRUS_CHECKED,
    // },
  ];

  bulkActions: DataTableBulkActions<ArchiveDataFile>[] = [];

  columnsToSkippedFilter: keyof ArchiveDataFile[] | string[] = [
    DepositDataFileHelper.RELATIVE_LOCATION,
    this._KEY_QUERY_PARAMETERS,
  ];

  dialogDataFileRef: MatDialogRef<SharedFileAndAipInformationContainer<ArchiveDataFile>>;

  get fileViewModeEnum(): typeof FileViewModeEnum {
    return FileViewModeEnum;
  }

  readonly KEY_PARAM_NAME: keyof Deposit & string = undefined;
  isInDisplayArchiveDataFileListMode: boolean = false;

  constructor(protected readonly _store: Store,
              protected readonly _route: ActivatedRoute,
              protected readonly _actions$: Actions,
              protected readonly _changeDetector: ChangeDetectorRef,
              protected readonly _dialog: MatDialog,
              protected readonly _securityService: SecurityService,
              @Inject(FILE_VISUALIZERS) private readonly _fileVisualizers: AbstractFileVisualizer[]) {
    super();

    this.columns = [
      {
        field: "fileName",
        header: LabelTranslateEnum.fileName,
        type: FieldTypeEnum.string,
        order: OrderEnum.ascending,
        filterableField: "file" as any,
        isSortable: false,
        isFilterable: true,
      },
      {
        field: "creation.when" as any,
        header: LabelTranslateEnum.created,
        type: FieldTypeEnum.datetime,
        order: OrderEnum.none,
        isFilterable: false,
        isSortable: false,
      },
      {
        field: "complianceLevel",
        header: LabelTranslateEnum.complianceLevel,
        type: FieldTypeEnum.singleSelect,
        order: OrderEnum.none,
        sortableField: "dataFile.complianceLevel" as any,
        filterableField: "dataFile.complianceLevel" as any,
        isSortable: false,
        isFilterable: false,
        translate: true,
        width: "185px",
        component: DataTableComponentEnum.conformityLevelStar,
        filterEnum: Enums.DataFile.ComplianceLevelEnumTranslate,
      },
      {
        field: "smartSize",
        header: LabelTranslateEnum.size,
        type: FieldTypeEnum.string,
        order: OrderEnum.none,
        sortableField: "dataFile.fileSize" as any,
        filterableField: "dataFile.fileSize" as any,
        isSortable: false,
        isFilterable: false,
        translate: true,
        width: "100px",
      },
    ];
  }

  ngOnInit(): void {
    super.ngOnInit();
    this.getCurrentModelOnParent();

    this.subscribe(this._store.select((s: LocalStateModel) => s.router.state.url)
      .pipe(
        distinctUntilChanged(),
        tap(url => {
          this.isInDisplayArchiveDataFileListMode = url.endsWith(HomePageRoutesEnum.files);
          this._changeDetector.detectChanges();
        }),
      ));
  }

  private getCurrentModelOnParent(): void {
    this._archiveId = this._route.parent.snapshot.paramMap.get(AppRoutesEnum.paramIdWithoutPrefixParam);
    this.getArchiveById(this._archiveId);
  }

  private getArchiveById(id: string): void {
    this.getSubResourceWithParentId(id);
  }

  getSubResourceWithParentId(id: string): void {
    this._store.dispatch(new HomeArchiveDataFileAction.GetAll(id, undefined, true));
    this._store.dispatch(new HomeArchiveDataFileAction.GetListFolder(id));
  }

  onQueryParametersEvent(queryParameters: QueryParameters, withRefresh: boolean = true): void {
    this._store.dispatch(new HomeArchiveDataFileAction.ChangeQueryParameters(this._archiveId, queryParameters, true, withRefresh));
    // this._changeDetector.detectChanges(); // Allow to display spinner the first time
  }

  download($event: ArchiveDataFile): void {
    this._store.dispatch(new HomeArchiveDataFileAction.Download(this._archiveId, $event));
  }

  selectFolder(folderFullName: string | undefined): void {
    if (isUndefined(folderFullName)) {
      this.currentFileViewMode = FileViewModeEnum.FlatView;
    } else {
      this.currentFileViewMode = FileViewModeEnum.FolderView;
    }
    this._store.dispatch(new HomeArchiveDataFileAction.ChangeCurrentFolder(folderFullName, true));
  }

  downloadFolder(folderName: string): void {
    this._store.dispatch(new DepositAction.Download(this._archiveId, folderName));
  }

  showDetail(archiveDataFile: ArchiveDataFile): void {
    this._store.dispatch(new Navigate([RoutesEnum.homeDetail, this._archiveId, HomePageRoutesEnum.files, archiveDataFile.resId]));
  }

  downloadDataFile(parentId: string, dataFile: ArchiveDataFile): void {
    this._store.dispatch(new HomeArchiveDataFileAction.Download(parentId, dataFile));
  }

  back(): void {
    this._store.dispatch(new Navigate([RoutesEnum.homeDetail, this._archiveId]));
  }
}
