import {
  ChangeDetectionStrategy,
  ChangeDetectorRef,
  Component,
  OnDestroy,
  OnInit,
} from "@angular/core";
import {MatDialog} from "@angular/material/dialog";
import {
  ActivatedRoute,
  Router,
} from "@angular/router";
import {HomeAction} from "@app/features/home/stores/home.action";
import {HomeState} from "@app/features/home/stores/home.state";
import {SharedAbstractPresentational} from "@app/shared/components/presentationals/shared-abstract/shared-abstract.presentational";
import {AppAction} from "@app/stores/app.action";
import {AppState} from "@app/stores/app.state";
import {AppCartAction} from "@app/stores/cart/app-cart.action";
import {environment} from "@environments/environment";
import {HomeHelper} from "@home/helpers/home.helper";
import {ArchiveStatisticsDto} from "@home/models/archive-statistics-dto.model";
import {Archive} from "@home/models/archive.model";
import {ArchiveAccessRightService} from "@home/services/archive-access-right.service";
import {HomeArchivePackageAction} from "@home/stores/archive/package/home-archive-package.action";
import {HomeArchivePackageState} from "@home/stores/archive/package/home-archive-package.state";
import {HomeArchiveRatingAction} from "@home/stores/archive/rating/home-archive-rating.action";
import {HomeArchiveRatingState} from "@home/stores/archive/rating/home-archive-rating.state";
import {ArchiveUserRating} from "@models";
import {Navigate} from "@ngxs/router-plugin";
import {
  Actions,
  ofActionCompleted,
  Store,
} from "@ngxs/store";
import {GetShortDoiWrapper} from "@shared/components/presentationals/shared-doi-menu/shared-doi-menu.presentational";
import {
  AppRoutesEnum,
  HomePageRoutesEnum,
  RoutesEnum,
} from "@shared/enums/routes.enum";
import {UrlQueryParamHelper} from "@shared/helpers/url-query-param.helper";
import {ArchiveMetadataPackages} from "@shared/models/business/archive-metadata-packages.model";
import {LocalStateModel} from "@shared/models/local-state.model";
import {SharedArchiveAction} from "@shared/stores/archive/shared-archive.action";
import {SharedArchiveState} from "@shared/stores/archive/shared-archive.state";
import {Observable} from "rxjs";
import {
  distinctUntilChanged,
  filter,
  take,
  tap,
} from "rxjs/operators";
import {
  isNotNullNorUndefined,
  isNullOrUndefined,
  isTrue,
  MappingObject,
  MappingObjectUtil,
  MARK_AS_TRANSLATABLE,
  MemoizedUtil,
} from "solidify-frontend";

@Component({
  selector: "dlcm-home-archive-detail-routable",
  templateUrl: "./home-archive-detail.routable.html",
  styleUrls: ["./home-archive-detail.routable.scss"],
  changeDetection: ChangeDetectionStrategy.OnPush,
})
export class HomeArchiveDetailRoutable extends SharedAbstractPresentational implements OnInit, OnDestroy {
  isLoadingPrepareDownloadObs: Observable<boolean>;
  isLoadingObs: Observable<boolean> = MemoizedUtil.isLoading(this._store, HomeState);
  currentObs: Observable<Archive> = MemoizedUtil.select(this._store, HomeState, state => state.current);
  archiveStatisticDtoObs: Observable<ArchiveStatisticsDto> = MemoizedUtil.select(this._store, HomeState, state => state.archiveStatisticDto);
  isLoggedInObs: Observable<boolean> = MemoizedUtil.select(this._store, AppState, state => state.isLoggedIn);
  isLoadingPackageObs: Observable<boolean> = MemoizedUtil.isLoading(this._store, HomeArchivePackageState);
  packageObs: Observable<ArchiveMetadataPackages> = MemoizedUtil.select(this._store, HomeArchivePackageState, state => state.package);

  listArchiveFromOrgUnitObs: Observable<Archive[]> = MemoizedUtil.select(this._store, HomeState, state => state.listRelativeArchive);
  isInDisplayArchiveDetailMode: boolean = true;

  listArchiveRatingObs: Observable<ArchiveUserRating[]> = MemoizedUtil.list(this._store, HomeArchiveRatingState);

  private _resId: string;

  constructor(private readonly _store: Store,
              private readonly _dialog: MatDialog,
              private readonly _actions$: Actions,
              private readonly _archiveAccessRightService: ArchiveAccessRightService,
              private readonly _router: Router,
              private readonly _changeDetector: ChangeDetectorRef,
              protected route: ActivatedRoute) {
    super();
  }

  ngOnInit(): void {
    super.ngOnInit();

    this.subscribe(this._store.select((s: LocalStateModel) => s.router.state.url)
      .pipe(
        distinctUntilChanged(),
        tap(url => {
          this.isInDisplayArchiveDetailMode = !url.includes(HomePageRoutesEnum.files) && !url.includes(HomePageRoutesEnum.collections);
          this._changeDetector.detectChanges();
        }),
      ));

    this.subscribe(this.route.paramMap.pipe(
      tap(p => {
        this._retrieveCurrentModelWithUrl();
        this._initIsLoadingPrepareDownloadObs();

        this.subscribe(this.currentObs.pipe(
          filter(archive => isNotNullNorUndefined(archive) && this._resId === archive.resId),
          take(1),
          tap(() => {
            this.checkArchiveQueryParams();
            this._store.dispatch(new HomeArchiveRatingAction.GetAllByCurrentUser(this._resId));
          }),
        ));
      }),
    ));
  }

  ngOnDestroy(): void {
    this._store.dispatch(new HomeAction.CleanCurrent());
  }

  private checkArchiveQueryParams(): void {
    this.subscribe(this._actions$.pipe(
      ofActionCompleted(HomeAction.SearchDetailSuccess),
      filter(result => isNotNullNorUndefined(result) && result.action.model.resId === this._resId),
      take(1),
      tap((result) => {
        if (isTrue(result.result.successful)) {
          this._store.dispatch(new HomeArchivePackageAction.GetById(this._resId));

          const hasAccessRequest = UrlQueryParamHelper.contains(environment.accessRequestRoute);

          if (!this._archiveAccessRightService.isDownloadAuthorized(result.action.model) && hasAccessRequest) {
            const queryParams: MappingObject<string | undefined> = UrlQueryParamHelper.getQueryParamMappingObject();
            const archive = MemoizedUtil.selectSnapshot(this._store, HomeState, state => state.current);
            this._archiveAccessRightService.openRequestArchiveDialog(this._resId, archive.organizationalUnitId);
            // clean url
            if (MappingObjectUtil.has(queryParams, environment.accessRequestRoute)) {
              this._router.navigate([RoutesEnum.homeDetail, this._resId], {queryParams: null});
            }
          }
        }
      }),
    ));
  }

  private _retrieveCurrentModelWithUrl(): void {
    this._resId = this.route.snapshot.paramMap.get(AppRoutesEnum.paramIdWithoutPrefixParam);
    this.getById(this._resId);
  }

  private getById(id: string): void {
    const archive = MemoizedUtil.selectSnapshot(this._store, HomeState, state => state.current);
    if (isNullOrUndefined(archive) || archive.resId !== id) {
      this._store.dispatch(new HomeAction.SearchDetail(id));
    }
    this._store.dispatch(new HomeAction.GetStatistics(id));
  }

  private _initIsLoadingPrepareDownloadObs(): void {
    this.isLoadingPrepareDownloadObs = SharedArchiveState.isDownloadPreparation(this._store, this._resId);
  }

  download(archive: Archive): void {
    this._store.dispatch(new SharedArchiveAction.Download(archive));
  }

  addToCart(archive: Archive): void {
    this._store.dispatch(new AppCartAction.AddToCart(archive));
  }

  askAccess(archive: Archive): void {
    this._archiveAccessRightService.requestAccessDataset(archive);
  }

  back(): void {
    const search = MemoizedUtil.selectSnapshot(this._store, HomeState, state => state.search);
    const viewMode = MemoizedUtil.selectSnapshot(this._store, HomeState, state => state.viewModeTableEnum);
    const facetsSelected = MemoizedUtil.selectSnapshot(this._store, HomeState, state => state.facetsSelected);
    HomeHelper.navigateToSearch(this._store, search, facetsSelected, viewMode);
  }

  getMessageBackButtonToTranslate(): string {
    return MARK_AS_TRANSLATABLE("home.archive.navigation.backToSearch");
  }

  getShortDoi(doiWrapper: GetShortDoiWrapper): void {
    this._store.dispatch(new AppAction.GetShortDoi(doiWrapper));
  }

  showArchive(archive: Archive): void {
    this._store.dispatch([
      new HomeAction.CleanCurrent(),
      new Navigate([RoutesEnum.homeDetail, archive.resId]),
    ]);
  }

  navigate(navigate: Navigate): void {
    this._store.dispatch(navigate);
  }

  rate(listArchiveUserRating: ArchiveUserRating[]): void {
    listArchiveUserRating.forEach(userRating => {
      if (isNullOrUndefined(userRating.user?.resId)) {
        this._store.dispatch(new HomeArchiveRatingAction.Create(this._resId, userRating));
      } else {
        this._store.dispatch(new HomeArchiveRatingAction.Update(this._resId, userRating));
      }
    });
  }
}
