import {
  ChangeDetectionStrategy,
  ChangeDetectorRef,
  Component,
  OnDestroy,
  OnInit,
  ViewChild,
} from "@angular/core";
import {MatDialogRef} from "@angular/material/dialog";
import {ActivatedRoute} from "@angular/router";
import {ArchiveDataFile} from "@home/models/archive-data-file.model";
import {Archive} from "@home/models/archive.model";
import {HomeArchiveCollectionAction} from "@home/stores/archive/collection/home-archive-collection.action";
import {HomeArchiveCollectionState} from "@home/stores/archive/collection/home-archive-collection.state";
import {Navigate} from "@ngxs/router-plugin";
import {
  Actions,
  Store,
} from "@ngxs/store";
import {SharedFileAndAipInformationContainer} from "@shared/components/containers/shared-file-and-aip-information/shared-file-and-aip-information.container";
import {SharedDataTablePresentational} from "@shared/components/presentationals/shared-data-table/shared-data-table.presentational";
import {SharedAbstractRoutable} from "@shared/components/routables/shared-abstract/shared-abstract.routable";
import {FieldTypeEnum} from "@shared/enums/field-type.enum";
import {LabelTranslateEnum} from "@shared/enums/label-translate.enum";
import {
  AppRoutesEnum,
  HomePageRoutesEnum,
  RoutesEnum,
} from "@shared/enums/routes.enum";
import {DataTableActions} from "@shared/models/data-table-actions.model";
import {DataTableBulkActions} from "@shared/models/data-table-bulk-actions.model";
import {DataTableColumns} from "@shared/models/data-table-columns.model";
import {LocalStateModel} from "@shared/models/local-state.model";
import {SecurityService} from "@shared/services/security.service";
import {Observable} from "rxjs";
import {
  distinctUntilChanged,
  tap,
} from "rxjs/operators";
import {
  MemoizedUtil,
  OrderEnum,
  QueryParameters,
} from "solidify-frontend";

@Component({
  selector: "dlcm-home-archive-collection-list-routable",
  templateUrl: "./home-archive-collection-list.routable.html",
  styleUrls: ["./home-archive-collection-list.routable.scss"],
  changeDetection: ChangeDetectionStrategy.OnPush,
})
export class HomeArchiveCollectionListRoutable extends SharedAbstractRoutable implements OnInit, OnDestroy {
  private _archiveId: string;
  private readonly _KEY_QUERY_PARAMETERS: keyof ArchiveDataFile = "status";

  isLoadingDataFileObs: Observable<boolean> = MemoizedUtil.isLoading(this._store, HomeArchiveCollectionState);
  listCollectionObs: Observable<ArchiveDataFile[]> = MemoizedUtil.select(this._store, HomeArchiveCollectionState, state => state.list);
  queryParametersObs: Observable<QueryParameters> = MemoizedUtil.select(this._store, HomeArchiveCollectionState, state => state.queryParameters);

  @ViewChild("dataTablePresentational")
  readonly dataTablePresentational: SharedDataTablePresentational<ArchiveDataFile>;

  columns: DataTableColumns<ArchiveDataFile>[];

  actions: DataTableActions<ArchiveDataFile>[] = [
    // {
    //   logo: IconNameEnum.download,
    //   callback: (archiveDataFile: ArchiveDataFile) => this.downloadDataFile(this._resId, archiveDataFile),
    //   placeholder: current => LabelTranslateEnum.download,
    //   displayOnCondition: current => current.status === Enums.DataFile.StatusEnum.PROCESSED || current.status === Enums.DataFile.StatusEnum.READY || current.status === Enums.DataFile.StatusEnum.VIRUS_CHECKED,
    // },
  ];

  bulkActions: DataTableBulkActions<ArchiveDataFile>[] = [];

  columnsToSkippedFilter: keyof ArchiveDataFile[] | string[] = [
    this._KEY_QUERY_PARAMETERS,
  ];

  dialogDataFileRef: MatDialogRef<SharedFileAndAipInformationContainer<ArchiveDataFile>>;

  readonly KEY_PARAM_NAME: keyof Archive & string = undefined;
  isInDisplayArchiveDataFileListMode: boolean = false;

  constructor(protected readonly _store: Store,
              protected readonly _route: ActivatedRoute,
              protected readonly _actions$: Actions,
              protected readonly _changeDetector: ChangeDetectorRef,
              protected readonly _securityService: SecurityService) {
    super();

    this.columns = [
      {
        field: "resId",
        header: LabelTranslateEnum.archive,
        type: FieldTypeEnum.string,
        order: OrderEnum.ascending,
        isSortable: false,
        isFilterable: false,
      },
      {
        field: "smartSize",
        header: LabelTranslateEnum.size,
        type: FieldTypeEnum.string,
        order: OrderEnum.none,
        sortableField: "dataFile.fileSize" as any,
        filterableField: "dataFile.fileSize" as any,
        isSortable: false,
        isFilterable: false,
        translate: true,
        width: "100px",
      },
    ];
  }

  ngOnInit(): void {
    super.ngOnInit();
    this.getCurrentModelOnParent();

    this.subscribe(this._store.select((s: LocalStateModel) => s.router.state.url)
      .pipe(
        distinctUntilChanged(),
        tap(url => {
          this.isInDisplayArchiveDataFileListMode = url.endsWith(HomePageRoutesEnum.collections);
          this._changeDetector.detectChanges();
        }),
      ));
  }

  private getCurrentModelOnParent(): void {
    this._archiveId = this._route.parent.snapshot.paramMap.get(AppRoutesEnum.paramIdWithoutPrefixParam);
    this.getArchiveById(this._archiveId);
  }

  private getArchiveById(id: string): void {
    this.getSubResourceWithParentId(id);
  }

  getSubResourceWithParentId(id: string): void {
    this._store.dispatch(new HomeArchiveCollectionAction.GetAll(id, undefined, true));
  }

  onQueryParametersEvent(queryParameters: QueryParameters, withRefresh: boolean = true): void {
    this._store.dispatch(new HomeArchiveCollectionAction.ChangeQueryParameters(this._archiveId, queryParameters, true, withRefresh));
    // this._changeDetector.detectChanges(); // Allow to display spinner the first time
  }

  download($event: ArchiveDataFile): void {
    this._store.dispatch(new HomeArchiveCollectionAction.Download(this._archiveId, $event));
  }

  showDetail(archiveDataFile: ArchiveDataFile): void {
    this._store.dispatch(new Navigate([RoutesEnum.homeDetail, archiveDataFile.resId]));
  }

  downloadDataFile(parentId: string, dataFile: ArchiveDataFile): void {
    this._store.dispatch(new HomeArchiveCollectionAction.Download(parentId, dataFile));
  }

  back(): void {
    this._store.dispatch(new Navigate([RoutesEnum.homeDetail, this._archiveId]));
  }
}
