import {
  AfterViewInit,
  ChangeDetectionStrategy,
  Component,
  ElementRef,
  Inject,
  Input,
  Renderer2,
} from "@angular/core";
import {WINDOW} from "@app/app.module";
import {TwitterChromeEnum} from "@app/features/home/enums/twitter-chrome.enum";
import {TwitterSourceTypeEnum} from "@app/features/home/enums/twitter-source-type.enum";
import {TwitterMetadata} from "@app/features/home/models/twitter-metadata.model";
import {
  DataSource,
  Options,
} from "@app/features/home/models/twitter-widget.model";
import {TwitterService} from "@app/features/home/services/twitter.service";
import {SharedAbstractPresentational} from "@app/shared/components/presentationals/shared-abstract/shared-abstract.presentational";
import {
  catchError,
  tap,
} from "rxjs/operators";
import {LoggingService} from "solidify-frontend";

@Component({
  selector: "dlcm-home-twitter-widget-presentational",
  templateUrl: "./home-twitter-widget.presentational.html",
  styleUrls: ["./home-twitter-widget.presentational.scss"],
  changeDetection: ChangeDetectionStrategy.OnPush,
})
export class HomeTwitterWidgetPresentational extends SharedAbstractPresentational implements AfterViewInit {
  @Input()
  sourceType: TwitterSourceTypeEnum | undefined;

  @Input()
  screenName: string | undefined;

  @Input()
  userId: string | undefined;

  @Input()
  tweetLimit: number | undefined;

  @Input()
  borderColor: string | undefined;

  @Input()
  ariaPolite: string | undefined;

  @Input()
  height: number | undefined;

  @Input()
  chrome: TwitterChromeEnum[] = [TwitterChromeEnum.noborders];

  constructor(private _renderer: Renderer2,
              private _element: ElementRef,
              private _twitterService: TwitterService,
              private _loggingService: LoggingService,
              @Inject(WINDOW) private _window: Window) {
    super();
  }

  ngAfterViewInit(): void {
    // https://developer.twitter.com/en/docs/twitter-for-websites/javascript-api/guides/scripting-factory-functions
    this.subscribe(this._twitterService.loadScript(this._renderer)
      .pipe(
        tap((twitterMetadata: TwitterMetadata) => {
          const nativeElement: Element = this._element.nativeElement;
          const dataSource: DataSource = {
            sourceType: this.sourceType,
            screenName: this.screenName,
            userId: this.userId,
          };
          const options: Options = {
            tweetLimit: this.tweetLimit,
            height: this.height,
            chrome: this.chrome,
            ariaPolite: this.ariaPolite,
            borderColor: this.borderColor,
          };

          twitterMetadata.widgets.createTimeline(dataSource, nativeElement, options)
            .then((iframe: any) => {
            })
            .catch((message: string) => {
              this._loggingService.logWarning("Twitter : Could not create widget", message);
            });
        }),
        catchError((err: any) => {
          this._loggingService.logWarning("Twitter : Error loading twitter widget", err);
          throw err;
        }),
      ));
  }
}
