import {
  ChangeDetectionStrategy,
  Component,
  Input,
  OnInit,
  Output,
} from "@angular/core";
import {FormBuilder} from "@angular/forms";
import {MatCheckboxChange} from "@angular/material/checkbox";
import {SharedAbstractPresentational} from "@app/shared/components/presentationals/shared-abstract/shared-abstract.presentational";
import {Enums} from "@enums";
import {environment} from "@environments/environment";
import {FacetProperty} from "@models";
import {BaseFormDefinition} from "@shared/models/base-form-definition.model";
import {
  BehaviorSubject,
  Observable,
} from "rxjs";
import {
  EnumUtil,
  Facet,
  FacetValue,
  isNullOrUndefined,
  MappingObject,
  MappingObjectUtil,
  ObservableUtil,
  PropertyName,
} from "solidify-frontend";
import Deposit = Enums.Deposit;

@Component({
  selector: "dlcm-home-search-facet",
  templateUrl: "./home-search-facet.presentational.html",
  styleUrls: ["./home-search-facet.presentational.scss"],
  changeDetection: ChangeDetectionStrategy.OnPush,
})
export class HomeSearchFacetPresentational extends SharedAbstractPresentational implements OnInit {
  formDefinition: FormComponentFormDefinition = new FormComponentFormDefinition();

  timeBeforeDisplayTooltip: number = environment.timeBeforeDisplayTooltipOnInput;
  numberActiveCriteria: number = 0;
  facetsSelected: MappingObject<string[]> = {};
  openFacets: string[] = [];
  hide: boolean = true;

  @Input()
  searchFacetProperties: FacetProperty[];

  @Input()
  set facetsSelectedInit(value: MappingObject<string[]>) {
    this._initFacetsSelected(value);
  }

  private isEnabled: boolean;

  private _initFacetsSelected(value: MappingObject<string[]>): void {
    if (this.isEnabled === true) {
      return;
    }
    this.facetsSelected = MappingObjectUtil.copy(value);
    this.isEnabled = true;

    this.numberActiveCriteria = 0;
    MappingObjectUtil.forEach(this.facetsSelected, values => this.numberActiveCriteria += values.length);
  }

  private readonly _facetChangeBS: BehaviorSubject<MappingObject<string[]> | undefined> = new BehaviorSubject<MappingObject<string[]> | undefined>(undefined);
  @Output("facetChange")
  readonly facetObs: Observable<MappingObject<string[]> | undefined> = ObservableUtil.asObservable(this._facetChangeBS);

  get mapFacetsEnabledKey(): string[] | IterableIterator<string> {
    return MappingObjectUtil.keys(this.facetsSelected);
  }

  isChecked(key: string, value: string): boolean {
    if (!MappingObjectUtil.has(this.facetsSelected, key)) {
      return false;
    }
    const list = MappingObjectUtil.get(this.facetsSelected, key);
    return list.indexOf(value) !== -1;
  }

  getListValue(key: string): string[] {
    if (!MappingObjectUtil.has(this.facetsSelected, key)) {
      return undefined;
    }
    return MappingObjectUtil.get(this.facetsSelected, key);
  }

  private _listFacets: Facet[];

  @Input()
  set listFacets(value: Facet[]) {
    this._listFacets = value;
  }

  get listFacets(): Facet[] {
    return this._listFacets;
  }

  constructor(private _fb: FormBuilder) {
    super();
  }

  ngOnInit(): void {
    super.ngOnInit();
  }

  clearFilters(): void {
    MappingObjectUtil.clear(this.facetsSelected);
    this.numberActiveCriteria = 0;
    this.dispatchChangeSelectionChange();
  }

  removeCriteria(key: string, value: string): void {
    if (MappingObjectUtil.has(this.facetsSelected, key)) {
      const listFacetsSelected = [...MappingObjectUtil.get(this.facetsSelected, key)];
      this.numberActiveCriteria--;

      const indexOf = listFacetsSelected.indexOf(value);
      if (indexOf !== -1) {
        listFacetsSelected.splice(indexOf, 1);
      }

      if (listFacetsSelected.length === 0) {
        MappingObjectUtil.delete(this.facetsSelected, key);
      } else {
        MappingObjectUtil.set(this.facetsSelected, key, listFacetsSelected);
      }
    }

    this.dispatchChangeSelectionChange();
  }

  changeCheckbox($event: MatCheckboxChange, key: string, value: string): void {
    const isActive = $event.checked;

    if (isActive) {
      this.numberActiveCriteria++;
    } else {
      this.numberActiveCriteria--;
    }

    let listFacetsSelected = [];
    if (MappingObjectUtil.has(this.facetsSelected, key)) {
      listFacetsSelected = [...MappingObjectUtil.get(this.facetsSelected, key)];
    }

    const indexOf = listFacetsSelected.indexOf(value);
    if (isActive && indexOf === -1) {
      listFacetsSelected.push(value);
    } else if (!isActive && indexOf !== -1) {
      listFacetsSelected.splice(indexOf, 1);
    }

    if (listFacetsSelected.length === 0) {
      MappingObjectUtil.delete(this.facetsSelected, key);
    } else {
      MappingObjectUtil.set(this.facetsSelected, key, listFacetsSelected);
    }
    this.dispatchChangeSelectionChange();
  }

  dispatchChangeSelectionChange(): void {
    this._facetChangeBS.next(this.facetsSelected);
  }

  getFacetValueLabel(facet: Facet, value: FacetValue): string {
    return this.getFacetValueLabelWithName(facet.name, value.value);
  }

  getFacetValueLabelWithName(facetName: string, facetValueValue: string): string {
    switch (facetName) {
      case Enums.Facet.Name.ACCESS_LEVELS:
        return EnumUtil.getLabel(Deposit.AccessEnumTranslate, facetValueValue);
      case Enums.Facet.Name.DATA_TAGS:
        return EnumUtil.getLabel(Deposit.DataSensitivityEnumTranslate, facetValueValue);
      default:
        return facetValueValue;
    }
  }

  isFacetValueVisible(facet: Facet, i: number): boolean {
    if (this.openFacets.indexOf(facet.name) !== -1) {
      return true;
    } else {
      const facetProperty = this.getSearchFacetProperty(facet.name);
      if (isNullOrUndefined(facetProperty.defaultVisibleValues)) {
        return true;
      } else {
        return i < facetProperty.defaultVisibleValues;
      }
    }
  }

  facetHasMoreValuesThanDefaultVisible(facet: Facet): boolean {
    const facetProperty = this.getSearchFacetProperty(facet.name);
    if (isNullOrUndefined(facetProperty.defaultVisibleValues)) {
      return false;
    } else {
      return facet.values.length > facetProperty.defaultVisibleValues;
    }
  }

  displayShowAllFacetValue(facet: Facet): boolean {
    return (this.facetHasMoreValuesThanDefaultVisible(facet) && this.openFacets.indexOf(facet.name) === -1);
  }

  displayShowLessFacetValue(facet: Facet): boolean {
    return (this.facetHasMoreValuesThanDefaultVisible(facet) && this.openFacets.indexOf(facet.name) !== -1);
  }

  getSearchFacetProperty(name: string): FacetProperty {
    for (const facetProperty of this.searchFacetProperties) {
      if (facetProperty.name === name) {
        return facetProperty;
      }
    }
  }

  showAllFacetValues(facet: Facet): void {
    this.openFacets.push(facet.name);
  }

  showLessFacetValues(facet: Facet): void {
    const index = this.openFacets.indexOf(facet.name);
    if (index > -1) {
      this.openFacets.splice(index, 1);
    }
  }

  toggleDisplayPanel(): void {
    this.hide = !this.hide;
  }

  hidePanel(): void {
    this.hide = true;
  }
}

class FormComponentFormDefinition extends BaseFormDefinition {
  @PropertyName() nameValue: string;
  @PropertyName() name: string;
  @PropertyName() value: string;
  @PropertyName() isActive: string;
}
