import {
  ChangeDetectionStrategy,
  ChangeDetectorRef,
  Component,
  Input,
  OnChanges,
  OnInit,
  Output,
  SimpleChanges,
} from "@angular/core";
import {
  AbstractControl,
  FormBuilder,
  FormGroup,
} from "@angular/forms";
import {SharedAbstractPresentational} from "@app/shared/components/presentationals/shared-abstract/shared-abstract.presentational";
import {ArchiveMetadata} from "@app/shared/models/business/archive-metadata.model";
import {MetadataUtil} from "@app/shared/utils/metadata.util";
import {Enums} from "@enums";
import {ArchiveStatisticsDto} from "@home/models/archive-statistics-dto.model";
import {
  Archive,
  ArchiveContributor,
} from "@home/models/archive.model";
import {AverageRating} from "@home/models/average-rating.model";
import {ArchiveAccessRightService} from "@home/services/archive-access-right.service";
import {ArchiveUserRating} from "@models";
import {Navigate} from "@ngxs/router-plugin";
import {GetShortDoiWrapper} from "@shared/components/presentationals/shared-doi-menu/shared-doi-menu.presentational";
import {ApiActionEnum} from "@shared/enums/api-action.enum";
import {AccessResourceApiEnum} from "@shared/enums/api.enum";
import {LabelTranslateEnum} from "@shared/enums/label-translate.enum";
import {
  HomePageRoutesEnum,
  RoutesEnum,
} from "@shared/enums/routes.enum";
import {ViewModeTableEnum} from "@shared/enums/view-mode-table.enum";
import {BaseFormDefinition} from "@shared/models/base-form-definition.model";
import {
  ArchiveMetadataPackages,
  MetadataPackagesEnum,
} from "@shared/models/business/archive-metadata-packages.model";
import {KeyValueInfo} from "@shared/models/key-value-info.model";
import {SecurityService} from "@shared/services/security.service";
import {RegexpUtil} from "@shared/utils/regexp.util";
import {
  BehaviorSubject,
  Observable,
} from "rxjs";
import {
  ClipboardUtil,
  DateUtil,
  EnumUtil,
  isNonEmptyString,
  isNotNullNorUndefined,
  isNullOrUndefined,
  NotificationService,
  ObservableUtil,
  PropertyName,
} from "solidify-frontend";

@Component({
  selector: "dlcm-home-archive-detail",
  templateUrl: "./home-archive-detail.presentational.html",
  styleUrls: ["./home-archive-detail.presentational.scss"],
  changeDetection: ChangeDetectionStrategy.OnPush,
})
export class HomeArchiveDetailPresentational extends SharedAbstractPresentational implements OnInit, OnChanges {
  formDefinition: FormComponentFormDefinition = new FormComponentFormDefinition();

  private readonly _downloadBS: BehaviorSubject<Archive | undefined> = new BehaviorSubject<Archive | undefined>(undefined);
  @Output("downloadChange")
  readonly downloadObs: Observable<Archive | undefined> = ObservableUtil.asObservable(this._downloadBS);

  private readonly _showArchiveBS: BehaviorSubject<Archive | undefined> = new BehaviorSubject<Archive | undefined>(undefined);
  @Output("showArchiveChange")
  readonly showArchiveObs: Observable<Archive | undefined> = ObservableUtil.asObservable(this._showArchiveBS);

  private readonly _addToCartBS: BehaviorSubject<Archive | undefined> = new BehaviorSubject<Archive | undefined>(undefined);
  @Output("addToCartChange")
  readonly addToCartObs: Observable<Archive | undefined> = ObservableUtil.asObservable(this._addToCartBS);

  private readonly _askAccessBS: BehaviorSubject<Archive | undefined> = new BehaviorSubject<Archive | undefined>(undefined);
  @Output("askAccessChange")
  readonly askAccessObs: Observable<Archive | undefined> = ObservableUtil.asObservable(this._askAccessBS);

  private readonly _navigateBS: BehaviorSubject<Navigate> = new BehaviorSubject<Navigate>(undefined);
  @Output("navigateChange")
  readonly navigateObs: Observable<Navigate> = ObservableUtil.asObservable(this._navigateBS);

  private readonly _rateBS: BehaviorSubject<ArchiveUserRating[]> = new BehaviorSubject<ArchiveUserRating[]>(undefined);
  @Output("rateChange")
  readonly rateObs: Observable<ArchiveUserRating[]> = ObservableUtil.asObservable(this._rateBS);

  @Input()
  archive: Archive;

  @Input()
  archiveStatisticDto: ArchiveStatisticsDto;

  @Input()
  archivePackages: ArchiveMetadataPackages;

  @Input()
  listArchiveUserRating: ArchiveUserRating[];

  @Input()
  relativeArchive: Archive[];

  get metadata(): ArchiveMetadata.PublicMetadata | undefined {
    if (isNullOrUndefined(this.archive) || isNullOrUndefined(this.archive.archiveMetadata)) {
      return undefined;
    }
    return this.archive.archiveMetadata.metadata;
  }

  listInfo: Info[];

  imageArchive: string | undefined;

  @Input()
  isLoadingPrepareDownload: boolean;

  @Input()
  isLoggedIn: boolean;

  viewModeList: ViewModeTableEnum = ViewModeTableEnum.tiles;

  get viewModeListEnum(): typeof ViewModeTableEnum {
    return ViewModeTableEnum;
  }

  private readonly _shortDoiBS: BehaviorSubject<GetShortDoiWrapper | undefined> = new BehaviorSubject<GetShortDoiWrapper | undefined>(undefined);
  @Output("getShortDoi")
  readonly shortDoiObs: Observable<GetShortDoiWrapper | undefined> = ObservableUtil.asObservable(this._shortDoiBS);

  isRatingEditableMode: boolean = false;

  formRatingQuality: FormGroup;
  formRatingUserfulness: FormGroup;

  get typeInfoEnum(): typeof TypeInfoEnum {
    return TypeInfoEnum;
  }

  get enumUtil(): typeof EnumUtil {
    return EnumUtil;
  }

  constructor(protected readonly _securityService: SecurityService,
              protected readonly _notificationService: NotificationService,
              readonly archiveAccessRightService: ArchiveAccessRightService,
              private readonly _fb: FormBuilder,
              private readonly _cd: ChangeDetectorRef) {
    super();
  }

  ngOnInit(): void {
    this.imageArchive = `${AccessResourceApiEnum.publicMetadata}/${this.archive.resId}/${ApiActionEnum.THUMBNAIL}`;
    this._defineListInfo();

    if (isNullOrUndefined(this.formRatingQuality)) {
      this.formRatingQuality = this.createFormGroupRating(Enums.Archive.RatingTypeEnum.QUALITY);
    }
    if (isNullOrUndefined(this.formRatingUserfulness)) {
      this.formRatingUserfulness = this.createFormGroupRating(Enums.Archive.RatingTypeEnum.USEFULNESS);
    }
  }

  private createFormGroupRating(ratingType: Enums.Archive.RatingTypeEnum, archiveUserRating: ArchiveUserRating = undefined): FormGroup {
    const formGroup = this._fb.group({
      [this.formDefinition.ratingType]: [ratingType, []],
      [this.formDefinition.grade]: [isNullOrUndefined(archiveUserRating) ? 0 : archiveUserRating.grade, []],
    });
    if (isNotNullNorUndefined(archiveUserRating) && ratingType === archiveUserRating.ratingType?.resId) {
      formGroup.addControl(this.formDefinition.userId, this._fb.control(archiveUserRating.user?.resId));
      formGroup.addControl(this.formDefinition.archiveId, this._fb.control(archiveUserRating.resId));
    }
    return formGroup;
  }

  ngOnChanges(changes: SimpleChanges): void {
    super.ngOnChanges(changes);
    if (isNotNullNorUndefined(changes.listArchiveUserRating?.currentValue)) {
      const listArchiveUserRating: ArchiveUserRating[] = changes.listArchiveUserRating.currentValue;
      listArchiveUserRating.forEach(archiveUserRating => {
        switch (archiveUserRating.ratingType?.resId) {
          case Enums.Archive.RatingTypeEnum.QUALITY:
            this.formRatingQuality = this.createFormGroupRating(Enums.Archive.RatingTypeEnum.QUALITY, archiveUserRating);
            break;
          case Enums.Archive.RatingTypeEnum.USEFULNESS:
            this.formRatingUserfulness = this.createFormGroupRating(Enums.Archive.RatingTypeEnum.USEFULNESS, archiveUserRating);
            break;
          default:
            break;
        }
      });
    }
  }

  private _defineListInfo(): void {
    this.listInfo = [
      {
        labelToTranslate: LabelTranslateEnum.organizationalUnit,
        value: this.archive.organizationalUnitId,
        type: TypeInfoEnum.orgUnit,
      },
      {
        labelToTranslate: LabelTranslateEnum.type,
        value: MetadataUtil.getType(this.metadata) === "DATASET" ? LabelTranslateEnum.dataset : MetadataUtil.getType(this.metadata),
        type: TypeInfoEnum.translate,
      },
      {
        labelToTranslate: LabelTranslateEnum.doi,
        value: MetadataUtil.getDOI(this.metadata),
        type: TypeInfoEnum.doi,
      },
      {
        labelToTranslate: LabelTranslateEnum.license,
        value: MetadataUtil.getLicenses(this.metadata),
      },
      {
        labelToTranslate: LabelTranslateEnum.embargoAccessLevel,
        value: EnumUtil.getLabel(Enums.Deposit.AccessEnumTranslate, MetadataUtil.getEmbargoAccessLevel(this.metadata)),
        type: TypeInfoEnum.translate,
      },
      {
        labelToTranslate: LabelTranslateEnum.embargoEndDate,
        value: DateUtil.convertDateToDateString(MetadataUtil.getEmbargoEndDate(this.metadata)),
      },
      {
        labelToTranslate: LabelTranslateEnum.submittedDate,
        value: DateUtil.convertDateToDateTimeString(MetadataUtil.getSubmittedDate(this.metadata)),
      },
      {
        labelToTranslate: LabelTranslateEnum.descriptionOther,
        value: MetadataUtil.getDescriptionOther(this.metadata),
      },
      {
        labelToTranslate: LabelTranslateEnum.descriptionTechnicalInfo,
        value: MetadataUtil.getDescriptionTechnicalInfo(this.metadata),
      },
      {
        labelToTranslate: LabelTranslateEnum.descriptionSeriesInformation,
        value: MetadataUtil.getDescriptionSeriesInformation(this.metadata),
      },
      {
        labelToTranslate: LabelTranslateEnum.descriptionTableOfContents,
        value: MetadataUtil.getDescriptionTableOfContents(this.metadata),
      },
      {
        labelToTranslate: LabelTranslateEnum.descriptionMethods,
        value: MetadataUtil.getDescriptionMethods(this.metadata),
      },
    ];
    this.listInfo = this.listInfo.filter(i => isNonEmptyString(i.value));
  }

  download(archive: Archive): void {
    this._downloadBS.next(archive);
  }

  getMetadataDatacite(): string {
    return MetadataUtil.getDatacite(this.metadata);
  }

  getShortDoi(longDoi: GetShortDoiWrapper): void {
    this._shortDoiBS.next(longDoi);
  }

  addToCart(archive: Archive): void {
    this._addToCartBS.next(archive);
  }

  askAccess(archive: Archive): void {
    this._askAccessBS.next(archive);
  }

  showDetail(archive: Archive): void {
    this._showArchiveBS.next(archive);
  }

  labelToTranslate(ratingType: string): string {
    return this.enumUtil.getLabel(Enums.Archive.RatingTypeEnumTranslate, ratingType);
  }

  navigateToFilesList(): void {
    if (this.archive.isCollection) {
      this._navigateBS.next(new Navigate([RoutesEnum.homeDetail, this.archive.resId, HomePageRoutesEnum.collections]));
    } else {
      this._navigateBS.next(new Navigate([RoutesEnum.homeDetail, this.archive.resId, HomePageRoutesEnum.files]));
    }
  }

  getMetadataVersion(archivePackages: ArchiveMetadataPackages): string {
    return archivePackages.metadata[MetadataPackagesEnum.metadataVersion];
  }

  getRetentionExpiration(archivePackages: ArchiveMetadataPackages): Date {
    const retentionEnd = archivePackages.metadata[MetadataPackagesEnum.aipRetentionEnd];
    return DateUtil.convertOffsetDateTimeIso8601ToDate(retentionEnd);
  }

  getDisposalApproval(archivePackages: ArchiveMetadataPackages): string {
    return archivePackages.metadata[MetadataPackagesEnum.aipDispositionApproval] ? LabelTranslateEnum.yes : LabelTranslateEnum.no;
  }

  getCreationDate(archivePackages: ArchiveMetadataPackages): Date {
    const creation = archivePackages.metadata[MetadataPackagesEnum.creation];
    return DateUtil.convertOffsetDateTimeIso8601ToDate(creation);
  }

  copyToClipboard(id: string): void {
    ClipboardUtil.copyStringToClipboard(id);
    this._notificationService.showInformation(LabelTranslateEnum.idCopyToClipboard);
  }

  getDepositCreationDate(archivePackages: ArchiveMetadataPackages): Date {
    const creation = archivePackages.metadata.packages.deposits.events.creation?.date;
    return DateUtil.convertOffsetDateTimeIso8601ToDate(creation);
  }

  getDepositApprobationDate(archivePackages: ArchiveMetadataPackages): Date {
    const approval = archivePackages.metadata.packages.deposits.events?.appraisal?.date;
    return DateUtil.convertOffsetDateTimeIso8601ToDate(approval);
  }

  submitRating(): void {
    const list: ArchiveUserRating[] = [];
    if (!this.formRatingQuality.pristine) {
      list.push(this._createRating(this.formRatingQuality));
    }
    if (!this.formRatingUserfulness.pristine) {
      list.push(this._createRating(this.formRatingUserfulness));
    }
    if (list.length > 0) {
      this._rateBS.next(list);
    }
    this.leaveRatingMode();
  }

  private _createRating(form: FormGroup): ArchiveUserRating {
    const rating: ArchiveUserRating = {
      archiveId: this.archive.resId,
      grade: form.get(this.formDefinition.grade).value,
      ratingType: {
        resId: form.get(this.formDefinition.ratingType).value,
      },
    };
    const userId = form.get(this.formDefinition.userId)?.value;
    if (isNotNullNorUndefined(userId)) {
      rating.user = {
        resId: userId,
      };
    }
    return rating;
  }

  enterInRatingMode(): void {
    this.isRatingEditableMode = true;
  }

  leaveRatingMode(): void {
    this.isRatingEditableMode = false;
  }

  getFormGroupRating(averageRating: AverageRating): AbstractControl | undefined {
    switch (averageRating.ratingType) {
      case Enums.Archive.RatingTypeEnum.QUALITY:
        return this.formRatingQuality;
      case Enums.Archive.RatingTypeEnum.USEFULNESS:
        return this.formRatingUserfulness;
      default:
        return undefined;
    }
  }

  getSensitivityTooltip(dataSensitivity: Enums.Deposit.DataSensitivityEnum): string {
    return (EnumUtil.getKeyValue(Enums.Deposit.DataSensitivityEnumTranslate, dataSensitivity) as KeyValueInfo)?.infoToTranslate;
  }

  getContributorTooltip(contributor: ArchiveContributor): string | undefined {
    if (contributor.affiliation.length === 0) {
      return undefined;
    }
    return contributor.affiliation.join(", ");
  }

  computeLink(text: string): string {
    if (isNullOrUndefined(text)) {
      return text;
    }
    return text.replace(RegexpUtil.urlInComplexText, match => `<a href="${match}" target="_blank" class="no-hover-animation">${match}</a>`);
  }
}

class Info {
  labelToTranslate: string;
  value?: string;
  code?: boolean = false;
  type?: TypeInfoEnum;
}

enum TypeInfoEnum {
  doi = 1,
  orgUnit = 2,
  translate = 3,
}

class FormComponentFormDefinition extends BaseFormDefinition {
  @PropertyName() ratingType: string;
  @PropertyName() grade: string;
  @PropertyName() userId: string;
  @PropertyName() archiveId: string;
}
