import {
  HttpClient,
  HttpErrorResponse,
  HttpEventType,
  HttpHeaders,
} from "@angular/common/http";
import {Injectable} from "@angular/core";
import {FileUploadStatusEnum} from "@app/features/deposit/enums/file-upload-status.enum";
import {DataFileUploadHelper} from "@app/features/deposit/helpers/data-file-upload.helper";
import {DepositDataFile} from "@app/features/deposit/models/deposit-data-file.model";
import {UploadEventModel} from "@app/features/deposit/models/upload-event.model";
import {UploadFileStatus} from "@app/features/deposit/models/upload-file-status.model";
import {DepositDataFileAction} from "@app/features/deposit/stores/data-file/deposit-data-file.action";
import {
  defaultDepositDataFileValue,
  DepositDataFileState,
  DepositDataFileStateModel,
} from "@app/features/deposit/stores/data-file/deposit-data-file.state";
import {
  DepositAction,
  depositActionNameSpace,
} from "@app/features/deposit/stores/deposit.action";
import {
  DepositPeopleState,
  DepositPersonStateModel,
} from "@app/features/deposit/stores/people/deposit-people.state";
import {DepositPersonAction} from "@app/features/deposit/stores/people/deposit-person.action";
import {ApiActionEnum} from "@app/shared/enums/api-action.enum";
import {PreIngestResourceApiEnum} from "@app/shared/enums/api.enum";
import {ErrorBackendKeyEnum} from "@app/shared/enums/error-backend-key.enum";
import {LocalStateEnum} from "@app/shared/enums/local-state.enum";
import {LocalModelAttributeEnum} from "@app/shared/enums/model-attribute.enum";
import {
  DepositRoutesEnum,
  RoutesEnum,
} from "@app/shared/enums/routes.enum";
import {AppAction} from "@app/stores/app.action";
import {AppAuthorizedOrganizationalUnitState} from "@app/stores/authorized-organizational-unit/app-authorized-organizational-unit.state";
import {DepositFormPresentational} from "@deposit/components/presentationals/deposit-form/deposit-form.presentational";
import {DepositTabStatusEnum} from "@deposit/enums/deposit-tab-status.enum";
import {ModeDepositTabEnum} from "@deposit/enums/mode-deposit-tab.enum";
import {
  DepositAipState,
  DepositAipStateModel,
} from "@deposit/stores/aip/deposit-aip.state";
import {
  defaultDepositAuthorizedOrgUnitValue,
  DepositAuthorizedOrganizationalUnitState,
  DepositAuthorizedOrganizationalUnitStateModel,
} from "@deposit/stores/authorized-organizational-unit/deposit-authorized-organizational-unit.state";
import {
  defaultDepositCollectionValue,
  DepositCollectionState,
  DepositCollectionStateModel,
} from "@deposit/stores/collection/deposit-collection.state";
import {DepositOrganizationalUnitAction} from "@deposit/stores/organizational-unit/deposit-organizational-unit.action";
import {
  defaultDepositOrganizationalUnitValue,
  DepositOrganizationalUnitState,
  DepositOrganizationalUnitStateModel,
} from "@deposit/stores/organizational-unit/deposit-organizational-unit.state";
import {
  DepositStatusHistoryState,
  DepositStatusHistoryStateModel,
} from "@deposit/stores/status-history/deposit-status-history.state";
import {Enums} from "@enums";
import {environment} from "@environments/environment";
import {Deposit} from "@models";
import {Navigate} from "@ngxs/router-plugin";
import {
  Action,
  Actions,
  ofActionCompleted,
  Selector,
  State,
  StateContext,
  Store,
} from "@ngxs/store";
import {ApiResourceNameEnum} from "@shared/enums/api-resource-name.enum";
import {PollingHelper} from "@shared/helpers/polling.helper";
import {DataFile} from "@shared/models/business/data-file.model";
import {FileListModel} from "@shared/models/business/file-list.model";
import {Result} from "@shared/models/business/result.model";
import {LocalStateModel} from "@shared/models/local-state.model";
import {DownloadService} from "@shared/services/download.service";
import {SecurityService} from "@shared/services/security.service";
import {SharedLanguageAction} from "@shared/stores/language/shared-language.action";
import {ResourceLogoStateModel} from "@shared/stores/resource-logo/resource-logo-state.model";
import {
  defaultResourceLogoStateInitValue,
  ResourceLogoState,
  ResourceLogoStateModeEnum,
} from "@shared/stores/resource-logo/resource-logo.state";
import {defaultStatusHistoryInitValue} from "@shared/stores/status-history/status-history.state";
import {
  Observable,
  of,
} from "rxjs";
import {
  catchError,
  map,
  switchMap,
  take,
  tap,
} from "rxjs/operators";
import {
  ApiService,
  CollectionTyped,
  defaultAssociationRemoteStateInitValue,
  defaultResourceStateInitValue,
  ErrorDto,
  ErrorHelper,
  HttpStatus,
  isEmptyArray,
  isNonEmptyString,
  isNotNullNorUndefined,
  isNullOrUndefined,
  isTrue,
  isUndefined,
  MappingObjectUtil,
  MARK_AS_TRANSLATABLE,
  MemoizedUtil,
  NotificationService,
  ObjectUtil,
  OverrideDefaultAction,
  QueryParameters,
  QueryParametersUtil,
  SolidifyStateError,
  StoreUtil,
  StringUtil,
  urlSeparator,
} from "solidify-frontend";

export interface DepositStateModel extends ResourceLogoStateModel<Deposit> {
  deposit_authorizedOrganizationalUnit: DepositAuthorizedOrganizationalUnitStateModel;
  deposit_dataFile: DepositDataFileStateModel;
  deposit_person: DepositPersonStateModel;
  deposit_collection: DepositCollectionStateModel;
  deposit_aip: DepositAipStateModel;
  deposit_organizationalUnit: DepositOrganizationalUnitStateModel;
  isLoadingDataFile: boolean;
  uploadStatus: UploadFileStatus[];
  deposit_statusHistory: DepositStatusHistoryStateModel;
  counterTabInProgress: number | undefined;
  counterTabInValidation: number | undefined;
  counterTabApproved: number | undefined;
  counterTabCompleted: number | undefined;
  counterTabRejected: number | undefined;
  counterTabInError: number | undefined;
  counterTabAll: number | undefined;
  excludedFileList: FileListModel | undefined;
  ignoredFileList: FileListModel | undefined;
  canEdit: boolean;
  canCreate: boolean;
  formPresentational: DepositFormPresentational | undefined;
  depositModeTabEnum: ModeDepositTabEnum;
  dataFileLogo: DataFile | undefined;
  activeListTabStatus: DepositTabStatusEnum;
}

@Injectable()
@State<DepositStateModel>({
  name: LocalStateEnum.deposit,
  defaults: {
    ...defaultResourceLogoStateInitValue(),
    deposit_authorizedOrganizationalUnit: defaultDepositAuthorizedOrgUnitValue(),
    queryParameters: new QueryParameters(environment.defaultPageSize),
    deposit_dataFile: defaultDepositDataFileValue(),
    deposit_person: {...defaultAssociationRemoteStateInitValue()},
    deposit_collection: defaultDepositCollectionValue(),
    deposit_aip: {...defaultResourceStateInitValue()},
    deposit_organizationalUnit: defaultDepositOrganizationalUnitValue(),
    isLoadingDataFile: false,
    uploadStatus: [],
    deposit_statusHistory: {...defaultStatusHistoryInitValue()},
    counterTabInProgress: undefined,
    counterTabInValidation: undefined,
    counterTabApproved: undefined,
    counterTabCompleted: undefined,
    counterTabRejected: undefined,
    counterTabInError: undefined,
    counterTabAll: undefined,
    excludedFileList: undefined,
    ignoredFileList: undefined,
    canEdit: false,
    canCreate: false,
    formPresentational: undefined,
    depositModeTabEnum: ModeDepositTabEnum.UNDEFINED,
    dataFileLogo: undefined,
    isLoadingLogo: false,
    activeListTabStatus: DepositTabStatusEnum.inProgress,
  },
  children: [
    DepositDataFileState,
    DepositPeopleState,
    DepositStatusHistoryState,
    DepositOrganizationalUnitState,
    DepositCollectionState,
    DepositAipState,
    DepositAuthorizedOrganizationalUnitState,
  ],
})
export class DepositState extends ResourceLogoState<DepositStateModel, Deposit> {
  private readonly _CATEGORY_KEY: string = "category";
  private readonly _TYPE_KEY: string = "type";
  private readonly _METADATA_TYPE_KEY: string = "metadataType";
  private readonly _FOLDER_KEY: string = "folder";
  private readonly _ZIP_EXTENSION: string = ".zip";
  private readonly _DEPOSIT_FILE_DOWNLOAD_PREFIX: string = "deposit_";
  private readonly _FOLDER_QUERY_PARAM: string = "folder";
  private readonly _DEPOSIT_REJECT_REASON: string = "reason";

  constructor(protected apiService: ApiService,
              protected store: Store,
              protected notificationService: NotificationService,
              protected actions$: Actions,
              protected readonly _securityService: SecurityService,
              private downloadService: DownloadService,
              protected _httpClient: HttpClient) {
    super(apiService, store, notificationService, actions$, {
      nameSpace: depositActionNameSpace,
      routeRedirectUrlAfterSuccessDeleteAction: RoutesEnum.deposit,
      notificationResourceCreateSuccessTextToTranslate: MARK_AS_TRANSLATABLE("deposit.notification.resource.create"),
      notificationResourceDeleteSuccessTextToTranslate: MARK_AS_TRANSLATABLE("deposit.notification.resource.delete"),
      notificationResourceUpdateSuccessTextToTranslate: MARK_AS_TRANSLATABLE("deposit.notification.resource.update"),
      keepCurrentStateAfterUpdate: true,
      keepCurrentStateBeforeCreate: true,
    }, _httpClient, ResourceLogoStateModeEnum.dataset);
  }

  protected get _urlResource(): string {
    return PreIngestResourceApiEnum.deposits;
  }

  protected get _urlLogoResource(): string {
    return this._urlResource;
  }

  @Selector()
  static isLoading(state: DepositStateModel): boolean {
    return StoreUtil.isLoadingState(state);
  }

  @Selector()
  static currentOrgUnitName(state: DepositStateModel): string | undefined {
    if (isNullOrUndefined(state.deposit_organizationalUnit) || isNullOrUndefined(state.deposit_organizationalUnit.current)) {
      return undefined;
    }
    return state.deposit_organizationalUnit.current.name;
  }

  @Selector()
  static currentTitle(state: DepositStateModel): string | undefined {
    if (isNullOrUndefined(state.current)) {
      return undefined;
    }
    return state.current.title;
  }

  @Selector()
  static isLoadingWithDependency(state: DepositStateModel): boolean {
    return this.isLoading(state)
      || StoreUtil.isLoadingState(state.deposit_person);
  }

  @Selector()
  static isReadyToBeDisplayed(state: DepositStateModel): boolean {
    return this.isReadyToBeDisplayedInCreateMode
      && !isNullOrUndefined(state.current)
      && !isNullOrUndefined(state.deposit_person.selected);
  }

  @Selector()
  static isReadyToBeDisplayedInCreateMode(state: DepositStateModel): boolean {
    return !isNullOrUndefined(state.deposit_organizationalUnit) && !isNullOrUndefined(state.deposit_organizationalUnit.current)
      && state.deposit_organizationalUnit.deposit_organizationalUnit_additionalFieldsForm.loaded;
  }

  @Action(DepositAction.LoadResource)
  loadResource(ctx: StateContext<DepositStateModel>, action: DepositAction.LoadResource): Observable<any> {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
    });

    return StoreUtil.dispatchParallelActionAndWaitForSubActionsCompletion(ctx, [
      {
        action: new SharedLanguageAction.GetAll(null, false, false),
        subActionCompletions: [
          this.actions$.pipe(ofActionCompleted(SharedLanguageAction.GetAllSuccess)),
          this.actions$.pipe(ofActionCompleted(SharedLanguageAction.GetAllFail)),
        ],
      },
    ]).pipe(
      tap(success => {
        if (success) {
          ctx.dispatch(new DepositAction.LoadResourceSuccess(action));
        } else {
          ctx.dispatch(new DepositAction.LoadResourceFail(action));
        }
      }),
    );
  }

  @Action(DepositAction.SetOrganizationalUnit)
  setOrganizationalUnit(ctx: StateContext<DepositStateModel>, action: DepositAction.SetOrganizationalUnit): void {
    const listAuthorizedOrgUnit = MemoizedUtil.listSnapshot(this.store, AppAuthorizedOrganizationalUnitState);
    if (isNullOrUndefined(listAuthorizedOrgUnit) || isEmptyArray(listAuthorizedOrgUnit)) {
      return;
    }
    const canCreate = this._securityService.canCreateDepositOnOrgUnit(action.organizationalUnitId);
    ctx.dispatch(new DepositAction.CanCreate(canCreate));
    const orgUnit = listAuthorizedOrgUnit.find(o => o.resId === action.organizationalUnitId);
    ctx.patchState({
      counterTabInProgress: undefined,
      counterTabInValidation: undefined,
      counterTabApproved: undefined,
      counterTabCompleted: undefined,
      counterTabRejected: undefined,
      counterTabInError: undefined,
      counterTabAll: undefined,
    });
    ctx.dispatch(new DepositOrganizationalUnitAction.GetByIdSuccess(action as any, orgUnit));
  }

  @OverrideDefaultAction()
  @Action(DepositAction.GetAllSuccess)
  getAllSuccess(ctx: StateContext<DepositStateModel>, action: DepositAction.GetAllSuccess): void {
    super.getAllSuccess(ctx, action);
    const searchItem = QueryParametersUtil.getSearchItems(ctx.getState().queryParameters);
    if (isNullOrUndefined(searchItem)) {
      return;
    }
    const statusKey: keyof Deposit = "status";
    const organizationalUnitIdKey: keyof Deposit = "organizationalUnitId";
    const numberInput = MappingObjectUtil.size(searchItem);
    const hasOrgUnitId = MappingObjectUtil.has(searchItem, organizationalUnitIdKey);
    const hasStatus = MappingObjectUtil.has(searchItem, statusKey);
    if ((numberInput === 2 && hasOrgUnitId && hasStatus) || (numberInput === 1 && hasOrgUnitId)) {
      let status: Enums.Deposit.StatusEnum | undefined = undefined;
      if (hasStatus) {
        status = MappingObjectUtil.get(searchItem, statusKey) as Enums.Deposit.StatusEnum;
      }
      ctx.dispatch(new DepositAction.RefreshCounterStatusTabSuccess(null, status, action.list._page.totalItems));
    }
  }

  @OverrideDefaultAction()
  @Action(DepositAction.Create)
  create(ctx: StateContext<DepositStateModel>, action: DepositAction.Create): Observable<Deposit> {
    return super.internalCreate(ctx, action)
      .pipe(
        tap(model => {
          this.updateSubResource(model, action, ctx)
            .subscribe(success => {
              if (success) {
                ctx.dispatch(new DepositAction.CreateSuccess(action, model));
              } else {
                ctx.dispatch(new DepositAction.CreateFail(action));
              }
            });
        }),
      );
  }

  @OverrideDefaultAction()
  @Action(DepositAction.CreateSuccess)
  createSuccess(ctx: StateContext<DepositStateModel>, action: DepositAction.CreateSuccess): void {
    super.createSuccess(ctx, action);
    ctx.dispatch([
      new Navigate([RoutesEnum.deposit, action.model.organizationalUnitId, DepositRoutesEnum.detail, action.model.resId, DepositRoutesEnum.data]),
      new DepositAction.SetOrganizationalUnit(action.model.organizationalUnitId),
    ]);
  }

  @OverrideDefaultAction()
  @Action(DepositAction.Update)
  update(ctx: StateContext<DepositStateModel>, action: DepositAction.Update): Observable<Deposit> {
    action.modelFormControlEvent.model.status = Enums.Deposit.StatusEnum.INPROGRESS;
    return super.internalUpdate(ctx, action)
      .pipe(
        tap(model => {
          this.updateSubResource(model, action, ctx)
            .subscribe(success => {
              if (success) {
                ctx.dispatch(new DepositAction.UpdateSuccess(action, model));
              } else {
                ctx.dispatch(new DepositAction.UpdateFail(action));
              }
            });
        }),
      );
  }

  @OverrideDefaultAction()
  @Action(DepositAction.UpdateSuccess)
  updateSuccess(ctx: StateContext<DepositStateModel>, action: DepositAction.UpdateSuccess): void {
    super.updateSuccess(ctx, action);

    const depositId = action.model.resId;
    ctx.dispatch([
      new DepositPersonAction.GetAll(depositId),
      new Navigate([RoutesEnum.deposit, action.model.organizationalUnitId, DepositRoutesEnum.detail, action.model.resId, DepositRoutesEnum.metadata]),
      new DepositAction.SetOrganizationalUnit(action.model.organizationalUnitId),
    ]);
  }

  private updateSubResource(model: Deposit, action: DepositAction.Create | DepositAction.Update, ctx: StateContext<DepositStateModel>): Observable<boolean> {
    const depositId = model.resId;
    const selectedPerson = ctx.getState().deposit_person.selected;
    const oldAuthors: string[] = isNullOrUndefined(selectedPerson) ? [] : selectedPerson.map(p => p.resId);
    const newAuthors: string[] = action.modelFormControlEvent.model[LocalModelAttributeEnum.authors];
    // Need to delete before submit new authors to update order
    return StoreUtil.dispatchSequentialActionAndWaitForSubActionsCompletion(ctx, [
      {
        action: new DepositPersonAction.DeleteList(depositId, oldAuthors),
        subActionCompletions: [
          this.actions$.pipe(ofActionCompleted(DepositPersonAction.DeleteListSuccess)),
          this.actions$.pipe(ofActionCompleted(DepositPersonAction.DeleteListFail)),
        ],
      },
      {
        action: new DepositPersonAction.Create(depositId, newAuthors),
        subActionCompletions: [
          this.actions$.pipe(ofActionCompleted(DepositPersonAction.CreateSuccess)),
          this.actions$.pipe(ofActionCompleted(DepositPersonAction.CreateFail)),
        ],
      },
    ]);
  }

  @Action(DepositAction.Submit)
  submit(ctx: StateContext<DepositStateModel>, action: DepositAction.Submit): Observable<Result> {
    ctx.dispatch(new Navigate([RoutesEnum.deposit, ctx.getState().deposit_organizationalUnit.current.resId, DepositRoutesEnum.detail, action.deposit.resId]));

    let submissionWithApproval = false;
    let submissionPolicy = action.deposit.submissionPolicy;
    if (isNullOrUndefined(submissionPolicy)) {
      submissionPolicy = action.deposit.organizationalUnit.defaultSubmissionPolicy;
    }
    if (!isNullOrUndefined(submissionPolicy)) {
      submissionWithApproval = submissionPolicy.submissionApproval;
    }

    let suffix = ApiActionEnum.APPROVE;
    if (isTrue(submissionWithApproval)) {
      suffix = ApiActionEnum.SUBMIT_FOR_APPROVAL;
    }
    return this.apiService.post<any, Result>(this._urlResource + urlSeparator + action.deposit.resId + urlSeparator + suffix)
      .pipe(
        tap(result => {
          if (result?.status === Enums.Result.ActionStatusEnum.EXECUTED) {
            ctx.dispatch(new DepositAction.SubmitSuccess(result));
          } else {
            ctx.dispatch(new DepositAction.SubmitFail());
          }
        }),
        catchError(error => {
          ctx.dispatch(new DepositAction.SubmitFail(ErrorHelper.extractValidationErrors(error)));
          throw new SolidifyStateError(this, error);
        }),
      );
  }

  @Action(DepositAction.SubmitSuccess)
  submitSuccess(ctx: StateContext<DepositStateModel>, action: DepositAction.SubmitSuccess): void {
    this.redirectToDepositDetail(ctx, action.result.resId);
    this.notificationService.showSuccess(MARK_AS_TRANSLATABLE("deposit.notification.submit.success"));
  }

  @Action(DepositAction.SubmitFail)
  submitFail(ctx: StateContext<DepositStateModel>, action: DepositAction.SubmitFail): void {
    let errorsMessage = StringUtil.stringEmpty;
    if (isNotNullNorUndefined(action.listValidationError)) {
      const errorMessageSeparator = ". ";
      action.listValidationError.forEach(validationError => {
        if (isNonEmptyString(errorsMessage)) {
          errorsMessage = errorMessageSeparator;
        }
        if (isNotNullNorUndefined(validationError.errorMessages)) {
          errorsMessage = errorsMessage + validationError.errorMessages.join(errorMessageSeparator);
        }
      });
    }
    this.notificationService.showError(MARK_AS_TRANSLATABLE("deposit.notification.submit.fail"), {errors: errorsMessage});
  }

  @Action(DepositAction.Approve)
  approve(ctx: StateContext<DepositStateModel>, action: DepositAction.Approve): Observable<Result> {
    return this.apiService.post<any, Result>(this._urlResource + urlSeparator + action.deposit.resId + urlSeparator + ApiActionEnum.APPROVE)
      .pipe(
        tap(result => {
          if (result?.status === Enums.Result.ActionStatusEnum.EXECUTED) {
            ctx.dispatch(new DepositAction.ApproveSuccess(result));
          } else {
            ctx.dispatch(new DepositAction.ApproveFail());
          }
        }),
        catchError(error => {
          ctx.dispatch(new DepositAction.ApproveFail());
          throw new SolidifyStateError(this, error);
        }),
      );
  }

  @Action(DepositAction.ApproveSuccess)
  approveSuccess(ctx: StateContext<DepositStateModel>, action: DepositAction.ApproveSuccess): void {
    this.redirectToDepositDetail(ctx, action.result.resId);
    this.notificationService.showSuccess(MARK_AS_TRANSLATABLE("deposit.notification.approve.success"));
  }

  @Action(DepositAction.ApproveFail)
  approveFail(ctx: StateContext<DepositStateModel>, action: DepositAction.ApproveFail): void {
    this.notificationService.showError(MARK_AS_TRANSLATABLE("deposit.notification.approve.fail"));
  }

  @Action(DepositAction.Reject)
  reject(ctx: StateContext<DepositStateModel>, action: DepositAction.Reject): Observable<Result> {
    return this.apiService.post<any, Result>(`${this._urlResource + urlSeparator + action.deposit.resId + urlSeparator + ApiActionEnum.REJECT}?${this._DEPOSIT_REJECT_REASON}=${action.message}`)
      .pipe(
        tap(result => {
          if (result?.status === Enums.Result.ActionStatusEnum.EXECUTED) {
            ctx.dispatch(new DepositAction.RejectSuccess(result));
          } else {
            ctx.dispatch(new DepositAction.RejectFail());
          }
        }),
        catchError(error => {
          ctx.dispatch(new DepositAction.RejectFail());
          throw new SolidifyStateError(this, error);
        }),
      );
  }

  @Action(DepositAction.RejectSuccess)
  rejectSuccess(ctx: StateContext<DepositStateModel>, action: DepositAction.RejectSuccess): void {
    this.redirectToDepositDetail(ctx, action.result.resId);
    this.notificationService.showSuccess(MARK_AS_TRANSLATABLE("deposit.notification.reject.success"));
  }

  @Action(DepositAction.RejectFail)
  rejectFail(ctx: StateContext<DepositStateModel>, action: DepositAction.RejectFail): void {
    this.notificationService.showError(MARK_AS_TRANSLATABLE("deposit.notification.reject.fail"));
  }

  @Action(DepositAction.BackToEdit)
  backToEdit(ctx: StateContext<DepositStateModel>, action: DepositAction.BackToEdit): Observable<Result> {
    return this.apiService.post<Result>(this._urlResource + urlSeparator + action.deposit.resId + urlSeparator + ApiActionEnum.ENABLE_REVISION)
      .pipe(
        tap(result => ctx.dispatch(new DepositAction.BackToEditSuccess(result))),
        catchError(error => {
          ctx.dispatch(new DepositAction.BackToEditFail());
          throw new SolidifyStateError(this, error);
        }),
      );
  }

  @Action(DepositAction.BackToEditSuccess)
  backToEditSuccess(ctx: StateContext<DepositStateModel>, action: DepositAction.BackToEditSuccess): void {
    this.redirectToDepositDetail(ctx, action.result.resId);
    this.notificationService.showSuccess(MARK_AS_TRANSLATABLE("deposit.notification.backToEdit.success"));
  }

  @Action(DepositAction.BackToEditFail)
  backToEditFail(ctx: StateContext<DepositStateModel>, action: DepositAction.BackToEditFail): void {
    this.notificationService.showError(MARK_AS_TRANSLATABLE("deposit.notification.backToEdit.fail"));
  }

  @Action(DepositAction.UploadDataFile)
  uploadDataFile(ctx: StateContext<DepositStateModel>, action: DepositAction.UploadDataFile): Observable<void> {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
    });

    const formData = new FormData();
    let apiAction = ApiActionEnum.UL;
    if (action.isArchive) {
      apiAction = ApiActionEnum.UL_ARCHIVE;
    } else {
      formData.append(this._FOLDER_KEY, isNullOrUndefined(action.fileUploadWrapper.subDirectory) ? StringUtil.stringEmpty : action.fileUploadWrapper.subDirectory);
    }
    formData.append(this._FILE_KEY, action.fileUploadWrapper.file, action.fileUploadWrapper.file.name);
    formData.append(this._CATEGORY_KEY, action.fileUploadWrapper.dataCategory);
    formData.append(this._TYPE_KEY, action.fileUploadWrapper.dataType);
    const metadataType = action.fileUploadWrapper.metadataType;
    if (!isNullOrUndefined(metadataType) && isNonEmptyString(metadataType)) {
      formData.append(this._METADATA_TYPE_KEY, metadataType);
    }
    const uploadFileStatus = DataFileUploadHelper.addToUploadStatus(ctx, action.fileUploadWrapper, action.isArchive);
    ctx.dispatch(new AppAction.PreventExit(MARK_AS_TRANSLATABLE("notification.uploadInProgress")));

    return this.apiService.upload<DepositDataFile>(`${PreIngestResourceApiEnum.deposits}/${action.parentId}/${apiAction}`, formData)
      .pipe(
        map((event: UploadEventModel) => {
          switch (event.type) {
            case HttpEventType.UploadProgress:
              const currentUploadFileStatus = DataFileUploadHelper.getCurrentUploadFileStatus(ctx, uploadFileStatus.guid);
              if (currentUploadFileStatus.status === FileUploadStatusEnum.canceled) {
                ctx.dispatch(new DepositAction.CancelDataFileSending(action.parentId, uploadFileStatus));
                throw environment.errorToSkipInErrorHandler;
              }
              DataFileUploadHelper.updateInProgressUploadFileStatus(ctx, currentUploadFileStatus, event);
              return;
            case HttpEventType.Response:
              if (event.status === HttpStatus.OK) {
                ctx.dispatch(new DepositAction.UploadDataFileSuccess(action, action.parentId, uploadFileStatus, event.body));
              } else {
                ctx.dispatch(new DepositAction.UploadDataFileFail(action, uploadFileStatus, undefined));
              }
              return;
            default:
              return;
          }
        }),
        catchError((error: Error | HttpErrorResponse) => {
          if (error === environment.errorToSkipInErrorHandler) {
            throw new SolidifyStateError(this, error);
          }
          let errorDto: ErrorDto = undefined;
          if (error instanceof HttpErrorResponse) {
            errorDto = error.error;
          }
          ctx.dispatch(new DepositAction.UploadDataFileFail(action, uploadFileStatus, errorDto));
          throw new SolidifyStateError(this, error);
        }),
      );
  }

  @Action(DepositAction.UploadDataFileSuccess)
  uploadDataFileSuccess(ctx: StateContext<DepositStateModel>, action: DepositAction.UploadDataFileSuccess): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
    this.notificationService.showSuccess(MARK_AS_TRANSLATABLE("deposit.file.upload.notification.success"));
    const uploadFileStatusCopied = ObjectUtil.clone(action.uploadFileStatus);
    DataFileUploadHelper.updateCompletedUploadFileStatus(ctx, uploadFileStatusCopied, action.depositDataFile);
    if (!this.checkIsPendingDownload(ctx)) {
      ctx.dispatch(new DepositDataFileAction.Refresh(action.parentId));
    }
  }

  @Action(DepositAction.UploadDataFileFail)
  uploadDataFileFail(ctx: StateContext<DepositStateModel>, action: DepositAction.UploadDataFileFail): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
    let errorMessage = MARK_AS_TRANSLATABLE("deposit.file.upload.notification.error");
    if (action.errorDto.message === ErrorBackendKeyEnum.UPLOAD_DUPLICATE_DATA_FILES) {
      errorMessage = MARK_AS_TRANSLATABLE("error.upload.duplicateDataFiles");
    }
    this.notificationService.showError(errorMessage);
    const uploadFileStatusCopied = ObjectUtil.clone(action.uploadFileStatus);
    DataFileUploadHelper.updateErrorUploadFileStatus(ctx, uploadFileStatusCopied, errorMessage);
    this.checkIsPendingDownload(ctx);
  }

  private checkIsPendingDownload(ctx: StateContext<DepositStateModel>): boolean {
    const isPendingDownload = ctx.getState().uploadStatus
      .filter(u => u.status === FileUploadStatusEnum.inProgress || u.status === FileUploadStatusEnum.started).length > 0;
    if (!isPendingDownload) {
      ctx.dispatch(new AppAction.CancelPreventExit());
    }
    return isPendingDownload;
  }

  @Action(DepositAction.RetrySendDataFile)
  retrySendDataFile(ctx: StateContext<DepositStateModel>, action: DepositAction.RetrySendDataFile): void {
    DataFileUploadHelper.removeToUploadStatus(ctx, action.uploadFileStatus);
    ctx.dispatch(new DepositAction.UploadDataFile(action.parentId, action.uploadFileStatus.fileUploadWrapper, action.uploadFileStatus.isArchive));
  }

  @Action(DepositAction.MarkAsCancelDataFileSending)
  markAsCancelDataFileSending(ctx: StateContext<DepositStateModel>, action: DepositAction.MarkAsCancelDataFileSending): void {
    const uploadFileStatus = DataFileUploadHelper.getCurrentUploadFileStatus(ctx, action.uploadFileStatus.guid);
    if (uploadFileStatus.status === FileUploadStatusEnum.inProgress) {
      DataFileUploadHelper.updateCancelUploadFileStatus(ctx, uploadFileStatus);
    } else {
      ctx.dispatch(new DepositAction.CancelDataFileSending(action.parentId, uploadFileStatus));
    }
  }

  @Action(DepositAction.CancelDataFileSending)
  cancelDataFileSending(ctx: StateContext<DepositStateModel>, action: DepositAction.MarkAsCancelDataFileSending): void {
    DataFileUploadHelper.removeToUploadStatus(ctx, action.uploadFileStatus);
    this.notificationService.showInformation(MARK_AS_TRANSLATABLE("deposit.file.upload.notification.info.canceled"));
    this.checkIsPendingDownload(ctx);
  }

  @Action(DepositAction.ReserveDOI)
  reserveDOI(ctx: StateContext<DepositStateModel>, action: DepositAction.ReserveDOI): Observable<Deposit> {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
    });
    return this.apiService.post(this._urlResource + urlSeparator + action.deposit.resId + urlSeparator + ApiActionEnum.RESERVE_DOI)
      .pipe(
        tap(deposit => ctx.dispatch(new DepositAction.ReserveDOISuccess(deposit))),
        catchError(error => {
          ctx.dispatch(new DepositAction.ReserveDOIFail());
          throw new SolidifyStateError(this, error);
        }),
      );
  }

  @Action(DepositAction.ReserveDOISuccess)
  reserveDOISuccess(ctx: StateContext<DepositStateModel>, action: DepositAction.ReserveDOISuccess): void {
    // We want to explicitly set to undefined before set the new value
    ctx.patchState({
      current: undefined,
    });
    ctx.patchState({
      current: action.deposit,
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
    this.notificationService.showSuccess(MARK_AS_TRANSLATABLE("deposit.notification.reserveDOI.success"));
  }

  @Action(DepositAction.ReserveDOIFail)
  reserveDOIFail(ctx: StateContext<DepositStateModel>, action: DepositAction.ReserveDOIFail): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
    this.notificationService.showError(MARK_AS_TRANSLATABLE("deposit.notification.reserveDOI.fail"));
  }

  @Action(DepositAction.RefreshAllCounterStatusTab)
  refreshAllCounterStatusTab(ctx: StateContext<DepositStateModel>, action: DepositAction.RefreshAllCounterStatusTab): Observable<any> {
    const listActionsWrappper = [];
    action.list.forEach(status => {
      listActionsWrappper.push({
        action: new DepositAction.RefreshCounterStatusTab(status),
        subActionCompletions: [
          this.actions$.pipe(ofActionCompleted(DepositAction.RefreshCounterStatusTabSuccess)),
          this.actions$.pipe(ofActionCompleted(DepositAction.RefreshCounterStatusTabFail)),
        ],
      });
    });

    return StoreUtil.dispatchParallelActionAndWaitForSubActionsCompletion(ctx, listActionsWrappper)
      .pipe(
        tap(success => {
          if (success) {
            ctx.dispatch(new DepositAction.RefreshAllCounterStatusTabSuccess(action));
          } else {
            ctx.dispatch(new DepositAction.RefreshAllCounterStatusTabFail(action));
          }
        }),
      );
  }

  @Action(DepositAction.RefreshCounterStatusTab)
  refreshCounterStatusTab(ctx: StateContext<DepositStateModel>, action: DepositAction.RefreshCounterStatusTab): Observable<number> {
    const queryParameters = new QueryParameters(environment.minimalPageSizeToRetrievePaginationInfo);
    const searchItems = QueryParametersUtil.getSearchItems(queryParameters);
    if (!isNullOrUndefined(ctx.getState().deposit_organizationalUnit) && !isNullOrUndefined(ctx.getState().deposit_organizationalUnit.current)) {
      MappingObjectUtil.set(searchItems, "organizationalUnitId", ctx.getState().deposit_organizationalUnit.current.resId);
    }
    if (!isUndefined(action.status)) {
      MappingObjectUtil.set(searchItems, "status", action.status);
    }
    return this.apiService.get<Deposit>(this._urlResource, queryParameters)
      .pipe(
        map((collection: CollectionTyped<Deposit>) => {
          const totalItem = collection._page.totalItems;
          ctx.dispatch(new DepositAction.RefreshCounterStatusTabSuccess(action, action.status, totalItem));
          return totalItem;
        }),
        catchError(error => {
          ctx.dispatch(new DepositAction.RefreshCounterStatusTabFail(action));
          throw new SolidifyStateError(this, error);
        }),
      );
  }

  @Action(DepositAction.RefreshCounterStatusTabSuccess)
  refreshCounterStatusTabSuccess(ctx: StateContext<DepositStateModel>, action: DepositAction.RefreshCounterStatusTabSuccess): void {
    let key: keyof DepositStateModel | undefined = undefined;
    switch (action.status) {
      case Enums.Deposit.StatusEnum.INPROGRESS:
        key = "counterTabInProgress";
        break;
      case Enums.Deposit.StatusEnum.INVALIDATION:
        key = "counterTabInValidation";
        break;
      case Enums.Deposit.StatusEnum.APPROVED:
        key = "counterTabApproved";
        break;
      case Enums.Deposit.StatusEnum.COMPLETED:
        key = "counterTabCompleted";
        break;
      case Enums.Deposit.StatusEnum.REJECTED:
        key = "counterTabRejected";
        break;
      case Enums.Deposit.StatusEnum.INERROR:
        key = "counterTabInError";
        break;
      case undefined:
        key = "counterTabAll";
        break;
    }

    ctx.patchState({
      [key]: action.counter,
    });
  }

  @Action(DepositAction.RefreshCounterStatusTabFail)
  refreshCounterStatusTabFail(ctx: StateContext<DepositStateModel>, action: DepositAction.RefreshCounterStatusTabFail): void {
  }

  @OverrideDefaultAction()
  @Action(DepositAction.Clean)
  clean(ctx: StateContext<DepositStateModel>, action: DepositAction.Clean): void {
    const canCreate = ctx.getState().canCreate;
    const depositOrganizationalUnitStateModel = ctx.getState().deposit_organizationalUnit;
    const ignoredFileList = ctx.getState().ignoredFileList;
    const excludedFileList = ctx.getState().excludedFileList;
    const isLoadingCounter = ctx.getState().isLoadingCounter;
    super.clean(ctx, action);
    ctx.patchState({
      canCreate,
      deposit_organizationalUnit: depositOrganizationalUnitStateModel,
      ignoredFileList,
      excludedFileList,
      isLoadingCounter,
    });
  }

  private redirectToDepositDetail(ctx: StateContext<DepositStateModel>, depositId: string): void {
    ctx.dispatch(new Navigate([RoutesEnum.deposit, ctx.getState().deposit_organizationalUnit.current.resId, DepositRoutesEnum.detail, depositId]));
    ctx.dispatch(new DepositAction.GetById(depositId));
  }

  @Action(DepositAction.ChangeEditProperty)
  changeEditProperty(ctx: StateContext<DepositStateModel>, action: DepositAction.ChangeEditProperty): void {
    ctx.patchState({
      canEdit: action.isEditable,
    });
  }

  @Action(DepositAction.GetExcludedListFiles)
  getExcludedListFiles(ctx: StateContext<DepositStateModel>, action: DepositAction.GetExcludedListFiles): Observable<FileListModel> {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
    });
    return this.apiService.getByIdInPath<FileListModel>(this._urlResource + urlSeparator + ApiActionEnum.LIST_EXCLUDE_FILES)
      .pipe(
        tap(fileList => ctx.dispatch(new DepositAction.GetExcludedListFilesSuccess(action, fileList))),
        catchError(error => {
          ctx.dispatch(new DepositAction.GetExcludedListFilesFail(action));
          throw new SolidifyStateError(this, error);
        }),
      );
  }

  @Action(DepositAction.GetExcludedListFilesSuccess)
  getExcludedListFilesSuccess(ctx: StateContext<DepositStateModel>, action: DepositAction.GetExcludedListFilesSuccess): void {
    ctx.patchState({
      excludedFileList: action.fileList,
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
  }

  @Action(DepositAction.GetExcludedListFilesFail)
  getExcludedListFilesFail(ctx: StateContext<DepositStateModel>, action: DepositAction.GetExcludedListFilesFail): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
  }

  @Action(DepositAction.GetIgnoredListFiles)
  getIgnoredListFiles(ctx: StateContext<DepositStateModel>, action: DepositAction.GetIgnoredListFiles): Observable<FileListModel> {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
    });
    return this.apiService.getByIdInPath<FileListModel>(this._urlResource + urlSeparator + ApiActionEnum.LIST_IGNORE_FILES)
      .pipe(
        tap(fileList => ctx.dispatch(new DepositAction.GetIgnoredListFilesSuccess(action, fileList))),
        catchError(error => {
          ctx.dispatch(new DepositAction.GetIgnoredListFilesFail(action));
          throw new SolidifyStateError(this, error);
        }),
      );
  }

  @Action(DepositAction.GetIgnoredListFilesSuccess)
  getIgnoredListFilesSuccess(ctx: StateContext<DepositStateModel>, action: DepositAction.GetIgnoredListFilesSuccess): void {
    ctx.patchState({
      ignoredFileList: action.fileList,
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
  }

  @Action(DepositAction.GetIgnoredListFilesFail)
  getIgnoredListFilesFail(ctx: StateContext<DepositStateModel>, action: DepositAction.GetIgnoredListFilesFail): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
  }

  @Action(DepositAction.CanCreate)
  canCreate(ctx: StateContext<DepositStateModel>, action: DepositAction.CanCreate): void {
    ctx.patchState({
      canCreate: action.canCreate,
    });
  }

  @Action(DepositAction.Download)
  download(ctx: StateContext<DepositStateModel>, action: DepositAction.Download): void {
    let fileName = this._DEPOSIT_FILE_DOWNLOAD_PREFIX + action.id;
    let url = `${this._urlResource}/${action.id}/${ApiActionEnum.DL}`;
    if (isNotNullNorUndefined(action.fullFolderName)) {
      url = url + `?${this._FOLDER_QUERY_PARAM}=${action.fullFolderName}`;
      fileName = fileName + action.fullFolderName;
    }
    this.downloadService.download(false, url, fileName + this._ZIP_EXTENSION);
  }

  @Action(DepositAction.ComputeModeTab)
  computeModeTab(ctx: StateContext<DepositStateModel>, action: DepositAction.ComputeModeTab): void {
    let mode: ModeDepositTabEnum;

    const url = this.store.selectSnapshot((s: LocalStateModel) => s.router.state.url);
    const currentDisplay = url.substring(url.lastIndexOf(urlSeparator) + 1, url.length);
    const baseUrl = url.substring(0, url.lastIndexOf(urlSeparator));
    const numberCollection = ctx.getState()?.deposit_collection?.numberCollections;
    const numberFiles = ctx.getState()?.deposit_dataFile?.numberFiles;

    let redirectionRouteSuffix = undefined;
    if (numberCollection > 0) {
      mode = ModeDepositTabEnum.COLLECTION;
      if (currentDisplay === DepositRoutesEnum.data || currentDisplay === DepositRoutesEnum.files) {
        redirectionRouteSuffix = DepositRoutesEnum.collections;
      }
    } else if (numberFiles > 0) {
      mode = ModeDepositTabEnum.FILE;
      if (currentDisplay === DepositRoutesEnum.data || currentDisplay === DepositRoutesEnum.collections) {
        redirectionRouteSuffix = DepositRoutesEnum.files;
      }
    } else {
      mode = ModeDepositTabEnum.FILE_OR_COLLECTION;
      if (currentDisplay === DepositRoutesEnum.files || currentDisplay === DepositRoutesEnum.collections) {
        redirectionRouteSuffix = DepositRoutesEnum.data;
      }
    }
    ctx.patchState({
      depositModeTabEnum: mode,
    });
    if (isNotNullNorUndefined(redirectionRouteSuffix)) {
      ctx.dispatch(new Navigate([baseUrl, redirectionRouteSuffix]));
    }
  }

  @Action(DepositAction.SetActiveListTabStatus)
  setActiveListTabStatus(ctx: StateContext<DepositStateModel>, action: DepositAction.SetActiveListTabStatus): void {
    ctx.patchState({
      activeListTabStatus: action.activeListTabStatus,
    });
  }

  @OverrideDefaultAction()
  @Action(DepositAction.UploadPhoto)
  uploadPhoto(ctx: StateContext<DepositStateModel>, action: DepositAction.UploadPhoto): Observable<any> {
    if (isNotNullNorUndefined(ctx.getState().dataFileLogo)) {
      ctx.dispatch(new DepositAction.DeletePhoto(action.resId, action.file));
      return this.actions$.pipe(
        ofActionCompleted(DepositAction.DeletePhotoSuccess),
        switchMap(result => {
          if (isTrue(result.result.successful)) {
            return this._uploadPhotoReady(ctx, action);
          }
          return of(false);
        }));
    } else {
      return this._uploadPhotoReady(ctx, action);
    }
  }

  private _uploadPhotoReady(ctx: StateContext<DepositStateModel>, action: DepositAction.UploadPhoto): Observable<any> {
    const parentId = ctx.getState().current?.resId;
    const actionUploadDataFile = new DepositAction.UploadDataFile(parentId, {
      dataCategory: Enums.DataFile.DataCategoryEnum.Internal,
      dataType: Enums.DataFile.DataTypeEnum.DatasetThumbnail,
      metadataType: undefined,
      subDirectory: undefined,
      file: action.file,
    });
    this.actions$.pipe(
      ofActionCompleted(DepositAction.UploadDataFileSuccess),
      take(1),
      tap((result) => {
        const actionUpdateSuccess = result.action as DepositAction.UploadDataFileSuccess;
        if (isTrue(result.result.successful) && actionUpdateSuccess.parentAction === actionUploadDataFile) {
          const actionUploadDatafileSuccess = actionUpdateSuccess.depositDataFile[0];
          ctx.patchState({
            dataFileLogo: actionUploadDatafileSuccess,
          });
          ctx.dispatch(new DepositAction.UploadPhotoSuccess(action));
        }
      }),
    ).subscribe();
    this.actions$.pipe(
      ofActionCompleted(DepositAction.UploadDataFileFail),
      take(1),
      tap((result) => {
        const actionUpdateSuccess = result.action as DepositAction.UploadDataFileFail;
        if (isTrue(result.result.successful) && actionUpdateSuccess.parentAction === actionUploadDataFile) {
          ctx.dispatch(new DepositAction.UploadPhotoFail(action));
        }
      }),
    ).subscribe();
    return ctx.dispatch(actionUploadDataFile);
  }

  @Action(DepositAction.CheckPhoto)
  checkPhoto(ctx: StateContext<DepositStateModel>, action: DepositAction.CheckPhoto): Observable<any> {
    const depositId = action.depositId;

    const queryParameters = new QueryParameters(1);
    const searchItems = QueryParametersUtil.getSearchItems(queryParameters);
    MappingObjectUtil.set(searchItems, "dataCategory", Enums.DataFile.DataCategoryEnum.Internal);
    MappingObjectUtil.set(searchItems, "dataType", Enums.DataFile.DataTypeEnum.DatasetThumbnail);
    return this.apiService.get<DataFile>(this._urlResource + urlSeparator + depositId + urlSeparator + ApiResourceNameEnum.DATAFILE, queryParameters)
      .pipe(
        tap((collection: CollectionTyped<DataFile>) => {
          if (collection._data.length === 1) {
            ctx.patchState({
              dataFileLogo: collection._data[0],
            });

            ctx.dispatch(new DepositAction.GetPhoto(depositId));
          }
        }),
        catchError(error => {
          throw new SolidifyStateError(this, error);
        }),
      );
  }

  @OverrideDefaultAction()
  @Action(DepositAction.GetPhoto)
  getPhoto(ctx: StateContext<DepositStateModel>, action: DepositAction.GetPhoto): Observable<any> {
    ctx.patchState({
      isLoadingLogo: true,
    });
    const depositId = ctx.getState().current.resId;
    const datafileId = ctx.getState().dataFileLogo.resId;

    let isReady = false;

    return PollingHelper.startPollingObs({
      initialIntervalRefreshInSecond: 1,
      incrementInterval: true,
      maximumIntervalRefreshInSecond: 5,
      continueUntil: () => !isReady,
      actionToDo: () => {
        const url = `${this._urlResource}/${depositId}/${ApiResourceNameEnum.DATAFILE}/${datafileId}`;
        this.apiService.getByIdInPath(url)
          .pipe(
            tap((result: DataFile) => {
              if (result.status === Enums.DataFile.StatusEnum.READY) {
                isReady = true;
                this._getPhotoReady(depositId, datafileId, ctx, action);
              }
              if (result.status === Enums.DataFile.StatusEnum.IN_ERROR) {
                isReady = true;
                ctx.dispatch(new DepositAction.GetPhotoFail(action));
              }
              if (result.status === Enums.DataFile.StatusEnum.CLEANED) {
                isReady = true;
                ctx.dispatch(new DepositAction.GetPhotoFail(action));
              }
            }),
            catchError(e => {
              isReady = true;
              ctx.dispatch(new DepositAction.GetPhotoFail(action));
              throw new SolidifyStateError(this, e);
            }),
          ).subscribe();
      },
    });
  }

  private _getPhotoReady(depositId: string, datafileId: string, ctx: StateContext<DepositStateModel>, action: DepositAction.GetPhoto): void {
    const url = `${this._urlResource}/${depositId}/${ApiResourceNameEnum.DATAFILE}/${datafileId}/${ApiActionEnum.DL}`;
    let headers = new HttpHeaders();
    headers = headers.set("Content-Disposition", "attachment; filename=logo");

    this._httpClient.get(url, {
      headers,
      responseType: "blob",
    }).pipe(
      tap((blobContent: Blob) => {
        ctx.dispatch(new DepositAction.GetPhotoSuccess(action, blobContent));
      }),
      catchError(e => {
        ctx.dispatch(new DepositAction.GetPhotoFail(action));
        throw new SolidifyStateError(this, e);
      }),
    ).subscribe();
  }

  @OverrideDefaultAction()
  @Action(DepositAction.DeletePhoto)
  deletePhoto(ctx: StateContext<DepositStateModel>, action: DepositAction.DeletePhoto): Observable<any> {
    const dataFileLogo = ctx.getState().dataFileLogo;
    if (isNullOrUndefined(dataFileLogo)) {
      return undefined;
    }
    const depositId = ctx.getState().current.resId;

    ctx.patchState({
      isLoadingLogo: true,
    });

    return ctx.dispatch(new DepositDataFileAction.Delete(depositId, dataFileLogo.resId)).pipe(
      tap(() => {
        ctx.dispatch(new DepositAction.DeletePhotoSuccess(action));
      }),
      catchError(e => {
        ctx.dispatch(new DepositAction.DeletePhotoFail(action));
        throw new SolidifyStateError(this, e);
      }),
    );
  }

  @OverrideDefaultAction()
  @Action(DepositAction.DeletePhotoSuccess)
  deletePhotoSuccess(ctx: StateContext<DepositStateModel>, action: DepositAction.DeletePhotoSuccess): void {
    ctx.patchState({
      isLoadingLogo: false,
      dataFileLogo: undefined,
      logo: undefined,
    });
  }
}
