import {HttpClient} from "@angular/common/http";
import {
  Inject,
  Injectable,
} from "@angular/core";
import {WINDOW} from "@app/app.module";
import {DepositDataFile} from "@app/features/deposit/models/deposit-data-file.model";
import {
  DepositDataFileAction,
  depositDataFileActionNameSpace,
} from "@app/features/deposit/stores/data-file/deposit-data-file.action";
import {ApiActionEnum} from "@app/shared/enums/api-action.enum";
import {ApiResourceNameEnum} from "@app/shared/enums/api-resource-name.enum";
import {PreIngestResourceApiEnum} from "@app/shared/enums/api.enum";
import {LocalStateEnum} from "@app/shared/enums/local-state.enum";
import {DepositDataFileHelper} from "@deposit/helpers/deposit-data-file.helper";
import {DepositDataFileMetadataTypeState} from "@deposit/stores/data-file/metadata-type/deposit-data-file-metadata-type.state";
import {
  DepositDataFileStatusHistoryState,
  DepositDataFileStatusHistoryStateModel,
} from "@deposit/stores/data-file/status-history/deposit-data-file-status-history.state";
import {DepositAction} from "@deposit/stores/deposit.action";
import {
  DepositState,
  DepositStateModel,
} from "@deposit/stores/deposit.state";
import {Enums} from "@enums";
import {environment} from "@environments/environment";
import {
  Action,
  Actions,
  ofActionCompleted,
  State,
  StateContext,
  Store,
} from "@ngxs/store";
import {LabelTranslateEnum} from "@shared/enums/label-translate.enum";
import {Result} from "@shared/models/business/result.model";
import {DownloadService} from "@shared/services/download.service";
import {defaultStatusHistoryInitValue} from "@shared/stores/status-history/status-history.state";
import {Observable} from "rxjs";
import {
  catchError,
  tap,
} from "rxjs/operators";
import {
  ApiService,
  CollectionTyped,
  CompositionState,
  CompositionStateModel,
  defaultCompositionStateInitValue,
  defaultResourceStateInitValue,
  isNullOrUndefined,
  MappingObjectUtil,
  MARK_AS_TRANSLATABLE,
  MemoizedUtil,
  NotificationService,
  OverrideDefaultAction,
  QueryParameters,
  QueryParametersUtil,
  SolidifyStateError,
  StoreUtil,
} from "solidify-frontend";

export const defaultDepositDataFileValue: () => DepositDataFileStateModel = () =>
  ({
    ...defaultCompositionStateInitValue(),
    deposit_dataFile_statusHistory: defaultStatusHistoryInitValue(),
    deposit_dataFile_metadataType: {
      ...defaultResourceStateInitValue(),
    },
    currentFolder: "" + DepositDataFileHelper.ROOT,
    folders: [],
    foldersWithIntermediateFolders: [],
    intermediateFolders: [],
    numberFilesNotReady: undefined,
    numberFilesInError: 0,
    numberFilesExcluded: 0,
    numberFilesIgnored: 0,
    numberFiles: undefined,
  });

export interface DepositDataFileStateModel extends CompositionStateModel<DepositDataFile> {
  deposit_dataFile_statusHistory: DepositDataFileStatusHistoryStateModel;
  currentFolder: string;
  folders: string[];
  foldersWithIntermediateFolders: string[];
  intermediateFolders: string[];
  numberFilesNotReady: number | undefined;
  numberFilesInError: number | undefined;
  numberFilesExcluded: number | undefined;
  numberFilesIgnored: number | undefined;
  numberFiles: number | undefined;
}

@Injectable()
@State<DepositDataFileStateModel>({
  name: LocalStateEnum.deposit_dataFile,
  defaults: {
    ...defaultDepositDataFileValue(),
  },
  children: [
    DepositDataFileStatusHistoryState,
    DepositDataFileMetadataTypeState,
  ],
})
export class DepositDataFileState extends CompositionState<DepositDataFileStateModel, DepositDataFile> {
  private readonly KEY_STATUS: keyof DepositDataFile = "status";
  private readonly KEY_RELATIVE_LOCATION: keyof DepositDataFile = "relativeLocation";

  constructor(protected apiService: ApiService,
              protected store: Store,
              protected notificationService: NotificationService,
              protected actions$: Actions,
              @Inject(WINDOW) private _window: Window,
              protected httpClient: HttpClient,
              private downloadService: DownloadService) {
    super(apiService, store, notificationService, actions$, {
      nameSpace: depositDataFileActionNameSpace,
      resourceName: ApiResourceNameEnum.DATAFILE,
    });
  }

  protected get _urlResource(): string {
    return PreIngestResourceApiEnum.deposits;
  }

  @Action(DepositDataFileAction.Resume)
  resume(ctx: StateContext<DepositDataFileStateModel>, action: DepositDataFileAction.Resume): Observable<Result> {
    return this.apiService.post<any, Result>(`${this._urlResource}/${action.parentId}/${this._resourceName}/${action.dataFile.resId}/${ApiActionEnum.RESUME}`)
      .pipe(
        tap(result => {
          if (result?.status === Enums.Result.ActionStatusEnum.EXECUTED) {
            this.notificationService.showInformation(LabelTranslateEnum.resourceResumed);
            ctx.dispatch(new DepositDataFileAction.GetAll(action.parentId, undefined, true));
          } else {
            this.notificationService.showError(LabelTranslateEnum.unableResumedResource);
          }
        }),
        catchError(error => {
          this.notificationService.showError(LabelTranslateEnum.unableResumedResource);
          throw new SolidifyStateError(this, error);
        }),
      );
  }

  @OverrideDefaultAction()
  @Action(DepositDataFileAction.GetAll)
  getAll<U>(ctx: StateContext<DepositDataFileStateModel>, action: DepositDataFileAction.GetAll): Observable<CollectionTyped<U>> {
    let queryParameters = action.queryParameters;
    if (isNullOrUndefined(queryParameters)) {
      queryParameters = QueryParametersUtil.clone(ctx.getState().queryParameters);
    }
    const currentFolder = ctx.getState().currentFolder;
    if (isNullOrUndefined(currentFolder)) {
      MappingObjectUtil.delete(QueryParametersUtil.getSearchItems(queryParameters), this.KEY_RELATIVE_LOCATION);

    } else {
      MappingObjectUtil.set(QueryParametersUtil.getSearchItems(queryParameters), this.KEY_RELATIVE_LOCATION, ctx.getState().currentFolder);
    }
    ctx.patchState({
      queryParameters: queryParameters,
    });
    return super.getAll<U>(ctx, action);
  }

  @Action(DepositDataFileAction.Refresh)
  refresh(ctx: StateContext<DepositDataFileStateModel>, action: DepositDataFileAction.Refresh): Observable<any> {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
    });

    return StoreUtil.dispatchParallelActionAndWaitForSubActionsCompletion(ctx, [
      {
        action: new DepositDataFileAction.GetAll(action.parentId, undefined, true),
        subActionCompletions: [
          this.actions$.pipe(ofActionCompleted(DepositDataFileAction.GetAllSuccess)),
          this.actions$.pipe(ofActionCompleted(DepositDataFileAction.GetAllFail)),
        ],
      },
      {
        action: new DepositDataFileAction.GetListFolder(action.parentId),
        subActionCompletions: [
          this.actions$.pipe(ofActionCompleted(DepositDataFileAction.GetListFolderSuccess)),
          this.actions$.pipe(ofActionCompleted(DepositDataFileAction.GetListFolderFail)),
        ],
      },
      {
        action: new DepositDataFileAction.GetNumberFilesNotReady(action.parentId),
        subActionCompletions: [
          this.actions$.pipe(ofActionCompleted(DepositDataFileAction.GetNumberFilesNotReadySuccess)),
          this.actions$.pipe(ofActionCompleted(DepositDataFileAction.GetNumberFilesNotReadyFail)),
        ],
      },
      {
        action: new DepositDataFileAction.GetNumberFiles(action.parentId),
        subActionCompletions: [
          this.actions$.pipe(ofActionCompleted(DepositDataFileAction.GetNumberFilesSuccess)),
          this.actions$.pipe(ofActionCompleted(DepositDataFileAction.GetNumberFilesFail)),
        ],
      },
    ]).pipe(
      tap(success => {
        if (success) {
          ctx.dispatch(new DepositDataFileAction.RefreshSuccess(action));
        } else {
          ctx.dispatch(new DepositDataFileAction.RefreshFail(action));
        }
      }),
    );
  }

  @Action(DepositDataFileAction.RefreshSuccess)
  refreshSuccess(ctx: StateContext<DepositDataFileStateModel>, action: DepositDataFileAction.RefreshSuccess): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
    ctx.dispatch(new DepositAction.ComputeModeTab());
  }

  @Action(DepositDataFileAction.RefreshFail)
  refreshFail(ctx: StateContext<DepositDataFileStateModel>, action: DepositDataFileAction.RefreshFail): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
  }

  @Action(DepositDataFileAction.Download)
  download(ctx: StateContext<DepositDataFileStateModel>, action: DepositDataFileAction.Download): void {
    const url = `${this._urlResource}/${action.parentId}/${ApiResourceNameEnum.DATAFILE}/${action.dataFile.resId}/${ApiActionEnum.DL}`;
    this.downloadService.download(false, url, action.dataFile.fileName, action.dataFile.fileSize);
  }

  @Action(DepositDataFileAction.ChangeCurrentFolder)
  changeCurrentFolder(ctx: StateContext<DepositDataFileStateModel>, action: DepositDataFileAction.ChangeCurrentFolder): void {
    ctx.patchState({
      currentFolder: action.newCurrentFolder,
    });
    const depositId = MemoizedUtil.currentSnapshot(this.store, DepositState).resId;
    if (!isNullOrUndefined(action.newCurrentFolder)) {
      ctx.dispatch(new DepositDataFileAction.GetListFolder(depositId));
    }
    if (action.refreshNewFolderContent) {
      let queryParameters = QueryParametersUtil.clone(ctx.getState().queryParameters);
      queryParameters = QueryParametersUtil.resetToFirstPage(queryParameters);
      ctx.dispatch(new DepositDataFileAction.GetAll(depositId, queryParameters));
    }
  }

  @Action(DepositDataFileAction.ChangeCurrentCategory)
  ChangeCurrentCategory(ctx: StateContext<DepositDataFileStateModel>, action: DepositDataFileAction.ChangeCurrentCategory): void {
    const depositId = MemoizedUtil.currentSnapshot(this.store, DepositState).resId;
    if (action.refreshNewCategory) {
      ctx.dispatch(new DepositDataFileAction.GetAll(depositId));
    }
  }

  @Action(DepositDataFileAction.GetListFolder)
  getListFolder(ctx: StateContext<DepositDataFileStateModel>, action: DepositDataFileAction.GetListFolder): Observable<string[]> {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
    });

    return this.httpClient.get<string[]>(`${this._urlResource}/${action.parentId}/${this._resourceName}/${ApiActionEnum.LIST_FOLDERS}`)
      .pipe(
        tap((listFolder: string[]) => {
          ctx.dispatch(new DepositDataFileAction.GetListFolderSuccess(action, listFolder));
        }),
        catchError(error => {
          ctx.dispatch(new DepositDataFileAction.GetListFolderFail(action));
          throw new SolidifyStateError(this, error);
        }),
      );
  }

  @Action(DepositDataFileAction.GetListFolderSuccess)
  getListFolderSuccess(ctx: StateContext<DepositDataFileStateModel>, action: DepositDataFileAction.GetListFolderSuccess): void {
    const folders = [...action.folder];
    const foldersWithIntermediateFolders = action.folder;
    const intermediateFolders = DepositDataFileHelper.createIntermediateFolders(foldersWithIntermediateFolders);
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
      folders,
      foldersWithIntermediateFolders,
      intermediateFolders,
    });
  }

  @Action(DepositDataFileAction.GetListFolderFail)
  getListFolderFail(ctx: StateContext<DepositDataFileStateModel>, action: DepositDataFileAction.GetListFolderFail): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
  }

  @Action(DepositDataFileAction.GetNumberFilesNotReady)
  getNumberFilesNotReady(ctx: StateContext<DepositDataFileStateModel>, action: DepositDataFileAction.GetNumberFilesNotReady): Observable<CollectionTyped<DepositDataFile>> {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
    });

    const queryParameters = new QueryParameters(environment.minimalPageSizeToRetrievePaginationInfo);
    const searchItems = queryParameters.search.searchItems;
    MappingObjectUtil.set(searchItems, "search", `i-status!READY`);
    MappingObjectUtil.set(searchItems, "match", "any");

    return this.apiService.get<DepositDataFile>(`${this._urlResource}/${action.parentId}/${this._resourceName}/${ApiActionEnum.SEARCH}`, queryParameters)
      .pipe(
        tap(collection => {
          ctx.dispatch(new DepositDataFileAction.GetNumberFilesNotReadySuccess(action, collection._page.totalItems));
        }),
        catchError(error => {
          ctx.dispatch(new DepositDataFileAction.GetNumberFilesNotReadyFail(action));
          throw new SolidifyStateError(this, error);
        }),
      );
  }

  @Action(DepositDataFileAction.GetNumberFilesNotReadySuccess)
  getNumberFilesNotReadySuccess(ctx: StateContext<DepositDataFileStateModel>, action: DepositDataFileAction.GetNumberFilesNotReadySuccess): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
      numberFilesNotReady: action.numberFilesNotReady,
    });
  }

  @Action(DepositDataFileAction.GetNumberFilesNotReadyFail)
  getNumberFilesNotReadyFail(ctx: StateContext<DepositDataFileStateModel>, action: DepositDataFileAction.GetNumberFilesNotReadyFail): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
  }

  @Action(DepositDataFileAction.GetNumberFilesInError)
  getNumberFilesInError(ctx: StateContext<DepositDataFileStateModel>, action: DepositDataFileAction.GetNumberFilesInError): Observable<CollectionTyped<DepositDataFile>> {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
    });

    const queryParameters = new QueryParameters(environment.minimalPageSizeToRetrievePaginationInfo);
    MappingObjectUtil.set(queryParameters.search.searchItems, this.KEY_STATUS, Enums.DataFile.StatusEnum.IN_ERROR);
    return this.apiService.get<DepositDataFile>(`${this._urlResource}/${action.parentId}/${this._resourceName}`, queryParameters)
      .pipe(
        tap(collection => {
          ctx.dispatch(new DepositDataFileAction.GetNumberFilesInErrorSuccess(action, collection._page.totalItems));
        }),
        catchError(error => {
          ctx.dispatch(new DepositDataFileAction.GetNumberFilesInErrorFail(action));
          throw new SolidifyStateError(this, error);
        }),
      );
  }

  @Action(DepositDataFileAction.GetNumberFilesInErrorSuccess)
  getNumberFilesInErrorSuccess(ctx: StateContext<DepositDataFileStateModel>, action: DepositDataFileAction.GetNumberFilesInErrorSuccess): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
      numberFilesInError: action.numberFilesInErrors,
    });
  }

  @Action(DepositDataFileAction.GetNumberFilesInErrorFail)
  getNumberFilesInErrorFail(ctx: StateContext<DepositDataFileStateModel>, action: DepositDataFileAction.GetNumberFilesInErrorFail): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
  }

  @Action(DepositDataFileAction.GetNumberFiles)
  getNumberFiles(ctx: StateContext<DepositDataFileStateModel>, action: DepositDataFileAction.GetNumberFiles): Observable<CollectionTyped<DepositDataFile>> {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
    });

    const queryParameters = new QueryParameters(environment.minimalPageSizeToRetrievePaginationInfo);
    return this.apiService.get<DepositDataFile>(`${this._urlResource}/${action.parentId}/${this._resourceName}`, queryParameters)
      .pipe(
        tap(collection => {
          ctx.dispatch(new DepositDataFileAction.GetNumberFilesSuccess(action, collection._page.totalItems));
        }),
        catchError(error => {
          ctx.dispatch(new DepositDataFileAction.GetNumberFilesFail(action));
          throw new SolidifyStateError(this, error);
        }),
      );
  }

  @Action(DepositDataFileAction.GetNumberFilesSuccess)
  getNumberFilesSuccess(ctx: StateContext<DepositDataFileStateModel>, action: DepositDataFileAction.GetNumberFilesSuccess): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
      numberFiles: action.numberFiles,
    });
  }

  @Action(DepositDataFileAction.GetNumberFilesFail)
  getNumberFilesFail(ctx: StateContext<DepositDataFileStateModel>, action: DepositDataFileAction.GetNumberFilesFail): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
  }

  @Action(DepositDataFileAction.GetNumberFilesExcluded)
  getNumberFilesExcluded(ctx: StateContext<DepositDataFileStateModel>, action: DepositDataFileAction.GetNumberFilesExcluded): Observable<CollectionTyped<DepositDataFile>> {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
    });

    const queryParameters = new QueryParameters(environment.minimalPageSizeToRetrievePaginationInfo);
    MappingObjectUtil.set(queryParameters.search.searchItems, this.KEY_STATUS, Enums.DataFile.StatusEnum.EXCLUDED_FILE);
    return this.apiService.get<DepositDataFile>(`${this._urlResource}/${action.parentId}/${this._resourceName}`, queryParameters)
      .pipe(
        tap(collection => {
          ctx.dispatch(new DepositDataFileAction.GetNumberFilesExcludedSuccess(action, collection._page.totalItems));
        }),
        catchError(error => {
          ctx.dispatch(new DepositDataFileAction.GetNumberFilesExcludedFail(action));
          throw new SolidifyStateError(this, error);
        }),
      );
  }

  @Action(DepositDataFileAction.GetNumberFilesExcludedSuccess)
  getNumberFilesExcludedSuccess(ctx: StateContext<DepositDataFileStateModel>, action: DepositDataFileAction.GetNumberFilesExcludedSuccess): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
      numberFilesExcluded: action.numberFilesExcluded,
    });
  }

  @Action(DepositDataFileAction.GetNumberFilesExcludedFail)
  getNumberFilesExcludedFail(ctx: StateContext<DepositDataFileStateModel>, action: DepositDataFileAction.GetNumberFilesExcludedFail): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
  }

  @Action(DepositDataFileAction.GetNumberFilesIgnored)
  getNumberFilesIgnored(ctx: StateContext<DepositDataFileStateModel>, action: DepositDataFileAction.GetNumberFilesIgnored): Observable<CollectionTyped<DepositDataFile>> {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
    });

    const queryParameters = new QueryParameters(environment.minimalPageSizeToRetrievePaginationInfo);
    MappingObjectUtil.set(queryParameters.search.searchItems, this.KEY_STATUS, Enums.DataFile.StatusEnum.IGNORED_FILE);
    return this.apiService.get<DepositDataFile>(`${this._urlResource}/${action.parentId}/${this._resourceName}`, queryParameters)
      .pipe(
        tap(collection => {
          ctx.dispatch(new DepositDataFileAction.GetNumberFilesIgnoredSuccess(action, collection._page.totalItems));
        }),
        catchError(error => {
          ctx.dispatch(new DepositDataFileAction.GetNumberFilesIgnoredFail(action));
          throw new SolidifyStateError(this, error);
        }),
      );
  }

  @Action(DepositDataFileAction.GetNumberFilesIgnoredSuccess)
  getNumberFilesIgnoredSuccess(ctx: StateContext<DepositDataFileStateModel>, action: DepositDataFileAction.GetNumberFilesIgnoredSuccess): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
      numberFilesIgnored: action.numberFilesIgnored,
    });
  }

  @Action(DepositDataFileAction.GetNumberFilesIgnoredFail)
  getNumberFilesIgnoredFail(ctx: StateContext<DepositDataFileStateModel>, action: DepositDataFileAction.GetNumberFilesIgnoredFail): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
  }

  @OverrideDefaultAction()
  @Action(DepositDataFileAction.DeleteSuccess)
  deleteSuccess(ctx: StateContext<DepositDataFileStateModel>, action: DepositDataFileAction.DeleteSuccess): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
    ctx.dispatch(new DepositDataFileAction.Refresh(action.parentId));
  }

  @Action(DepositDataFileAction.Validate)
  validate(ctx: StateContext<DepositDataFileStateModel>, action: DepositDataFileAction.Validate): Observable<Result> {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
    });

    return this.httpClient.post<Result>(`${this._urlResource}/${action.parentId}/${this._resourceName}/${action.dataFile.resId}/${ApiActionEnum.VALIDATE}`, null)
      .pipe(
        tap(result => {
          if (result?.status === Enums.Result.ActionStatusEnum.EXECUTED) {
            ctx.dispatch(new DepositDataFileAction.ValidateSuccess(action));
          } else {
            ctx.dispatch(new DepositDataFileAction.ValidateFail(action, result));
          }
        }),
        catchError(error => {
          ctx.dispatch(new DepositDataFileAction.ValidateFail(action));
          throw new SolidifyStateError(this, error);
        }),
      );
  }

  @Action(DepositDataFileAction.ValidateSuccess)
  validateSuccess(ctx: StateContext<DepositDataFileStateModel>, action: DepositDataFileAction.ValidateSuccess): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
    this.notificationService.showSuccess(MARK_AS_TRANSLATABLE("notification.deposit.file.validate.success"));
    ctx.dispatch(new DepositDataFileAction.Refresh((action.parentAction as DepositDataFileAction.Validate).parentId));
  }

  @Action(DepositDataFileAction.ValidateFail)
  validateFail(ctx: StateContext<DepositDataFileStateModel>, action: DepositDataFileAction.ValidateFail): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
    this.notificationService.showError(MARK_AS_TRANSLATABLE("notification.deposit.file.validate.fail"));
  }

  @Action(DepositDataFileAction.Move)
  move(ctx: StateContext<DepositDataFileStateModel>, action: DepositDataFileAction.Move): Observable<DepositDataFile> {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
    });

    return this.httpClient.patch<DepositDataFile>(`${this._urlResource}/${action.parentId}/${this._resourceName}/${action.dataFile.resId}`, {
      relativeLocation: action.dataFile.relativeLocation,
    } as DepositDataFile)
      .pipe(
        tap(result => {
          ctx.dispatch(new DepositDataFileAction.MoveSuccess(action, result));
        }),
        catchError(error => {
          ctx.dispatch(new DepositDataFileAction.MoveFail(action));
          throw new SolidifyStateError(this, error);
        }),
      );
  }

  @Action(DepositDataFileAction.MoveSuccess)
  moveSuccess(ctx: StateContext<DepositDataFileStateModel>, action: DepositDataFileAction.MoveSuccess): void {
    const relativeLocation = (action.parentAction as DepositDataFileAction.Move).dataFile.relativeLocation;
    ctx.dispatch(new DepositDataFileAction.ChangeCurrentFolder(relativeLocation, true));
    this.notificationService.showSuccess(MARK_AS_TRANSLATABLE("notification.deposit.file.move.success"));
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
  }

  @Action(DepositDataFileAction.MoveFail)
  moveFail(ctx: StateContext<DepositDataFileStateModel>, action: DepositDataFileAction.MoveFail): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
    this.notificationService.showError(MARK_AS_TRANSLATABLE("notification.deposit.file.move.fail"));
  }

  @Action(DepositDataFileAction.ChangeDataCategory)
  changeDataCategory(ctx: StateContext<DepositDataFileStateModel>, action: DepositDataFileAction.ChangeDataCategory): Observable<DepositDataFile> {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
    });

    return this.httpClient.patch<DepositDataFile>(`${this._urlResource}/${action.parentId}/${this._resourceName}/${action.dataFile.resId}`, {
      dataCategory: action.dataFile.dataCategory,
      dataType: action.dataFile.dataType,
      metadataType: !isNullOrUndefined(action.dataFile.metadataType) ? action.dataFile.metadataType : null,
    } as DepositDataFile)
      .pipe(
        tap(result => {
          ctx.dispatch(new DepositDataFileAction.ChangeDataCategorySuccess(action, result));
        }),
        catchError(error => {
          ctx.dispatch(new DepositDataFileAction.ChangeDataCategoryFail(action));
          throw new SolidifyStateError(this, error);
        }),
      );
  }

  @Action(DepositDataFileAction.ChangeDataCategorySuccess)
  changeDataCategorySuccess(ctx: StateContext<DepositDataFileStateModel>, action: DepositDataFileAction.ChangeDataCategorySuccess): void {
    const dataCategory = (action.parentAction as DepositDataFileAction.ChangeDataCategory).dataFile.dataCategory;
    const dataType = (action.parentAction as DepositDataFileAction.ChangeDataCategory).dataFile.dataType;
    const depositId = MemoizedUtil.currentSnapshot(this.store, DepositState).resId;

    ctx.dispatch(new DepositDataFileAction.ChangeCurrentCategory(dataCategory, dataType, true));
    ctx.dispatch(new DepositDataFileAction.Refresh(depositId));
    this.notificationService.showSuccess(MARK_AS_TRANSLATABLE("notification.deposit.file.changeDataCategory.success"));
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
  }

  @Action(DepositDataFileAction.ChangeDataCategoryFail)
  changeDataCategoryFail(ctx: StateContext<DepositDataFileStateModel>, action: DepositDataFileAction.ChangeDataCategoryFail): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
    this.notificationService.showError(MARK_AS_TRANSLATABLE("notification.deposit.file.changeDataCategory.fail"));
  }

  @Action(DepositDataFileAction.DeleteAll)
  deleteAll(ctx: StateContext<DepositDataFileStateModel>, action: DepositDataFileAction.DeleteAll): Observable<string[]> {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
    });

    return this.apiService.delete<string[]>(`${this._urlResource}/${action.parentId}/${this._resourceName}`, action.listResId)
      .pipe(
        tap(collection => {
          ctx.dispatch(new DepositDataFileAction.DeleteAllSuccess(action));
        }),
        catchError(error => {
          ctx.dispatch(new DepositDataFileAction.DeleteAllFail(action));
          throw new SolidifyStateError(this, error);
        }),
      );
  }

  @Action(DepositDataFileAction.DeleteAllSuccess)
  deleteAllSuccess(ctx: StateContext<DepositDataFileStateModel>, action: DepositDataFileAction.DeleteAllSuccess): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
      queryParameters: QueryParametersUtil.resetToFirstPage(ctx.getState().queryParameters),
    });
    ctx.dispatch(new DepositDataFileAction.Refresh((action.parentAction as DepositDataFileAction.DeleteAll).parentId));
  }

  @Action(DepositDataFileAction.DeleteAllFail)
  deleteAllFail(ctx: StateContext<DepositDataFileStateModel>, action: DepositDataFileAction.DeleteAllFail): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
  }

  @Action(DepositDataFileAction.ResumeAll)
  resumeAll(ctx: StateContext<DepositDataFileStateModel>, action: DepositDataFileAction.ResumeAll): Observable<string[]> {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
    });

    return this.apiService.post<string[]>(`${this._urlResource}/${action.parentId}/${this._resourceName}/${ApiActionEnum.RESUME}`, action.listResId)
      .pipe(
        tap(collection => {
          ctx.dispatch(new DepositDataFileAction.ResumeAllSuccess(action));
        }),
        catchError(error => {
          ctx.dispatch(new DepositDataFileAction.ResumeAllFail(action));
          throw new SolidifyStateError(this, error);
        }),
      );
  }

  @Action(DepositDataFileAction.ResumeAllSuccess)
  resumeAllSuccess(ctx: StateContext<DepositDataFileStateModel>, action: DepositDataFileAction.ResumeAllSuccess): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
      queryParameters: QueryParametersUtil.resetToFirstPage(ctx.getState().queryParameters),
    });
    ctx.dispatch(new DepositDataFileAction.Refresh((action.parentAction as DepositDataFileAction.ResumeAll).parentId));
  }

  @Action(DepositDataFileAction.ResumeAllFail)
  resumeAllFail(ctx: StateContext<DepositDataFileStateModel>, action: DepositDataFileAction.ResumeAllFail): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
  }

  @Action(DepositDataFileAction.ValidateAll)
  validateAll(ctx: StateContext<DepositDataFileStateModel>, action: DepositDataFileAction.ValidateAll): Observable<string[]> {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
    });

    return this.apiService.post<string[]>(`${this._urlResource}/${action.parentId}/${this._resourceName}/${ApiActionEnum.VALIDATE}`, action.listResId)
      .pipe(
        tap(collection => {
          ctx.dispatch(new DepositDataFileAction.ValidateAllSuccess(action));
        }),
        catchError(error => {
          ctx.dispatch(new DepositDataFileAction.ValidateAllFail(action));
          throw new SolidifyStateError(this, error);
        }),
      );
  }

  @Action(DepositDataFileAction.ValidateAllSuccess)
  validateAllSuccess(ctx: StateContext<DepositDataFileStateModel>, action: DepositDataFileAction.ValidateAllSuccess): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
      queryParameters: QueryParametersUtil.resetToFirstPage(ctx.getState().queryParameters),
    });
    ctx.dispatch(new DepositDataFileAction.Refresh((action.parentAction as DepositDataFileAction.ValidateAll).parentId));
  }

  @Action(DepositDataFileAction.ValidateAllFail)
  validateAllFail(ctx: StateContext<DepositDataFileStateModel>, action: DepositDataFileAction.ValidateAllFail): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
  }

  @Action(DepositDataFileAction.DeleteFolder)
  deleteFolder(ctx: StateContext<DepositStateModel>, action: DepositDataFileAction.DeleteFolder): Observable<string> {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
    });

    return this.apiService.post<string>(`${this._urlResource}/${action.parentId}/${this._resourceName}/${ApiActionEnum.DELETE_FOLDER}?${this.KEY_RELATIVE_LOCATION}=${action.fullFolderName}`)
      .pipe(
        tap(result => {
          ctx.dispatch(new DepositDataFileAction.DeleteFolderSuccess(action));
        }),
        catchError(error => {
          ctx.dispatch(new DepositDataFileAction.DeleteFolderFail(action));
          throw new SolidifyStateError(this, error);
        }),
      );
  }

  @Action(DepositDataFileAction.DeleteFolderSuccess)
  downloadFolderSuccess(ctx: StateContext<DepositDataFileStateModel>, action: DepositDataFileAction.DeleteFolderSuccess): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
      currentFolder: "" + DepositDataFileHelper.ROOT,
    });
    this.notificationService.showSuccess(MARK_AS_TRANSLATABLE("notification.deposit.folder.deleteWithSuccess"));
    ctx.dispatch(new DepositDataFileAction.Refresh((action.parentAction as DepositDataFileAction.DeleteFolder).parentId));
  }

  @Action(DepositDataFileAction.DeleteFolderFail)
  downloadFolderFail(ctx: StateContext<DepositDataFileStateModel>, action: DepositDataFileAction.DeleteFolderFail): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
    this.notificationService.showError(MARK_AS_TRANSLATABLE("notification.deposit.folder.deleteFail"));
  }
}
