import {Injectable} from "@angular/core";
import {
  DepositCollectionAction,
  depositCollectionActionNameSpace,
} from "@deposit/stores/collection/deposit-collection.action";
import {DepositCollectionStatusHistoryState} from "@deposit/stores/collection/status-history/deposit-collection-status-history.state";
import {DepositDataFileStateModel} from "@deposit/stores/data-file/deposit-data-file.state";
import {DepositAction} from "@deposit/stores/deposit.action";
import {environment} from "@environments/environment";
import {Aip} from "@models";
import {Navigate} from "@ngxs/router-plugin";
import {
  Action,
  Actions,
  ofActionCompleted,
  State,
  StateContext,
  Store,
} from "@ngxs/store";
import {ApiResourceNameEnum} from "@shared/enums/api-resource-name.enum";
import {PreIngestResourceApiEnum} from "@shared/enums/api.enum";
import {LocalStateEnum} from "@shared/enums/local-state.enum";
import {
  AppRoutesEnum,
  RoutesEnum,
  SharedAipRoutesEnum,
  urlSeparator,
} from "@shared/enums/routes.enum";
import {defaultStatusHistoryInitValue} from "@shared/stores/status-history/status-history.state";
import {Observable} from "rxjs";
import {
  catchError,
  tap,
} from "rxjs/operators";
import {
  ApiService,
  AssociationRemoteState,
  AssociationRemoteStateModel,
  CollectionTyped,
  defaultAssociationRemoteStateInitValue,
  NotificationService,
  OverrideDefaultAction,
  QueryParameters,
  SolidifyStateError,
  StoreUtil,
} from "solidify-frontend";

export const defaultDepositCollectionValue: () => DepositCollectionStateModel = () =>
  ({
    ...defaultAssociationRemoteStateInitValue(),
    deposit_collection_statusHistory: defaultStatusHistoryInitValue(),
    numberCollections: undefined,
  });

export interface DepositCollectionStateModel extends AssociationRemoteStateModel<Aip> {
  numberCollections: number | undefined;
}

@Injectable()
@State<DepositCollectionStateModel>({
  name: LocalStateEnum.deposit_collection,
  defaults: {
    ...defaultDepositCollectionValue(),
  },
  children: [
    DepositCollectionStatusHistoryState,
  ],
})
export class DepositCollectionState extends AssociationRemoteState<DepositCollectionStateModel, Aip> {
  constructor(protected apiService: ApiService,
              protected store: Store,
              protected notificationService: NotificationService,
              protected actions$: Actions) {
    super(apiService, store, notificationService, actions$, {
      nameSpace: depositCollectionActionNameSpace,
      resourceName: ApiResourceNameEnum.AIP,
    });
  }

  protected get _urlResource(): string {
    return PreIngestResourceApiEnum.deposits;
  }

  @Action(DepositCollectionAction.ChangeQueryParameters)
  changeQueryParameters(ctx: StateContext<DepositCollectionStateModel>, action: DepositCollectionAction.ChangeQueryParameters): void {
    ctx.patchState({
      queryParameters: action.queryParameters,
    });
    ctx.dispatch(new DepositCollectionAction.GetAll(action.parentId, undefined, action.keepCurrentContext));
  }

  @Action(DepositCollectionAction.Refresh)
  refresh(ctx: StateContext<DepositCollectionStateModel>, action: DepositCollectionAction.Refresh): Observable<any> {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
    });

    return StoreUtil.dispatchParallelActionAndWaitForSubActionsCompletion(ctx, [
      {
        action: new DepositCollectionAction.GetAll(action.parentId, undefined, true),
        subActionCompletions: [
          this.actions$.pipe(ofActionCompleted(DepositCollectionAction.GetAllSuccess)),
          this.actions$.pipe(ofActionCompleted(DepositCollectionAction.GetAllFail)),
        ],
      },
      {
        action: new DepositCollectionAction.GetNumberCollections(action.parentId),
        subActionCompletions: [
          this.actions$.pipe(ofActionCompleted(DepositCollectionAction.GetNumberCollectionsSuccess)),
          this.actions$.pipe(ofActionCompleted(DepositCollectionAction.GetNumberCollectionsFail)),
        ],
      },
    ]).pipe(
      tap(success => {
        if (success) {
          ctx.dispatch(new DepositCollectionAction.RefreshSuccess(action));
        } else {
          ctx.dispatch(new DepositCollectionAction.RefreshFail(action));
        }
      }),
    );
  }

  @Action(DepositCollectionAction.RefreshSuccess)
  refreshSuccess(ctx: StateContext<DepositCollectionStateModel>, action: DepositCollectionAction.RefreshSuccess): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
    ctx.dispatch(new DepositAction.ComputeModeTab());
  }

  @Action(DepositCollectionAction.RefreshFail)
  refreshFail(ctx: StateContext<DepositCollectionStateModel>, action: DepositCollectionAction.RefreshFail): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
  }

  @OverrideDefaultAction()
  @Action(DepositCollectionAction.DeleteSuccess)
  deleteSuccess(ctx: StateContext<DepositDataFileStateModel>, action: DepositCollectionAction.DeleteSuccess): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
    ctx.dispatch(new DepositCollectionAction.Refresh((action.parentAction as DepositCollectionAction.Delete).parentId));
  }

  @Action(DepositCollectionAction.GoToAip)
  goToAip(ctx: StateContext<DepositCollectionStateModel>, action: DepositCollectionAction.GoToAip): void {
    const pathAipDetail = RoutesEnum.preservationPlanningAip + urlSeparator + 1 + urlSeparator + SharedAipRoutesEnum.aipDetail + AppRoutesEnum.separator;
    const path = [pathAipDetail, action.aip.resId];
    ctx.dispatch(new Navigate(path));
  }

  @Action(DepositCollectionAction.AddAip)
  addAip(ctx: StateContext<DepositCollectionStateModel>, action: DepositCollectionAction.AddAip): void {
    ctx.dispatch(new DepositCollectionAction.Create(action.parentId, action.aipIds));
    ctx.dispatch(new DepositCollectionAction.Refresh(action.parentId));
  }

  @Action(DepositCollectionAction.GetNumberCollections)
  getNumberCollections(ctx: StateContext<DepositCollectionStateModel>, action: DepositCollectionAction.GetNumberCollections): Observable<CollectionTyped<Aip>> {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
    });

    const queryParameters = new QueryParameters(environment.minimalPageSizeToRetrievePaginationInfo);
    return this.apiService.get<Aip>(`${this._urlResource}/${action.parentId}/${this._resourceName}`, queryParameters)
      .pipe(
        tap(collection => {
          ctx.dispatch(new DepositCollectionAction.GetNumberCollectionsSuccess(action, collection._page.totalItems));
        }),
        catchError(error => {
          ctx.dispatch(new DepositCollectionAction.GetNumberCollectionsFail(action));
          throw new SolidifyStateError(this, error);
        }),
      );
  }

  @Action(DepositCollectionAction.GetNumberCollectionsSuccess)
  getNumberCollectionsSuccess(ctx: StateContext<DepositCollectionStateModel>, action: DepositCollectionAction.GetNumberCollectionsSuccess): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
      numberCollections: action.numberCollections,
    });
  }

  @Action(DepositCollectionAction.GetNumberCollectionsFail)
  getNumberCollectionsFail(ctx: StateContext<DepositCollectionStateModel>, action: DepositCollectionAction.GetNumberCollectionsFail): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
  }
}
