import {FileUploadStatusEnum} from "@app/features/deposit/enums/file-upload-status.enum";
import {DepositDataFile} from "@app/features/deposit/models/deposit-data-file.model";
import {FileUploadWrapper} from "@app/features/deposit/models/file-upload-wrapper.model";
import {UploadEventModel} from "@app/features/deposit/models/upload-event.model";
import {UploadFileStatus} from "@app/features/deposit/models/upload-file-status.model";
import {DepositStateModel} from "@app/features/deposit/stores/deposit.state";
import {Guid} from "@app/shared/models/guid.model";
import {StateContext} from "@ngxs/store";

export class DataFileUploadHelper {
  static addToUploadStatus(ctx: StateContext<DepositStateModel>, fileUploadWrapper: FileUploadWrapper, isArchive: boolean): UploadFileStatus {
    const uploadFileStatus = this.createNewUploadFileStatus(fileUploadWrapper, isArchive);
    ctx.patchState({
      uploadStatus: [
        ...ctx.getState().uploadStatus,
        uploadFileStatus,
      ],
    });
    return uploadFileStatus;
  }

  private static createNewUploadFileStatus(fileUploadWrapper: FileUploadWrapper, isArchive: boolean): UploadFileStatus {
    return {
      guid: Guid.MakeNew(),
      fileUploadWrapper: fileUploadWrapper,
      status: FileUploadStatusEnum.started,
      progressPercentage: 0,
      isArchive: isArchive,
    } as UploadFileStatus;
  }

  static updateInProgressUploadFileStatus(ctx: StateContext<DepositStateModel>, uploadFileStatus: UploadFileStatus, event: UploadEventModel): void {
    uploadFileStatus.status = FileUploadStatusEnum.inProgress;
    uploadFileStatus.progressPercentage = this.getUploadProgress(event);
    this.updateUploadFileStatus(ctx, uploadFileStatus);
  }

  static getUploadProgress(event: UploadEventModel): number {
    return Math.round(100 * event.loaded / event.total);
  }

  static updateCompletedUploadFileStatus(ctx: StateContext<DepositStateModel>, uploadFileStatus: UploadFileStatus, depositDataFile: DepositDataFile): void {
    uploadFileStatus.progressPercentage = 100;
    uploadFileStatus.status = FileUploadStatusEnum.completed;
    uploadFileStatus.result = depositDataFile;
    this.updateUploadFileStatus(ctx, uploadFileStatus);
    setTimeout(() => {
      this.removeToUploadStatus(ctx, uploadFileStatus);
    }, 5000);
  }

  static updateErrorUploadFileStatus(ctx: StateContext<DepositStateModel>, uploadFileStatus: UploadFileStatus, messageToTranslate: string): void {
    uploadFileStatus.status = FileUploadStatusEnum.inError;
    uploadFileStatus.errorMessageToTranslate = messageToTranslate;
    this.updateUploadFileStatus(ctx, uploadFileStatus);
  }

  static updateCancelUploadFileStatus(ctx: StateContext<DepositStateModel>, uploadFileStatus: UploadFileStatus): void {
    uploadFileStatus.status = FileUploadStatusEnum.canceled;
    this.updateUploadFileStatus(ctx, uploadFileStatus);
  }

  static updateUploadFileStatus(ctx: StateContext<DepositStateModel>, uploadFileStatus: UploadFileStatus): void {
    const uploadStatus = [...ctx.getState().uploadStatus];
    const index = this.getIndexCurrentUploadFileStatus(ctx, uploadFileStatus.guid);
    if (index !== -1 && index < uploadStatus.length) {
      uploadStatus[index] = uploadFileStatus;
      ctx.patchState({
        uploadStatus,
      });
    }
  }

  static removeToUploadStatus(ctx: StateContext<DepositStateModel>, uploadFileStatus: UploadFileStatus): void {
    const uploadStatus = [...ctx.getState().uploadStatus];
    const index = this.getIndexCurrentUploadFileStatus(ctx, uploadFileStatus.guid);
    if (index !== -1 && index < uploadStatus.length) {
      uploadStatus.splice(index, 1);
      ctx.patchState({
        uploadStatus,
      });
    }
  }

  static getCurrentUploadFileStatus(ctx: StateContext<DepositStateModel>, guid: Guid): UploadFileStatus | null {
    const index = this.getIndexCurrentUploadFileStatus(ctx, guid);
    if (index !== -1) {
      return Object.assign({}, ctx.getState().uploadStatus[index]);
    }
    return null;
  }

  private static getIndexCurrentUploadFileStatus(ctx: StateContext<DepositStateModel>, guid: Guid): number {
    const uploadStatus = ctx.getState().uploadStatus;
    return uploadStatus.findIndex(u => u.guid === guid);
  }
}
