import {
  Component,
  OnInit,
} from "@angular/core";
import {AppAuthorizedOrganizationalUnitAction} from "@app/stores/authorized-organizational-unit/app-authorized-organizational-unit.action";
import {AppAuthorizedOrganizationalUnitState} from "@app/stores/authorized-organizational-unit/app-authorized-organizational-unit.state";
import {DepositTabStatusEnum} from "@deposit/enums/deposit-tab-status.enum";
import {Navigate} from "@ngxs/router-plugin";
import {
  Actions,
  ofActionCompleted,
  Store,
} from "@ngxs/store";
import {SharedAbstractRoutable} from "@shared/components/routables/shared-abstract/shared-abstract.routable";
import {
  AppRoutesEnum,
  RoutesEnum,
} from "@shared/enums/routes.enum";
import {UserPreferencesUtil} from "@shared/utils/user-preferences.util";
import {tap} from "rxjs/operators";
import {
  isEmptyArray,
  isNullOrUndefined,
  MemoizedUtil,
  StoreUtil,
  NotificationService,
  MARK_AS_TRANSLATABLE
} from "solidify-frontend";


@Component({
  selector: "dlcm-deposit-root-routable",
  templateUrl: "./deposit-root.routable.html",
  styleUrls: ["./deposit-root.routable.scss"],
})
export class DepositRootRoutable extends SharedAbstractRoutable implements OnInit {
  noOrgUnit: boolean = false;

  constructor(private readonly _store: Store,
              private readonly _actions$: Actions,
              private notificationService: NotificationService) {
    super();
  }

  ngOnInit(): void {
    this.subscribe(StoreUtil.dispatchParallelActionAndWaitForSubActionsCompletion(this._store, [
      {
        action: new AppAuthorizedOrganizationalUnitAction.GetAll(undefined, false, false),
        subActionCompletions: [
          this._actions$.pipe(ofActionCompleted(AppAuthorizedOrganizationalUnitAction.GetAllSuccess)),
          this._actions$.pipe(ofActionCompleted(AppAuthorizedOrganizationalUnitAction.GetAllFail)),
        ],
      },
    ]).pipe(
      tap(success => {
        if (!success) {
          return;
        }

        const listAuthorizedOrgUnit = MemoizedUtil.listSnapshot(this._store, AppAuthorizedOrganizationalUnitState);
        if (isEmptyArray(listAuthorizedOrgUnit)) {
          this.noOrgUnit = true;
          this.notificationService.showInformation(MARK_AS_TRANSLATABLE("notification.deposit.noOrgUnit"));
          this._store.dispatch(new Navigate([AppRoutesEnum.preservationSpace]));
          return;
        }
        const preferredOrgUnit = UserPreferencesUtil.getPreferredOrgUnitInDepositMenu(listAuthorizedOrgUnit);
        if (isNullOrUndefined(preferredOrgUnit)) {
          return;
        }
        this._store.dispatch(new Navigate([RoutesEnum.deposit, preferredOrgUnit.resId, DepositTabStatusEnum.inProgress]));
      }),
    ));
  }
}

