import {
  AfterViewInit,
  ChangeDetectionStrategy,
  ChangeDetectorRef,
  Component,
  OnDestroy,
  OnInit,
  ViewChild,
} from "@angular/core";
import {MatDialog} from "@angular/material/dialog";
import {ActivatedRoute} from "@angular/router";
import {AppState} from "@app/stores/app.state";
import {AppPersonState} from "@app/stores/person/app-person.state";
import {AppSystemPropertyState} from "@app/stores/system-property/app-system-property.state";
import {DepositFormPresentational} from "@deposit/components/presentationals/deposit-form/deposit-form.presentational";
import {DepositService} from "@deposit/services/deposit.service";
import {depositActionNameSpace} from "@deposit/stores/deposit.action";
import {
  DepositState,
  DepositStateModel,
} from "@deposit/stores/deposit.state";
import {DepositOrganizationalUnitAdditionalFieldsFormAction} from "@deposit/stores/organizational-unit/additional-fields-form/deposit-organizational-unit-additional-fields-form.action";
import {DepositOrganizationalUnitAdditionalFieldsFormState} from "@deposit/stores/organizational-unit/additional-fields-form/deposit-organizational-unit-additional-fields-form.state";
import {DepositOrganizationalUnitState} from "@deposit/stores/organizational-unit/deposit-organizational-unit.state";
import {DepositPeopleState} from "@deposit/stores/people/deposit-people.state";
import {
  AdditionalFieldsForm,
  Deposit,
  Language,
  OrganizationalUnit,
  Person,
  PreservationPolicy,
  SubmissionPolicy,
  SystemProperty,
} from "@models";
import {Navigate} from "@ngxs/router-plugin";
import {
  Actions,
  Select,
  Store,
} from "@ngxs/store";
import {SharedAbstractDetailEditRoutable} from "@shared/components/routables/shared-abstract-detail-edit/shared-abstract-detail-edit.routable";
import {LocalStateEnum} from "@shared/enums/local-state.enum";
import {
  AppRoutesEnum,
  DepositRoutesEnum,
  RoutesEnum,
} from "@shared/enums/routes.enum";
import {SharedLanguageState} from "@shared/stores/language/shared-language.state";
import {SharedOrgUnitPreservationPolicyAction} from "@shared/stores/organizational-unit/preservation-policy/shared-organizational-unit-preservation-policy.action";
import {SharedOrganizationalUnitPreservationPolicyState} from "@shared/stores/organizational-unit/preservation-policy/shared-organizational-unit-preservation-policy.state";
import {SharedOrgUnitAction} from "@shared/stores/organizational-unit/shared-organizational-unit.action";
import {SharedOrgUnitSubmissionPolicyAction} from "@shared/stores/organizational-unit/submission-policy/shared-organizational-unit-submission-policy.action";
import {SharedOrganizationalUnitSubmissionPolicyState} from "@shared/stores/organizational-unit/submission-policy/shared-organizational-unit-submission-policy.state";
import {SharedPersonState} from "@shared/stores/person/shared-person.state";
import {Observable} from "rxjs";
import {
  filter,
  skipWhile,
  take,
  tap,
} from "rxjs/operators";
import {
  isNotNullNorUndefined,
  MemoizedUtil,
  Override,
  OverrideProperty,
} from "solidify-frontend";

@Component({
  selector: "dlcm-deposit-metadata-routable",
  templateUrl: "./deposit-metadata.routable.html",
  styleUrls: ["./deposit-metadata.routable.scss"],
  changeDetection: ChangeDetectionStrategy.OnPush,
})

export class DepositMetadataRoutable extends SharedAbstractDetailEditRoutable<Deposit, DepositStateModel> implements OnInit, OnDestroy, AfterViewInit {
  @Select(DepositState.isLoadingWithDependency) isLoadingWithDependencyObs: Observable<boolean>;
  @Select(DepositState.isReadyToBeDisplayed) isReadyToBeDisplayedObs: Observable<boolean>;
  @Select(AppState.currentUserApplicationRoleResId) currentUserApplicationRoleObs: Observable<string>;
  personConnectedObs: Observable<Person> = MemoizedUtil.current(this._store, AppPersonState);
  languagesObs: Observable<Language[]> = MemoizedUtil.list(this._store, SharedLanguageState);
  listPersonObs: Observable<Person[]> = MemoizedUtil.list(this._store, SharedPersonState);
  selectedPersonObs: Observable<Person[]> = MemoizedUtil.selected(this._store, DepositPeopleState);
  currentDepositObs: Observable<Deposit> = MemoizedUtil.current(this._store, DepositState);
  listPreservationPoliciesObs: Observable<PreservationPolicy[]> = MemoizedUtil.selected(this._store, SharedOrganizationalUnitPreservationPolicyState);
  listSubmissionPoliciesObs: Observable<SubmissionPolicy[]> = MemoizedUtil.selected(this._store, SharedOrganizationalUnitSubmissionPolicyState);
  canEditObs: Observable<boolean> = MemoizedUtil.select(this._store, DepositState, state => state.canEdit);
  orgUnitObs: Observable<OrganizationalUnit> = MemoizedUtil.select(this._store, DepositOrganizationalUnitState, (state) => state.current);
  additionalFieldsFormObs: Observable<AdditionalFieldsForm> = MemoizedUtil.current(this._store, DepositOrganizationalUnitAdditionalFieldsFormState);
  systemPropertyObs: Observable<SystemProperty> = MemoizedUtil.select(this._store, AppSystemPropertyState, state => state.current);
  logoOrgUnitObs: Observable<string> = MemoizedUtil.select(this._store, DepositOrganizationalUnitState, state => state.logo);

  _orgUnitResId: string;

  @OverrideProperty()
  @ViewChild("formPresentational")
  readonly formPresentational: DepositFormPresentational;

  readonly KEY_PARAM_NAME: keyof Deposit & string = "title";

  isTour: boolean = false;

  constructor(protected readonly _store: Store,
              protected readonly _route: ActivatedRoute,
              protected readonly _actions$: Actions,
              protected readonly _changeDetector: ChangeDetectorRef,
              protected readonly _dialog: MatDialog,
              private readonly _depositService: DepositService) {
    super(_store, _route, _actions$, _changeDetector, _dialog, LocalStateEnum.deposit, depositActionNameSpace);
  }

  ngOnInit(): void {
    super.ngOnInit();

    if (MemoizedUtil.selectSnapshot(this._store, AppState, state => state.isInTourMode)) {
      this.isTour = true;
      this._changeDetector.detectChanges();
      return;
    }

    this.retrieveResIdFromUrl();
    this.initPolicies();
    this.subscribe(this.isReadyToBeDisplayedObs.pipe(
      filter(ready => ready && isNotNullNorUndefined(this.formPresentational)),
      tap(ready => {
        this._depositService.formPresentational = this.formPresentational;
      })),
    );
  }

  protected retrieveResIdFromUrl(): void {
    this._orgUnitResId = this._route.parent.parent.snapshot.paramMap.get(AppRoutesEnum.paramIdOrgUnitWithoutPrefixParam);
    this._resId = this._route.parent.snapshot.paramMap.get(AppRoutesEnum.paramIdWithoutPrefixParam);
  }

  adaptPolicies(organizationalUnitId: string): void {
    this._store.dispatch(new SharedOrgUnitAction.GetById(organizationalUnitId));
    this._store.dispatch(new SharedOrgUnitPreservationPolicyAction.GetAll(organizationalUnitId));
    this._store.dispatch(new SharedOrgUnitSubmissionPolicyAction.GetAll(organizationalUnitId));
  }

  adaptAdditionalFieldsForm(deposit: Deposit): void {
    if (isNotNullNorUndefined(deposit.additionalFieldsValues) && isNotNullNorUndefined(deposit.additionalFieldsFormId)) {
      this._store.dispatch(new DepositOrganizationalUnitAdditionalFieldsFormAction.GetById(deposit.organizationalUnitId, deposit.additionalFieldsFormId));
    }
  }

  initPolicies(): void {
    this.subscribe(this.currentDepositObs.pipe(
      skipWhile((deposit: Deposit) => deposit === null || deposit === undefined),
      take(1),
      tap(deposit => {
        this.adaptPolicies(deposit.organizationalUnitId);
        this.adaptAdditionalFieldsForm(deposit);
      })));
  }

  ngOnDestroy(): void {
    super.ngOnDestroy();
    this._store.dispatch(new SharedOrgUnitPreservationPolicyAction.Clear());
    this._store.dispatch(new SharedOrgUnitSubmissionPolicyAction.Clear());
  }

  @Override()
  edit(): void {
    if (this.isEdit) {
      return;
    }
    this._store.dispatch(new Navigate([RoutesEnum.deposit, this._orgUnitResId, DepositRoutesEnum.detail, this._resId, DepositRoutesEnum.metadata, DepositRoutesEnum.edit], {}, {skipLocationChange: true}));
  }

  protected getSubResourceWithParentId(id: string): void {
  }
}
