import {
  ChangeDetectorRef,
  Component,
  ElementRef,
  OnInit,
} from "@angular/core";
import {
  AbstractControl,
  FormBuilder,
  FormGroup,
  Validators,
} from "@angular/forms";
import {MatDialog} from "@angular/material/dialog";
import {MatTabChangeEvent} from "@angular/material/tabs";
import {
  ActivatedRoute,
  NavigationCancel,
  NavigationEnd,
  Router,
} from "@angular/router";
import {
  DepositAction,
  depositActionNameSpace,
} from "@app/features/deposit/stores/deposit.action";
import {
  DepositState,
  DepositStateModel,
} from "@app/features/deposit/stores/deposit.state";
import {SharedAbstractListRoutable} from "@app/shared/components/routables/shared-abstract-list/shared-abstract-list.routable";
import {FieldTypeEnum} from "@app/shared/enums/field-type.enum";
import {LocalStateEnum} from "@app/shared/enums/local-state.enum";
import {
  AppRoutesEnum,
  DepositRoutesEnum,
  RoutesEnum,
} from "@app/shared/enums/routes.enum";
import {AppState} from "@app/stores/app.state";
import {
  DepositTabStatusEnum,
  DepositTabStatusName,
} from "@deposit/enums/deposit-tab-status.enum";
import {depositAuthorizedOrganizationalUnitNameSpace} from "@deposit/stores/authorized-organizational-unit/deposit-authorized-organizational-unit.action";
import {DepositAuthorizedOrganizationalUnitState} from "@deposit/stores/authorized-organizational-unit/deposit-authorized-organizational-unit.state";
import {Enums} from "@enums";
import {environment} from "@environments/environment";
import {
  Deposit,
  OrganizationalUnit,
} from "@models";
import {Navigate} from "@ngxs/router-plugin";
import {
  Actions,
  Store,
} from "@ngxs/store";
import {DataTableComponentEnum} from "@shared/enums/data-table-component.enum";
import {IconNameEnum} from "@shared/enums/icon-name.enum";
import {LabelTranslateEnum} from "@shared/enums/label-translate.enum";
import {LocalStorageEnum} from "@shared/enums/local-storage.enum";
import {TourEnum} from "@shared/enums/tour.enum";
import {PollingHelper} from "@shared/helpers/polling.helper";
import {BaseFormDefinition} from "@shared/models/base-form-definition.model";
import {DataTableActions} from "@shared/models/data-table-actions.model";
import {DataTableColumns} from "@shared/models/data-table-columns.model";
import {RouterExtService} from "@shared/services/router-ext.service";
import {UserPreferencesUtil} from "@shared/utils/user-preferences.util";
import {Observable} from "rxjs";
import {
  distinctUntilChanged,
  filter,
  tap,
} from "rxjs/operators";
import {
  FormValidationHelper,
  isNotNullNorUndefined,
  isNullOrUndefined,
  isUndefined,
  MappingObjectUtil,
  MemoizedUtil,
  OrderEnum,
  Override,
  PropertyName,
  QueryParameters,
  QueryParametersUtil,
  ResourceNameSpace,
  SolidifyValidator,
  Sort,
} from "solidify-frontend";

@Component({
  selector: "dlcm-deposit-list-routable",
  templateUrl: "./deposit-list.routable.html",
  styleUrls: ["./deposit-list.routable.scss"],
})
export class DepositListRoutable extends SharedAbstractListRoutable<Deposit, DepositStateModel> implements OnInit {
  formDefinition: FormComponentFormDefinition = new FormComponentFormDefinition();

  isInTourModeObs: Observable<boolean> = MemoizedUtil.select(this._store, AppState, state => state.isInTourMode);

  readonly KEY_CREATE_BUTTON: string = LabelTranslateEnum.createDeposit;
  readonly KEY_BACK_BUTTON: string | undefined = undefined;
  readonly KEY_PARAM_NAME: keyof Deposit & string = "title";
  readonly KEY_ORGANIZATIONAL_UNIT: keyof Deposit & string = "organizationalUnitId";
  readonly KEY_STATUS: keyof Deposit & string = "status";
  readonly KEY_STATUS_LIST: string = "statusList";

  depositAuthorizedOrganizationalUnitSort: Sort = {
    field: "name",
    order: OrderEnum.ascending,
  };
  depositAuthorizedOrganizationalUnitNameSpace: ResourceNameSpace = depositAuthorizedOrganizationalUnitNameSpace;
  depositAuthorizedOrganizationalUnitState: typeof DepositAuthorizedOrganizationalUnitState = DepositAuthorizedOrganizationalUnitState;

  form: FormGroup;

  lastSelectionOrgUnitLocalStorageKey: LocalStorageEnum = LocalStorageEnum.singleSelectLastSelectionOrgUnit;

  private _orgUnitResId: string;
  private _isInitialized: boolean = false;
  listTabStatus: DepositTabStatus[] = [
    {
      tabEnum: DepositTabStatusEnum.inProgress,
      depositStatusEnum: Enums.Deposit.StatusEnum.INPROGRESS,
      labelToTranslate: DepositTabStatusName.inProgress,
      elementCount: MemoizedUtil.select(this._store, DepositState, s => s.counterTabInProgress),
    },
    {
      tabEnum: DepositTabStatusEnum.inValidation,
      depositStatusEnum: Enums.Deposit.StatusEnum.INVALIDATION,
      labelToTranslate: DepositTabStatusName.inValidation,
      elementCount: MemoizedUtil.select(this._store, DepositState, s => s.counterTabInValidation),
    },
    {
      tabEnum: DepositTabStatusEnum.completed,
      depositStatusEnum: Enums.Deposit.StatusEnum.COMPLETED,
      labelToTranslate: DepositTabStatusName.completed,
      // elementCount: MemoizedUtil.select(this._store, DepositState, s => s.counterTabCompleted),
    },
    {
      tabEnum: DepositTabStatusEnum.inError,
      depositStatusEnum: Enums.Deposit.StatusEnum.INERROR,
      labelToTranslate: DepositTabStatusName.inError,
      elementCount: MemoizedUtil.select(this._store, DepositState, s => s.counterTabInError),
      class: "in-error",
    },
    {
      tabEnum: DepositTabStatusEnum.all,
      depositStatusEnum: undefined,
      labelToTranslate: DepositTabStatusName.all,
      // elementCount: MemoizedUtil.select(this._store, DepositState, s => s.counterTabAll),
    },
  ];
  currentTabStatus: DepositTabStatus | undefined;
  selectedTabIndex: number;

  columnsSkippedToClear: string[] = [this.KEY_ORGANIZATIONAL_UNIT, this.KEY_STATUS];

  columnStatus: DataTableColumns = {
    field: this.KEY_STATUS,
    header: LabelTranslateEnum.status,
    type: FieldTypeEnum.singleSelect,
    order: OrderEnum.none,
    isFilterable: true,
    isSortable: true,
    translate: true,
    filterEnum: Enums.Deposit.StatusEnumTranslate,
    component: DataTableComponentEnum.status,
  };
  private INTERVAL_REFRESH_IN_SECOND: number = environment.refreshTabStatusCounterIntervalInSecond;

  get tourEnum(): typeof TourEnum {
    return TourEnum;
  }

  constructor(protected readonly _store: Store,
              protected readonly _changeDetector: ChangeDetectorRef,
              protected readonly _route: ActivatedRoute,
              protected readonly _routerExt: RouterExtService,
              protected readonly _actions$: Actions,
              protected readonly _dialog: MatDialog,
              private readonly _fb: FormBuilder,
              protected readonly _router: Router) {
    super(_store, _changeDetector, _route, _routerExt, _actions$, _dialog, LocalStateEnum.deposit, depositActionNameSpace, {
      canCreate: MemoizedUtil.select(_store, DepositState, state => state.canCreate),
    });
    this.cleanIsNeeded = true;
  }

  ngOnInit(): void {
    // Allow to preserve query parameters (pagination, sort, filter) when come from detail page
    if (isNotNullNorUndefined(this._routerExt.getPreviousUrl())) {
      if (this._routerExt.getPreviousUrl().includes(DepositRoutesEnum.detail)) {
        this.cleanIsNeeded = false;
      } else {
        this.clean();
      }
    }

    super.ngOnInit();

    this.retrieveOrgUnitInUrl();

    if (MemoizedUtil.selectSnapshot(this._store, AppState, state => state.isInTourMode)) {
      this.form = this._fb.group({
        [this.formDefinition.organizationalUnitId]: [this._orgUnitResId, [Validators.required, SolidifyValidator]],
      });

      return;
    }
    this.updateCurrentTab();
    this.subscribe(this.observeRoutingUpdate());

    this.form = this._fb.group({
      [this.formDefinition.organizationalUnitId]: [this._orgUnitResId, [Validators.required, SolidifyValidator]],
    });

    this.subscribe(this._route.url.pipe(distinctUntilChanged()),
      value => {
        // TAB or ORGUNIT change
        this.retrieveOrgUnitInUrl();
        if (this._isInitialized === true) {
          let queryParameters = QueryParametersUtil.clone(MemoizedUtil.queryParametersSnapshot(this._store, DepositState));
          queryParameters = QueryParametersUtil.resetToFirstPage(queryParameters);
          this.onQueryParametersEvent(queryParameters, false);
        }
      });

    this.subscribe(this.form.get(this.formDefinition.organizationalUnitId).valueChanges.pipe(
      distinctUntilChanged(),
      tap(resId => {
        // ORGUNIT CHANGE
        this._store.dispatch(new Navigate([RoutesEnum.deposit, resId, this.currentTabStatus.tabEnum]));
        this._store.dispatch(new DepositAction.SetOrganizationalUnit(resId));
        this.refreshCounter();
      }),
    ));

    this._isInitialized = true;

    this.refreshCounter();
    this.subscribe(PollingHelper.startPollingObs({
      initialIntervalRefreshInSecond: this.INTERVAL_REFRESH_IN_SECOND,
      resetIntervalWhenUserMouseEvent: true,
      incrementInterval: true,
      maximumIntervalRefreshInSecond: 60 * 10,
      stopRefreshAfterMaximumIntervalReached: true,
      actionToDo: () => this.refreshCounter(),
    }));
  }

  private refreshCounter(): void {
    const listTabWithCounter = this.listTabStatus.filter(t => !isNullOrUndefined(t.elementCount)).map(tabStatus => tabStatus.depositStatusEnum);
    this._store.dispatch(new DepositAction.RefreshAllCounterStatusTab(listTabWithCounter));
  }

  getFormControl(key: string): AbstractControl {
    return FormValidationHelper.getFormControl(this.form, key);
  }

  private retrieveOrgUnitInUrl(): void {
    this._orgUnitResId = this._route.snapshot.parent.paramMap.get(AppRoutesEnum.paramIdOrgUnitWithoutPrefixParam);
  }

  get formValidationHelper(): typeof FormValidationHelper {
    return FormValidationHelper;
  }

  private observeRoutingUpdate(): Observable<any> {
    return this._router.events
      .pipe(
        filter(event => event instanceof NavigationCancel || event instanceof NavigationEnd),
        distinctUntilChanged(),
        tap(event => {
          this.updateCurrentTab();
        }),
      );
  }

  private updateCurrentTab(): void {
    const tab: DepositTabStatusEnum = this._route.snapshot.paramMap.get(DepositRoutesEnum.paramTabWithoutPrefix) as DepositTabStatusEnum;
    this.selectedTabIndex = this.listTabStatus.findIndex(t => t.tabEnum === tab);
    if (this.selectedTabIndex !== -1) {
      this.currentTabStatus = this.listTabStatus[this.selectedTabIndex];
    }

    this._store.dispatch(new DepositAction.SetActiveListTabStatus(this.currentTabStatus.tabEnum));
  }

  private download(deposit: Deposit): void {
    this._store.dispatch(new DepositAction.Download(deposit.resId));
  }

  onTabChanged(tabEvent: MatTabChangeEvent): void {
    const index = tabEvent.index;
    this.currentTabStatus = this.listTabStatus[index];
    this.selectedTabIndex = index;
    this._store.dispatch(new Navigate([AppRoutesEnum.deposit, this._orgUnitResId, this.currentTabStatus.tabEnum]));
  }

  conditionDisplayEditButton(model: Deposit | undefined): boolean {
    if (isUndefined(model)) {
      return true;
    }
    return model.status === Enums.Deposit.StatusEnum.INPROGRESS || model.status === Enums.Deposit.StatusEnum.INERROR;
  }

  conditionDisplayDeleteButton(model: Deposit | undefined): boolean {
    if (isUndefined(model)) {
      return true;
    }
    return model.status === Enums.Deposit.StatusEnum.INPROGRESS || model.status === Enums.Deposit.StatusEnum.INERROR;
  }

  defineColumns(): void {
    this.columns = [
      {
        field: "title",
        header: LabelTranslateEnum.title,
        type: FieldTypeEnum.string,
        order: OrderEnum.none,
        isFilterable: true,
        isSortable: true,
      },
      {
        field: "publicationDate",
        header: LabelTranslateEnum.publicationDate,
        type: FieldTypeEnum.date,
        order: OrderEnum.none,
        isFilterable: true,
        isSortable: true,
      },
      {
        field: "creation.when" as any,
        header: LabelTranslateEnum.created,
        type: FieldTypeEnum.datetime,
        order: OrderEnum.none,
        isFilterable: true,
        isSortable: true,
      },
      {
        field: "lastUpdate.when" as any,
        header: LabelTranslateEnum.updated,
        type: FieldTypeEnum.datetime,
        order: OrderEnum.descending,
        isFilterable: true,
        isSortable: true,
      },
    ];
  }

  @Override()
  protected defineActions(): DataTableActions<Deposit>[] {
    return [
      {
        logo: IconNameEnum.download,
        callback: (model: Deposit) => this.download(model),
        placeholder: current => LabelTranslateEnum.download,
        displayOnCondition: (model: Deposit) => !isNullOrUndefined(model) && !isNullOrUndefined(model.status) && model.status === Enums.Package.StatusEnum.COMPLETED,
        isWrapped: false,
      },
    ];
  }

  @Override()
  goToEdit(deposit: Deposit): void {
    this._store.dispatch(new Navigate([AppRoutesEnum.deposit, deposit.organizationalUnitId, DepositRoutesEnum.detail, deposit.resId, DepositRoutesEnum.edit], {}, {skipLocationChange: true}));
  }

  showDetail(deposit: Deposit): void {
    this._store.dispatch(new Navigate([AppRoutesEnum.deposit, deposit.organizationalUnitId, DepositRoutesEnum.detail, deposit.resId]));
  }

  create(element: ElementRef): void {
    this._store.dispatch(new Navigate([RoutesEnum.deposit, this._orgUnitResId, DepositRoutesEnum.create], {}, {skipLocationChange: true}));
  }

  private updateQueryParameterWithOrgUnit(queryParameters: QueryParameters, orgUnit: OrganizationalUnit | undefined): QueryParameters | undefined {
    if (isNullOrUndefined(orgUnit)) {
      return queryParameters;
    }
    UserPreferencesUtil.setPreferredOrgUnitInDepositMenu(orgUnit.resId);
    MappingObjectUtil.set(queryParameters.search.searchItems, this.KEY_ORGANIZATIONAL_UNIT, orgUnit.resId);
    return queryParameters;
  }

  private updateQueryParameterWithStatus(queryParameters: QueryParameters): QueryParameters | undefined {
    if (isNullOrUndefined(this.currentTabStatus)) {
      return queryParameters;
    }
    const indexOf = this.columns.findIndex(c => c.field === this.KEY_STATUS);
    const depositStatus = this.currentTabStatus.depositStatusEnum;
    if (isUndefined(this.currentTabStatus.depositStatusEnum)) {
      if (indexOf === -1) {
        this.columns.push(this.columnStatus);
        this.columns = [...this.columns];
        MappingObjectUtil.delete(queryParameters.search.searchItems, this.KEY_STATUS);
        MappingObjectUtil.delete(queryParameters.search.searchItems, this.KEY_STATUS_LIST);
      }
    } else if (depositStatus === Enums.Deposit.StatusEnum.COMPLETED) {
      if (indexOf === -1) {
        this.columns.push(this.columnStatus);
        this.columns = [...this.columns];
        MappingObjectUtil.delete(queryParameters.search.searchItems, this.KEY_STATUS);
        MappingObjectUtil.delete(queryParameters.search.searchItems, this.KEY_STATUS_LIST);
      }
      MappingObjectUtil.set(queryParameters.search.searchItems, this.KEY_STATUS_LIST, `${Enums.Deposit.StatusEnum.COMPLETED},${Enums.Deposit.StatusEnum.CLEANED}`);
    } else {
      if (indexOf !== -1) {
        this.columns.splice(indexOf, 1);
        this.columns = [...this.columns];
      }
      MappingObjectUtil.delete(queryParameters.search.searchItems, this.KEY_STATUS_LIST);
      MappingObjectUtil.set(queryParameters.search.searchItems, this.KEY_STATUS, depositStatus);
    }
    return queryParameters;
  }

  onQueryParametersEvent(queryParameters: QueryParameters, needCopy: boolean = true): void {
    if (needCopy) {
      queryParameters = QueryParametersUtil.clone(queryParameters);
    }
    this.updateQueryParameterWithOrgUnit(queryParameters, {resId: this._orgUnitResId});
    this.updateQueryParameterWithStatus(queryParameters);
    super.onQueryParametersEvent(queryParameters);
  }
}

class FormComponentFormDefinition extends BaseFormDefinition {
  @PropertyName() organizationalUnitId: string;
}

interface DepositTabStatus {
  // tabIndex: DepositTabStatusIndexEnum;
  tabEnum: DepositTabStatusEnum;
  depositStatusEnum: Enums.Deposit.StatusEnum | undefined;
  labelToTranslate: string;
  elementCount?: Observable<number> | undefined;
  class?: string;
}
