import {PersonRole} from "@admin/models/person-role.model";
import {
  ChangeDetectorRef,
  Component,
  OnDestroy,
  OnInit,
  ViewChild,
} from "@angular/core";
import {MatDialog} from "@angular/material/dialog";
import {MatTabGroup} from "@angular/material/tabs";
import {
  ActivatedRoute,
  Router,
} from "@angular/router";
import {
  DepositAction,
  depositActionNameSpace,
} from "@app/features/deposit/stores/deposit.action";
import {
  DepositState,
  DepositStateModel,
} from "@app/features/deposit/stores/deposit.state";
import {LocalStateEnum} from "@app/shared/enums/local-state.enum";
import {AppState} from "@app/stores/app.state";
import {AppBannerAction} from "@app/stores/banner/app-banner.action";
import {AppBannerState} from "@app/stores/banner/app-banner.state";
import {AppOrgUnitPersonRoleAction} from "@app/stores/organizational-unit-person-role/app-organizational-unit-person-role.action";
import {DepositRejectDialog} from "@deposit/components/dialogs/deposit-reject/deposit-reject.dialog";
import {ModeDepositTabEnum} from "@deposit/enums/mode-deposit-tab.enum";
import {
  DepositHelper,
  DepositMode,
} from "@deposit/helpers/deposit.helper";
import {DepositService} from "@deposit/services/deposit.service";
import {DepositCollectionAction} from "@deposit/stores/collection/deposit-collection.action";
import {DepositCollectionState} from "@deposit/stores/collection/deposit-collection.state";
import {DepositDataFileAction} from "@deposit/stores/data-file/deposit-data-file.action";
import {DepositDataFileState} from "@deposit/stores/data-file/deposit-data-file.state";
import {DepositPersonAction} from "@deposit/stores/people/deposit-person.action";
import {DepositStatusHistoryAction} from "@deposit/stores/status-history/deposit-status-history.action";
import {DepositStatusHistoryState} from "@deposit/stores/status-history/deposit-status-history.state";
import {Enums} from "@enums";
import {environment} from "@environments/environment";
import {Deposit} from "@models";
import {Navigate} from "@ngxs/router-plugin";
import {
  Actions,
  ofActionCompleted,
  Select,
  Store,
} from "@ngxs/store";
import {Tab} from "@shared/components/containers/shared-tabs/shared-tabs.container";
import {SharedHistoryDialog} from "@shared/components/dialogs/shared-history/shared-history.dialog";
import {SharedAbstractDetailEditRoutable} from "@shared/components/routables/shared-abstract-detail-edit/shared-abstract-detail-edit.routable";
import {BannerColorEnum} from "@shared/enums/banner-color.enum";
import {IconNameEnum} from "@shared/enums/icon-name.enum";
import {LabelTranslateEnum} from "@shared/enums/label-translate.enum";
import {
  AppRoutesEnum,
  DepositRoutesEnum,
  RoutesEnum,
} from "@shared/enums/routes.enum";
import {TourEnum} from "@shared/enums/tour.enum";
import {PollingHelper} from "@shared/helpers/polling.helper";
import {ExtraButtonToolbar} from "@shared/models/extra-button-toolbar.model";
import {StatusHistoryDialog} from "@shared/models/status-history-dialog.model";
import {StatusHistory} from "@shared/models/status-history.model";
import {RouterExtService} from "@shared/services/router-ext.service";
import {SecurityService} from "@shared/services/security.service";
import {
  BehaviorSubject,
  combineLatest,
  Observable,
} from "rxjs";

import {
  distinctUntilChanged,
  filter,
  flatMap,
  map,
  take,
  tap,
} from "rxjs/operators";
import {
  isFalse,
  isNotNullNorUndefined,
  isNullOrUndefined,
  isTrue,
  MARK_AS_TRANSLATABLE,
  MemoizedUtil,
  ObservableUtil,
  Override,
  QueryParameters,
} from "solidify-frontend";
import {SharedConfirmDialog, SharedConfirmDialogData} from "@shared/components/dialogs/shared-confirm/shared-confirm.dialog";

@Component({
  selector: "dlcm-deposit-detail-edit-routable",
  templateUrl: "./deposit-detail-edit.routable.html",
  styleUrls: ["./deposit-detail-edit.routable.scss"],
})
export class DepositDetailEditRoutable extends SharedAbstractDetailEditRoutable<Deposit, DepositStateModel> implements OnInit, OnDestroy {
  historyObs: Observable<StatusHistory[]> = MemoizedUtil.select(this._store, DepositStatusHistoryState, state => state.history);
  isLoadingHistoryObs: Observable<boolean> = MemoizedUtil.isLoading(this._store, DepositStatusHistoryState);
  isLoadingObs: Observable<boolean> = MemoizedUtil.isLoading(this._store, DepositState);
  queryParametersObs: Observable<QueryParameters> = MemoizedUtil.select(this._store, DepositStatusHistoryState, state => state.queryParameters);
  numberFilesNotReady: Observable<number | undefined> = MemoizedUtil.select(this._store, DepositDataFileState, state => state.numberFilesNotReady);
  numberFilesObs: Observable<number | undefined> = MemoizedUtil.select(this._store, DepositDataFileState, state => state.numberFiles);
  numberCollectionObs: Observable<number | undefined> = MemoizedUtil.select(this._store, DepositCollectionState, state => state.numberCollections);
  @Select(AppState.currentOrgUnitPersonRoleResId) currentPersonRoleObs: Observable<string[]>;
  @Select(AppState.currentOrgUnitPerson) currentPersonObs: Observable<PersonRole>;

  depositModeTabEnumObs: Observable<ModeDepositTabEnum> = MemoizedUtil.select(this._store, DepositState, state => state.depositModeTabEnum);

  mode: DepositMode;

  @ViewChild("matTabGroup")
  readonly matTabGroup: MatTabGroup;

  private get rootUrl(): string[] {
    return [AppRoutesEnum.deposit, this._orgUnitResId, DepositRoutesEnum.detail, this._resId];
  }

  get modeDepositTabEnum(): typeof ModeDepositTabEnum {
    return ModeDepositTabEnum;
  }

  get tourEnum(): typeof TourEnum {
    return TourEnum;
  }

  listTabs: Tab[] = [
    {
      id: TabEnum.METADATA,
      suffixUrl: DepositRoutesEnum.metadata,
      icon: IconNameEnum.metadata,
      titleToTranslate: LabelTranslateEnum.metadata,
      route: () => [...this.rootUrl, DepositRoutesEnum.metadata],
    },
    {
      id: TabEnum.DATA,
      suffixUrl: DepositRoutesEnum.data,
      icon: IconNameEnum.files,
      titleToTranslate: LabelTranslateEnum.data,
      route: () => [...this.rootUrl, DepositRoutesEnum.data],
      conditionDisplay: () => MemoizedUtil.selectSnapshot(this._store, AppState, state => state.isInTourMode) || this.depositModeTabEnumObs.pipe(map(mode => mode === ModeDepositTabEnum.FILE_OR_COLLECTION)),
    },
    {
      id: TabEnum.FILES,
      suffixUrl: DepositRoutesEnum.files,
      icon: IconNameEnum.files,
      titleToTranslate: LabelTranslateEnum.files,
      route: () => [...this.rootUrl, DepositRoutesEnum.files],
      conditionDisplay: () => combineLatest([
        this.depositModeTabEnumObs.pipe(map(mode => mode === ModeDepositTabEnum.FILE)),
        MemoizedUtil.select(this._store, DepositState, state => state?.current?.dataFileNumber > 0 || isNotNullNorUndefined(state?.logo)),
      ]).pipe(
        distinctUntilChanged(),
        map(([isFileMode, isFilePresent]) => isFileMode || isFilePresent),
      ),
    },
    {
      id: TabEnum.COLLECTIONS,
      suffixUrl: DepositRoutesEnum.collections,
      icon: IconNameEnum.files,
      titleToTranslate: LabelTranslateEnum.collections,
      route: () => [...this.rootUrl, DepositRoutesEnum.collections],
      conditionDisplay: () => this.depositModeTabEnumObs.pipe(map(mode => mode === ModeDepositTabEnum.COLLECTION)),
    },
  ];

  private currentTab: Tab;

  messageReasonUnableSubmit: string | undefined;
  canDoValidatorActions: boolean = false;
  canDoAlterationActions: boolean = false;
  canSubmitAction: boolean = false;
  canReserveDoiAction: boolean = false;

  canDoAlterationActionsBS: BehaviorSubject<boolean | undefined> = new BehaviorSubject(undefined);
  readonly canDoAlterationActionsObs: Observable<boolean | undefined> = ObservableUtil.asObservable(this.canDoAlterationActionsBS);

  private _orgUnitResId: string | undefined = undefined;

  readonly KEY_PARAM_NAME: keyof Deposit & string = "title";

  isTour: boolean = false;
  listExtraButtons: ExtraButtonToolbar<Deposit>[] = [
    {
      color: "primary",
      typeButton: "flat-button",
      icon: IconNameEnum.save,
      labelToTranslate: (current) => LabelTranslateEnum.save,
      order: 40,
      callback: () => this.depositService?.formPresentational?.onSubmit(),
      displayCondition: (resource) => this.isEdit,
      disableCondition: () => this.depositService?.formPresentational?.form?.pristine || this.depositService?.formPresentational?.form?.invalid || this.depositService?.formPresentational?.formFormly?.invalid,
    },
    {
      color: "primary",
      icon: IconNameEnum.download,
      displayCondition: current => !isNullOrUndefined(current) && !isNullOrUndefined(current.status) && current.status === Enums.Package.StatusEnum.COMPLETED,
      callback: () => this.download(),
      labelToTranslate: (current) => LabelTranslateEnum.download,
      order: 40,
    },
    {
      color: "primary",
      icon: IconNameEnum.edit,
      displayCondition: current => !this.isEdit && !isNullOrUndefined(current) && (current.status === Enums.Deposit.StatusEnum.INVALIDATION || current.status === Enums.Deposit.StatusEnum.REJECTED || current.status === Enums.Deposit.StatusEnum.INERROR),
      callback: () => this.backToRevision(),
      disableCondition: current => isNullOrUndefined(current),
      labelToTranslate: (current) => LabelTranslateEnum.returnToEditing,
      order: 21,
    },
    {
      color: "primary",
      icon: IconNameEnum.reserveDoi,
      displayCondition: current => this.displayReserveDOIButton(),
      callback: () => this.reserveDOI(),
      disableCondition: current => isNullOrUndefined(current),
      labelToTranslate: (current) => LabelTranslateEnum.reserveDoi,
      tooltipToTranslate: MARK_AS_TRANSLATABLE("deposit.tooltips.doi"),
      order: 50,
    },
    {
      color: "primary",
      icon: IconNameEnum.submit,
      displayCondition: current => this.displaySubmitButton(),
      callback: () => this.submit(),
      disableCondition: current => isNullOrUndefined(current) || !this.canSubmitAction,
      labelToTranslate: (current) => LabelTranslateEnum.submit,
      typeButton: "flat-button",
      order: 51,
    },
    {
      color: "success",
      icon: IconNameEnum.approve,
      displayCondition: current => !this.isEdit && this.canDoValidatorActions,
      callback: () => this.approve(),
      disableCondition: current => isNullOrUndefined(current),
      labelToTranslate: (current) => LabelTranslateEnum.approve,
      typeButton: "flat-button",
      order: 52,
    },
    {
      color: "warn",
      icon: IconNameEnum.unapprove,
      displayCondition: current => !this.isEdit && this.canDoValidatorActions,
      callback: () => this.reject(),
      disableCondition: current => isNullOrUndefined(current),
      labelToTranslate: (current) => LabelTranslateEnum.reject,
      typeButton: "flat-button",
      order: 53,
    },
  ];

  constructor(protected readonly _store: Store,
              protected readonly _route: ActivatedRoute,
              protected readonly _actions$: Actions,
              protected readonly _changeDetector: ChangeDetectorRef,
              protected readonly _dialog: MatDialog,
              protected readonly _router: Router,
              protected readonly _routerExt: RouterExtService,
              private readonly _securityService: SecurityService,
              public readonly depositService: DepositService) {
    super(_store, _route, _actions$, _changeDetector, _dialog, LocalStateEnum.deposit, depositActionNameSpace);
  }

  ngOnInit(): void {
    this.mode = DepositHelper.determineMode(this._store, this._routerExt);
    super.ngOnInit();

    if (MemoizedUtil.selectSnapshot(this._store, AppState, state => state.isInTourMode)) {
      this.isTour = true;
      return;
    }

    this.retrieveResource();
    this.retrieveCurrentModelWithUrl();
    this._computeCurrentUserRight();
    this._computeEditableField();
    this._createPollingListenerForFileReady();
    this._createPollingListenerCompletedStatus();
  }

  private _createPollingListenerForFileReady(): void {
    this.subscribe(PollingHelper.startPollingObs({
      initialIntervalRefreshInSecond: environment.refreshDepositSubmittedIntervalInSecond,
      incrementInterval: true,
      maximumIntervalRefreshInSecond: 60,
      stopRefreshAfterMaximumIntervalReached: true,
      resetIntervalWhenUserMouseEvent: true,
      doBeforeFilter: () => {
        const shouldContinuePollingBefore = this.shouldContinuePolling;
        this.computedShouldContinuePolling();
        if (shouldContinuePollingBefore && !this.shouldContinuePolling) {
          this._store.dispatch(new DepositDataFileAction.Refresh(this._resId));
        }
      },
      filter: () => this.shouldContinuePolling,
      actionToDo: () => {
        this._store.dispatch(new DepositDataFileAction.GetNumberFilesInError(this._resId));
        this._store.dispatch(new DepositDataFileAction.GetNumberFilesIgnored(this._resId));
        this._store.dispatch(new DepositDataFileAction.GetNumberFilesExcluded(this._resId));
        this._store.dispatch(new DepositDataFileAction.GetNumberFilesNotReady(this._resId));
      },
    }));
  }

  private _createPollingListenerCompletedStatus(): void {
    this.subscribe(PollingHelper.startPollingObs({
      initialIntervalRefreshInSecond: environment.refreshDepositSubmittedIntervalInSecond,
      incrementInterval: true,
      maximumIntervalRefreshInSecond: 60,
      stopRefreshAfterMaximumIntervalReached: true,
      resetIntervalWhenUserMouseEvent: true,
      filter: () => this.statusShouldContinuePolling(),
      actionToDo: () => {
        this._store.dispatch(new DepositAction.GetById(this._resId));
      },
    }));
  }

  private shouldContinuePolling: boolean = false;

  private statusShouldContinuePolling(): boolean {
    const status = MemoizedUtil.selectSnapshot(this._store, DepositState, state => state.current?.status);
    return isNotNullNorUndefined(status) && (status === Enums.Deposit.StatusEnum.SUBMITTED || status === Enums.Deposit.StatusEnum.CHECKED || status === Enums.Deposit.StatusEnum.APPROVED);
  }

  private computedShouldContinuePolling(): void {
    this.shouldContinuePolling = MemoizedUtil.selectSnapshot(this._store, DepositDataFileState, state => state.numberFilesInError === 0)
      && MemoizedUtil.selectSnapshot(this._store, DepositDataFileState, state => state.numberFilesIgnored === 0)
      && MemoizedUtil.selectSnapshot(this._store, DepositDataFileState, state => state.numberFilesExcluded === 0)
      && MemoizedUtil.selectSnapshot(this._store, DepositDataFileState, state => state.numberFilesNotReady > 0)
      && MemoizedUtil.selectSnapshot(this._store, DepositState, state => state?.current?.status !== Enums.Deposit.StatusEnum.CLEANED);
  }

  ngOnDestroy(): void {
    super.ngOnDestroy();
    this.cleanState();
    this._store.dispatch(new AppBannerAction.Hide());
  }

  protected retrieveResIdFromUrl(): void {
    this._resId = this._route.snapshot.paramMap.get(AppRoutesEnum.paramIdWithoutPrefixParam);
    this._orgUnitResId = this._route.parent.snapshot.paramMap.get(AppRoutesEnum.paramIdOrgUnitWithoutPrefixParam);
  }

  private _computeEditableField(): void {
    this.subscribe(this.canDoAlterationActionsObs.pipe(
      distinctUntilChanged(),
      tap(isEdit => {
        this._store.dispatch(new DepositAction.ChangeEditProperty(isEdit));
      }),
    ));
  }

  private _computeCurrentUserRight(): void {
    this.subscribe(this.currentObs.pipe(
      filter(deposit => !isNullOrUndefined(deposit) && deposit.resId === this._resId && isFalse(this.isEdit) && deposit.status === Enums.Deposit.StatusEnum.INPROGRESS),
      tap(deposit => {
        this.canReserveDoiAction = isNullOrUndefined(deposit.doi);
        this._computeBannerMessage();
      }),
    ));

    this.subscribe(this.currentObs.pipe(
      distinctUntilChanged(),
      filter(deposit => !isNullOrUndefined(deposit) && deposit.resId === this._resId),
      tap(deposit => {
        if (deposit.status === Enums.Deposit.StatusEnum.CLEANED) {
          this._computeBannerMessage();
          return;
        }
        const canDoValidatorActions = this._securityService.depositInValidationStep(deposit);
        const canDoAlterationActions = this._securityService.depositInEditionStep(deposit);

        this.subscribe(combineLatest([this.numberCollectionObs, this.numberFilesObs, this.numberFilesNotReady])
          .pipe(
            distinctUntilChanged(),
            filter(([numberCollections, numberFiles, numberFilesNotReady]) =>
              !isNullOrUndefined(numberCollections) &&
              !isNullOrUndefined(numberFiles) &&
              !isNullOrUndefined(numberFilesNotReady)),
            tap(([numberCollections, numberFiles, numberFilesNotReady]) => {
              let canSubmitAction = deposit.status === Enums.Deposit.StatusEnum.INPROGRESS;
              this.messageReasonUnableSubmit = undefined;
              if (numberFiles === 0 && numberCollections === 0) {
                this.messageReasonUnableSubmit = MARK_AS_TRANSLATABLE("deposit.error.noFileOrCollection");
                canSubmitAction = false;
              }
              if (numberFilesNotReady > 0) {
                this.messageReasonUnableSubmit = MARK_AS_TRANSLATABLE("deposit.error.fileNotReady");
                canSubmitAction = false;
              }
              this.canSubmitAction = canSubmitAction;
              this._computeBannerMessage();
            }),
          ));

        if (this._securityService.isRootOrAdmin()) {
          this.canDoValidatorActions = canDoValidatorActions;
          this.canDoAlterationActions = canDoAlterationActions;
          this.canDoAlterationActionsBS.next(this.canDoAlterationActions);
          this._computeBannerMessage();
          return;
        }

        const currentPerson = this._store.selectSnapshot(AppState.currentPerson);
        const currentOrgUnitPerson = this._store.selectSnapshot(AppState.currentOrgUnitPerson);
        if (this.checkPersonRole(currentOrgUnitPerson, deposit)) {
          this.computeCanDoValidatorAndAlterationWithCurrentRole(currentOrgUnitPerson, canDoValidatorActions, canDoAlterationActions, deposit);
          return;
        }

        this._store.dispatch(new AppOrgUnitPersonRoleAction.GetById(deposit.organizationalUnitId, currentPerson.resId));
        this.subscribe(this._actions$.pipe(
          ofActionCompleted(AppOrgUnitPersonRoleAction.GetByIdSuccess),
          flatMap(() => combineLatest([this.currentPersonObs])
            .pipe(
              filter(([personRole]) => this.checkPersonRole(personRole, deposit)),
              distinctUntilChanged(),
              tap(([personRole]) => {
                this.computeCanDoValidatorAndAlterationWithCurrentRole(personRole, canDoValidatorActions, canDoAlterationActions, deposit);
              }),
            )),
          take(1),
        ));

      }),
    ));
  }

  private computeCanDoValidatorAndAlterationWithCurrentRole(personRole: PersonRole, canDoValidatorActions: boolean, canDoAlterationActions: boolean, deposit: Deposit): void {
    let roles = [];
    if (!isNullOrUndefined(personRole.roles)) {
      roles = personRole.roles.map(r => r.resId);
    }

    this.canDoValidatorActions = canDoValidatorActions;
    this.canDoAlterationActions = canDoAlterationActions;
    if (isTrue(this.canDoValidatorActions)) {
      this.canDoValidatorActions = this._securityService.canApproveOrRejectDeposit(deposit, roles);
    }
    if (isTrue(this.canDoAlterationActions)) {
      this.canDoAlterationActions = this._securityService.canEditDeposit(deposit, roles);
    }
    this.canDoAlterationActionsBS.next(this.canDoAlterationActions);
    this._changeDetector.detectChanges();
    this._computeBannerMessage();
  }

  private checkPersonRole(personRole: PersonRole, deposit: Deposit): boolean {
    return !isNullOrUndefined(personRole) && !isNullOrUndefined(deposit) && personRole.orgUnitId === deposit.organizationalUnitId;
  }

  private displaySubmitButton(): boolean {
    return !this.isEdit && this.canDoAlterationActions && isFalse(MemoizedUtil.isLoadingSnapshot(this._store, DepositState));
  }

  private displayReserveDOIButton(): boolean {
    return !this.isEdit && this.canDoAlterationActions && this.canReserveDoiAction;
  }

  @Override()
  edit(): void {
    this._store.dispatch(new Navigate([RoutesEnum.deposit, this._orgUnitResId, DepositRoutesEnum.detail, this._resId, DepositRoutesEnum.metadata, DepositRoutesEnum.edit], {}, {skipLocationChange: true}));
  }

  getSubResourceWithParentId(id: string): void {
    this._store.dispatch(new DepositPersonAction.GetAll(id));
    this._store.dispatch(new DepositDataFileAction.Refresh(id));
    this._store.dispatch(new DepositCollectionAction.Refresh(id));
    this._store.dispatch(new DepositAction.CheckPhoto(id));
  }

  submit(): void {
    this.subscribe(this._dialog.open(SharedConfirmDialog, {
      data: {
        titleToTranslate: MARK_AS_TRANSLATABLE("deposit.dialog.submit.title"),
        messageToTranslate: MARK_AS_TRANSLATABLE("deposit.dialog.submit.message"),
        confirmButtonToTranslate: LabelTranslateEnum.yesImSure,
        cancelButtonToTranslate: LabelTranslateEnum.cancel,
        colorConfirm: "primary",
      } as SharedConfirmDialogData,
    }).afterClosed().pipe(
      tap((isConfirmed: boolean | undefined) => {
        if (isNullOrUndefined(isConfirmed)) {
          return;
        }
        if (isTrue(isConfirmed)) {
          this.subscribe(this.currentObs.pipe(
            take(1),
            tap(deposit => this._store.dispatch(new DepositAction.Submit(deposit))),
          ));
        }
      }),
    ));
  }

  reserveDOI(): void {
    this.subscribe(this.currentObs.pipe(
      take(1),
      tap(deposit => this._store.dispatch(new DepositAction.ReserveDOI(deposit))),
    ));
  }

  showHistory(): void {
    this._dialog.open(SharedHistoryDialog, {
      width: environment.modalWidth,
      data: {
        parentId: null,
        resourceResId: this._resId,
        name: this._state,
        statusHistory: this.historyObs,
        isLoading: this.isLoadingHistoryObs,
        queryParametersObs: this.queryParametersObs,
        state: DepositStatusHistoryAction,
      } as StatusHistoryDialog,
    });
  }

  approve(): void {
    this.subscribe(this.currentObs.pipe(
      take(1),
      tap(deposit => this._store.dispatch(new DepositAction.Approve(deposit))),
    ));
  }

  reject(): void {
    this.subscribe(this.currentObs.pipe(
      take(1),
      tap(deposit => {
        this._dialog.open(DepositRejectDialog, {
          minWidth: "500px",
        }).afterClosed().pipe(
          take(1),
          tap((message: string | undefined) => {
            if (isNullOrUndefined(message)) {
              return;
            }
            this._store.dispatch(new DepositAction.Reject(deposit, message));
          }),
        ).subscribe();
      }),
    ));
  }

  backToRevision(): void {
    this.subscribe(this.currentObs.pipe(
      take(1),
      tap(deposit => this._store.dispatch(new DepositAction.BackToEdit(deposit))),
    ));
  }

  backToList(): void {
    if (this.mode === DepositMode.DEPOSIT) {
      const depositTabStatus = MemoizedUtil.selectSnapshot(this._store, DepositState, state => state.activeListTabStatus);
      this._store.dispatch(new Navigate([AppRoutesEnum.deposit, this._orgUnitResId, depositTabStatus]));
    } else {
      this._store.dispatch(new Navigate([RoutesEnum.preservationPlanningDeposit]));
    }
  }

  @Override()
  backToDetail(): void {
    super.backToDetail([...this.rootUrl, DepositRoutesEnum.metadata]);
  }

  setCurrentTab($event: Tab): void {
    this.currentTab = $event;
    this._computeBannerMessage();
  }

  download(): void {
    this._store.dispatch(new DepositAction.Download(this._resId));
  }

  private _computeBannerMessage(): void {
    if (this.isEdit) {
      return;
    }
    const bannerInfo = this.getBannerDisplayInfo();
    const bannerStateModel = MemoizedUtil.selectSnapshot(this._store, AppBannerState, state => state);
    if (bannerInfo.display === bannerStateModel.display
      && bannerInfo.color === bannerStateModel.color
      && bannerInfo.message === bannerStateModel.message
    ) {
      return;
    }
    if (bannerInfo.display) {
      this._store.dispatch(new AppBannerAction.Show(bannerInfo.message, bannerInfo.color));
    } else {
      this._store.dispatch(new AppBannerAction.Hide());
    }
  }

  getBannerDisplayInfo(): BannerDisplayInfo {
    const bannerMessage = {} as BannerDisplayInfo;
    bannerMessage.display = true;

    const depositStatus = MemoizedUtil.selectSnapshot(this._store, DepositState, state => state.current?.status);

    if (depositStatus === Enums.Deposit.StatusEnum.INPROGRESS && isFalse(this.canDoAlterationActions)) {
      bannerMessage.display = false;
      return bannerMessage;
    }

    if (isFalse(this.canSubmitAction) && !isNullOrUndefined(this.messageReasonUnableSubmit)) {
      bannerMessage.message = this.messageReasonUnableSubmit;
      bannerMessage.color = BannerColorEnum.warn;
      return bannerMessage;
    }

    if (isNullOrUndefined(depositStatus)) {
      bannerMessage.display = false;
      return bannerMessage;
    }

    if (depositStatus === Enums.Deposit.StatusEnum.INERROR) {
      bannerMessage.message = MARK_AS_TRANSLATABLE("deposit.alert.status.error");
      bannerMessage.color = BannerColorEnum.error;
      return bannerMessage;
    }

    if (depositStatus === Enums.Deposit.StatusEnum.REJECTED) {
      bannerMessage.message = MARK_AS_TRANSLATABLE("deposit.alert.status.rejected");
      bannerMessage.color = BannerColorEnum.error;
      return bannerMessage;
    }

    if (depositStatus === Enums.Deposit.StatusEnum.COMPLETED || depositStatus === Enums.Deposit.StatusEnum.CLEANED) {
      bannerMessage.message = MARK_AS_TRANSLATABLE("deposit.alert.status.completed");
      bannerMessage.color = BannerColorEnum.success;
      return bannerMessage;
    }

    if (depositStatus === Enums.Deposit.StatusEnum.APPROVED ||
      depositStatus === Enums.Deposit.StatusEnum.CHECKED ||
      depositStatus === Enums.Deposit.StatusEnum.PAUSED ||
      depositStatus === Enums.Deposit.StatusEnum.SUBMITTED) {
      bannerMessage.message = MARK_AS_TRANSLATABLE("deposit.alert.status.processed");
      bannerMessage.color = BannerColorEnum.success;
      return bannerMessage;
    }

    bannerMessage.display = false;
    return bannerMessage;
  }

}

enum TabEnum {
  METADATA = "METADATA",
  DATA = "DATA",
  FILES = "FILES",
  COLLECTIONS = "COLLECTIONS",
}

interface BannerDisplayInfo {
  color: BannerColorEnum | undefined;
  message: string | undefined;
  display: boolean;
}
