import {
  ChangeDetectionStrategy,
  ChangeDetectorRef,
  Component,
  ElementRef,
  Input,
  OnInit,
  ViewChild,
} from "@angular/core";
import {
  FormBuilder,
  FormGroup,
  Validators,
} from "@angular/forms";
import {MatDialog} from "@angular/material/dialog";
import {SharedAbstractFormPresentational} from "@app/shared/components/presentationals/shared-abstract-form/shared-abstract-form.presentational";
import {BaseFormDefinition} from "@app/shared/models/base-form-definition.model";
import {sharedLicenseActionNameSpace} from "@app/shared/stores/license/shared-license.action";
import {sharedPersonActionNameSpace} from "@app/shared/stores/person/shared-person.action";
import {
  DepositOrderAuthorDialog,
  DepositOrderAuthorDialogWrapper,
} from "@deposit/components/dialogs/deposit-order-author/deposit-order-author.dialog";
import {DepositPersonDialog} from "@deposit/components/dialogs/deposit-person/deposit-person.dialog";
import {depositActionNameSpace} from "@deposit/stores/deposit.action";
import {DepositState} from "@deposit/stores/deposit.state";
import {Enums} from "@enums";
import {environment} from "@environments/environment";
import {
  AdditionalFieldsForm,
  AipInfoEmbargo,
  Deposit,
  Language,
  License,
  OrganizationalUnit,
  Person,
  PreservationPolicy,
  SubmissionPolicy,
} from "@models";
import {FormlyFieldConfig} from "@ngx-formly/core";
import {TranslateService} from "@ngx-translate/core";
import {SharedSearchableMultiSelectPresentational} from "@shared/components/presentationals/shared-searchable-multi-select/shared-searchable-multi-select.presentational";
import {LocalStorageEnum} from "@shared/enums/local-storage.enum";
import {RoutesEnum} from "@shared/enums/routes.enum";
import {TourEnum} from "@shared/enums/tour.enum";
import {BreakpointService} from "@shared/services/breakpoint.service";
import {SharedLicenseState} from "@shared/stores/license/shared-license.state";
import {
  SharedPersonState,
  SharedPersonStateModel,
} from "@shared/stores/person/shared-person.state";
import {ResourceLogoNameSpace} from "@shared/stores/resource-logo/resource-logo-namespace.model";
import {Observable} from "rxjs";
import {
  distinctUntilChanged,
  filter,
  tap,
} from "rxjs/operators";
import {
  DateUtil,
  EnumUtil,
  isNonEmptyArray,
  isNotNullNorUndefined,
  isNullOrUndefined,
  KeyValue,
  MappingObjectUtil,
  OrderEnum,
  OverrideProperty,
  PropertyName,
  ResourceNameSpace,
  SolidifyValidator,
  Sort,
  urlSeparator,
} from "solidify-frontend";

@Component({
  selector: "dlcm-deposit-form",
  templateUrl: "./deposit-form.presentational.html",
  styleUrls: ["./deposit-form.presentational.scss"],
  changeDetection: ChangeDetectionStrategy.OnPush,
})
export class DepositFormPresentational extends SharedAbstractFormPresentational<Deposit> implements OnInit {
  formDefinition: FormComponentFormDefinition = new FormComponentFormDefinition();

  accessEnumValues: KeyValue[] = Enums.Deposit.AccessEnumTranslate;
  dataSensitivityEnumValues: KeyValue[] = Enums.Deposit.DataSensitivityEnumTranslate;
  embargoEnumValues: KeyValue[] = [];

  additionalFieldsValues: Object;

  formFormly: FormGroup;

  @Input()
  orgUnitLogo: string;

  @Input()
  isTourMode: boolean = false;

  addFieldsForm: AdditionalFieldsForm;

  get depositActionNameSpace(): ResourceLogoNameSpace {
    return depositActionNameSpace;
  }

  get depositState(): typeof DepositState {
    return DepositState;
  }

  get accessEnum(): typeof Enums.Deposit.AccessEnum {
    return Enums.Deposit.AccessEnum;
  }

  get tourEnum(): typeof TourEnum {
    return TourEnum;
  }

  @Input()
  set additionalFieldsForm(additionalFieldsForm: AdditionalFieldsForm) {
    if (isNotNullNorUndefined(additionalFieldsForm)) {
      this.addFieldsForm = additionalFieldsForm;
      this.fields = JSON.parse(additionalFieldsForm.description);
    }
  }

  fields: FormlyFieldConfig[];

  @Input()
  defaultPlatformLicenseId: string;

  @Input()
  isReady: boolean = false;

  @Input()
  languages: Language[];

  @Input()
  listSubmissionPolicies: SubmissionPolicy[];

  @Input()
  listPreservationPolicies: PreservationPolicy[];

  @Input()
  selectedPersons: Person[];

  @Input()
  listPersons: Person[];

  @Input()
  personConnected: Person;

  @Input()
  currentApplicationRole: Enums.UserApplicationRole.UserApplicationRoleEnum;

  @Input()
  orgUnit: OrganizationalUnit;

  @ViewChild("authorMultiSearchableSelect")
  authorMultiSearchableSelect: SharedSearchableMultiSelectPresentational<SharedPersonStateModel, Person>;

  defaultLicenseId: string;
  commonValueLicenseId: string[];

  readonly TIME_BEFORE_DISPLAY_TOOLTIP: number = environment.timeBeforeDisplayTooltipOnInput;

  get userApplicationRoleEnum(): typeof Enums.UserApplicationRole.UserApplicationRoleEnum {
    return Enums.UserApplicationRole.UserApplicationRoleEnum;
  }

  sharedPersonSort: Sort<Person> = {
    field: "lastName",
    order: OrderEnum.ascending,
  };
  sharedPersonActionNameSpace: ResourceNameSpace = sharedPersonActionNameSpace;
  sharedPersonState: typeof SharedPersonState = SharedPersonState;

  sharedLicenseSort: Sort<License> = {
    field: "title",
    order: OrderEnum.ascending,
  };
  sharedLicenseActionNameSpace: ResourceNameSpace = sharedLicenseActionNameSpace;
  sharedLicenseState: typeof SharedLicenseState = SharedLicenseState;

  licenseCallback: (value: License) => string = (value: License) => value.openLicenseId + " (" + value.title + ")";

  lastSelectionContributorLocalStorageKey: LocalStorageEnum = LocalStorageEnum.multiSelectLastSelectionContributor;

  @OverrideProperty()
  protected _listFieldNameToDisplayErrorInToast: string[] = [
    this.formDefinition.organizationalUnitId,
  ];

  get metadataVersionEnum(): typeof Enums.Deposit.MetadataVersionEnum {
    return Enums.Deposit.MetadataVersionEnum;
  }

  constructor(protected readonly _changeDetectorRef: ChangeDetectorRef,
              protected readonly _elementRef: ElementRef,
              private readonly _fb: FormBuilder,
              private readonly _dialog: MatDialog,
              public readonly breakpointService: BreakpointService,
              private readonly _translateService: TranslateService) {
    super(_changeDetectorRef, _elementRef);
  }

  ngOnInit(): void {
    if (this.isTourMode) {
      this.bindFormTo({
        title: "Deposit Tour",
        additionalFieldsValues: "{}",
        status: Enums.Deposit.StatusEnum.INPROGRESS,
      } as any);
      return;
    }
    super.ngOnInit();
    this.validationLicenseRequiredWhenAccessPublic();
    this.validationEmbargo();
    this.subscribe(this._listenerAccessLevel());
    this.defaultLicenseId = this.orgUnit?.defaultLicense?.resId ?? this.defaultPlatformLicenseId;
    this.commonValueLicenseId = [this.defaultLicenseId];
  }

  private _listenerAccessLevel(): Observable<string> {
    return this.form.get(this.formDefinition.access).valueChanges.pipe(
      tap(() => {
        this.form.get(this.formDefinition.dataSensitivity).updateValueAndValidity();
      }),
    );
  }

  private validationLicenseRequiredWhenAccessPublic(): void {
    const licenseFormControl = this.form.get(this.formDefinition.licenseId);
    this.subscribe(this.form.get(this.formDefinition.access).valueChanges.pipe(
      distinctUntilChanged(),
      tap(accessLevel => {
        if (accessLevel === Enums.Deposit.AccessEnum.PUBLIC) {
          licenseFormControl.setValidators([Validators.required]);
        } else {
          licenseFormControl.setValidators([]);
        }
        licenseFormControl.updateValueAndValidity();
      }),
    ));
  }

  protected initNewForm(): void {
    this.initEmbargoAccessLevel(Enums.Deposit.AccessEnum.PUBLIC);
    const formDesc = {
      [this.formDefinition.organizationalUnitId]: [this.orgUnit.resId, [Validators.required, SolidifyValidator]],
      [this.formDefinition.title]: ["", [Validators.required, SolidifyValidator]],
      [this.formDefinition.description]: ["", [Validators.required, SolidifyValidator]],
      [this.formDefinition.languageId]: [""],
      [this.formDefinition.publicationDate]: [new Date(), [Validators.required, SolidifyValidator]],
      [this.formDefinition.collectionBegin]: [""],
      [this.formDefinition.collectionEnd]: [""],
      [this.formDefinition.access]: [Enums.Deposit.AccessEnum.PUBLIC],
      [this.formDefinition.dataSensitivity]: [Enums.Deposit.DataSensitivityEnum.UNDEFINED],
      [this.formDefinition.hasEmbargo]: [false],
      [this.formDefinition.licenseId]: ["", [Validators.required, SolidifyValidator]],
      [this.formDefinition.submissionPolicyId]: [this.orgUnit?.defaultSubmissionPolicy?.resId, [Validators.required, SolidifyValidator]],
      [this.formDefinition.preservationPolicyId]: [this.orgUnit?.defaultPreservationPolicy?.resId, [Validators.required, SolidifyValidator]],
      [this.formDefinition.authors]: ["", [Validators.required, SolidifyValidator]],
      [this.formDefinition.keywords]: [[...this.orgUnit?.keywords], [SolidifyValidator]],
      [this.formDefinition.doi]: [null, {disabled: true}],
      [this.formDefinition.embargoAccess]: [null],
      [this.formDefinition.embargoNumberMonths]: [""],
    };

    if (isNotNullNorUndefined(this.addFieldsForm)) {
      formDesc[this.formDefinition.additionalFieldsFormId] = [this.addFieldsForm.resId, [Validators.required, SolidifyValidator]];
    }

    this.form = this._fb.group(formDesc);
    this.formFormly = this._fb.group({});
  }

  protected bindFormTo(deposit: Deposit): void {
    this.initEmbargoAccessLevel(deposit.access);
    let formDesc = {
      [this.formDefinition.organizationalUnitId]: [deposit.organizationalUnitId, [Validators.required, SolidifyValidator]],
      [this.formDefinition.title]: [deposit.title, [Validators.required, SolidifyValidator]],
      [this.formDefinition.description]: [deposit.description, [Validators.required, SolidifyValidator]],
      [this.formDefinition.languageId]: [deposit.languageId],
      [this.formDefinition.publicationDate]: [deposit.publicationDate, [Validators.required, SolidifyValidator]],
      [this.formDefinition.collectionBegin]: [DateUtil.convertOffsetDateTimeIso8601ToDate(deposit.collectionBegin), [SolidifyValidator]],
      [this.formDefinition.collectionEnd]: [DateUtil.convertOffsetDateTimeIso8601ToDate(deposit.collectionEnd), [SolidifyValidator]],
      [this.formDefinition.access]: [deposit.access],
      [this.formDefinition.dataSensitivity]: [deposit.dataSensitivity],
      [this.formDefinition.hasEmbargo]: [deposit.hasEmbargo],
      [this.formDefinition.licenseId]: [deposit.licenseId],
      [this.formDefinition.submissionPolicyId]: [isNullOrUndefined(deposit.submissionPolicyId) ? this.orgUnit?.defaultSubmissionPolicy?.resId : deposit.submissionPolicyId, [Validators.required, SolidifyValidator]],
      [this.formDefinition.preservationPolicyId]: [isNullOrUndefined(deposit.preservationPolicyId) ? this.orgUnit?.defaultPreservationPolicy?.resId : deposit.preservationPolicyId, [Validators.required, SolidifyValidator]],
      [this.formDefinition.authors]: [this.selectedPersons?.map(p => p.resId), [Validators.required, SolidifyValidator]],
      [this.formDefinition.keywords]: [isNullOrUndefined(deposit.keywords) ? [] : [...deposit.keywords], [SolidifyValidator]],
      [this.formDefinition.doi]: [deposit.doi, {disabled: true}],
      [this.formDefinition.embargoAccess]: [isNullOrUndefined(deposit.embargo) ? "" : deposit.embargo.access],
      [this.formDefinition.embargoNumberMonths]: [isNullOrUndefined(deposit.embargo) ? "" : deposit.embargo.months],
      [this.formDefinition.embargoStartDate]: [isNullOrUndefined(deposit.embargo?.startDate) ? "" : DateUtil.convertDateToDateTimeString(new Date(deposit.embargo.startDate))],
      [this.formDefinition.status]: [deposit.status, [Validators.required, SolidifyValidator]],
    };

    if (isNotNullNorUndefined(deposit.additionalFieldsFormId)) {
      formDesc = {
        ...formDesc,
        [this.formDefinition.additionalFieldsFormId]: [deposit.additionalFieldsFormId, [Validators.required, SolidifyValidator]],
      };
    }

    this.form = this._fb.group(formDesc);

    this.formFormly = this._fb.group({}); // TODO INIT FORM WITH MODEL
    this.additionalFieldsValues = JSON.parse(deposit.additionalFieldsValues);

    this.isValidWhenDisable = this.form.valid;
  }

  protected treatmentBeforeSubmit(deposit: Deposit): Deposit {
    deposit.publicationDate = DateUtil.convertToLocalDateDateSimple(deposit.publicationDate);
    deposit.collectionBegin = DateUtil.convertToOffsetDateTimeIso8601(deposit.collectionBegin);
    deposit.collectionEnd = DateUtil.convertToOffsetDateTimeIso8601(deposit.collectionEnd);
    deposit.collectionEnd = DateUtil.convertToOffsetDateTimeIso8601(deposit.collectionEnd);
    if (isNotNullNorUndefined(this.formFormly.value) && MappingObjectUtil.size(this.formFormly.value) > 0) {
      deposit.additionalFieldsValues = JSON.stringify(this.formFormly.value);
    } else {
      deposit.additionalFieldsValues = null;
    }
    if (deposit.hasEmbargo) {
      const newEmbargo: AipInfoEmbargo = {
        access: this.form.get(this.formDefinition.embargoAccess).value === "" ? null : this.form.get(this.formDefinition.embargoAccess).value,
        months: this.form.get(this.formDefinition.embargoNumberMonths).value === "" ? null : this.form.get(this.formDefinition.embargoNumberMonths).value,
      };
      deposit.embargo = newEmbargo;
    } else {
      deposit.embargo = null;
    }
    delete deposit[this.formDefinition.embargoNumberMonths];
    delete deposit[this.formDefinition.embargoAccess];
    return deposit;
  }

  private validationEmbargo(): void {
    const embargoDurationFormControl = this.form.get(this.formDefinition.embargoNumberMonths);
    const embargoAccessFormControl = this.form.get(this.formDefinition.embargoAccess);
    this.subscribe(this.form.get(this.formDefinition.hasEmbargo).valueChanges.pipe(
      distinctUntilChanged(),
      tap(hasEmbargo => {
        if (hasEmbargo === true) {
          embargoDurationFormControl.setValidators([Validators.required, Validators.min(1)]);
          embargoAccessFormControl.setValidators([Validators.required]);
        } else {
          embargoDurationFormControl.setValidators([]);
          embargoAccessFormControl.setValidators([]);
        }
        embargoDurationFormControl.updateValueAndValidity();
        embargoAccessFormControl.updateValueAndValidity();
      }),
    ));
  }

  protected disableSpecificField(): void {
    this.form.get(this.formDefinition.doi).disable();
  }

  addPeople(): void {
    this.subscribe(
      this._dialog.open(DepositPersonDialog, {
        width: "90%",
      }).afterClosed().pipe(
        tap((person: Person) => {
          if (isNullOrUndefined(person)) {
            return;
          }
          this.listPersons = [...this.listPersons, person];
          this.addAuthor(person);
        }),
      ),
    );
  }

  addMeAsAuthor(): void {
    this.addAuthor(this.personConnected);
  }

  addAuthor(person: Person): void {
    this.authorMultiSearchableSelect.add(person);
  }

  personConnectedAlreadyInAuthor(): boolean {
    return this.form.get(this.formDefinition.authors).value.includes(this.personConnected.resId);
  }

  getSipPath(sipId: string): string {
    return "#" + urlSeparator + RoutesEnum.preservationPlanningSipDetail + urlSeparator + sipId;
  }

  navigateToPerson(person: Person): void {
    this.navigate([RoutesEnum.adminPersonDetail, person.resId]);
  }

  orderAuthors(listContributor: Person[]): void {
    this.subscribe(
      this._dialog.open(DepositOrderAuthorDialog, {
        width: "500px",
        data: {
          authorsId: this.form.get(this.formDefinition.authors).value,
          persons: listContributor,
        } as DepositOrderAuthorDialogWrapper,
      }).afterClosed().pipe(
        filter((authorsIdSorted: string[]) => isNonEmptyArray(authorsIdSorted)),
        tap((authorsIdSorted: string[]) => {
          const formControl = this.form.get(this.formDefinition.authors);
          formControl.setValue(authorsIdSorted);
          formControl.markAsDirty();
        }),
      ),
    );
  }

  formlyChange(): void {
    if (this.form.dirty) {
      return;
    }
    this.form.markAsDirty();
    this._changeDetectorRef.detectChanges();
  }

  adaptEmbargoAccessLevel(value: Enums.Deposit.AccessEnum): void {
    this.initEmbargoAccessLevel(value);
    this.form.get(this.formDefinition.embargoAccess).setValue(null);
  }

  initEmbargoAccessLevel(value: Enums.Deposit.AccessEnum): void {
    this.embargoEnumValues = [];
    this.embargoEnumValues = Enums.Deposit.EmbargoAccessLevelEnumList(value);
  }

  get enumUtil(): typeof EnumUtil {
    return EnumUtil;
  }

  get statusEnum(): typeof Enums.Deposit.StatusEnum {
    return Enums.Deposit.StatusEnum;
  }

  get depositStatusEnumToTranslate(): typeof Enums.Deposit.StatusEnumTranslate {
    return Enums.Deposit.StatusEnumTranslate;
  }
}

class FormComponentFormDefinition extends BaseFormDefinition {
  @PropertyName() organizationalUnitId: string;
  @PropertyName() title: string;
  @PropertyName() description: string;
  @PropertyName() languageId: string;
  @PropertyName() publicationDate: string;
  @PropertyName() collectionBegin: string;
  @PropertyName() collectionEnd: string;
  @PropertyName() authors: string;
  @PropertyName() access: string;
  @PropertyName() dataSensitivity: string;
  @PropertyName() hasEmbargo: string;
  @PropertyName() embargoAccess: string;
  @PropertyName() embargoNumberMonths: string;
  @PropertyName() embargoStartDate: string;
  @PropertyName() licenseId: string;
  @PropertyName() submissionPolicyId: string;
  @PropertyName() preservationPolicyId: string;
  @PropertyName() keywords: string;
  @PropertyName() doi: string;
  @PropertyName() additionalFieldsFormId: string;
  @PropertyName() status: string;
}
