import {
  ChangeDetectionStrategy,
  Component,
  Input,
  Output,
} from "@angular/core";
import {MatDialog} from "@angular/material/dialog";
import {FileUploadStatusEnum} from "@app/features/deposit/enums/file-upload-status.enum";
import {DepositDataFile} from "@app/features/deposit/models/deposit-data-file.model";
import {UploadFileStatus} from "@app/features/deposit/models/upload-file-status.model";
import {SharedAbstractPresentational} from "@app/shared/components/presentationals/shared-abstract/shared-abstract.presentational";
import {
  BehaviorSubject,
  Observable,
} from "rxjs";
import {
  isNullOrUndefined,
  ObservableUtil,
} from "solidify-frontend";

@Component({
  selector: "dlcm-deposit-file",
  templateUrl: "./deposit-file.presentational.html",
  styleUrls: ["./deposit-file.presentational.scss"],
  changeDetection: ChangeDetectionStrategy.OnPush,
})
export class DepositFilePresentational extends SharedAbstractPresentational {
  @Input()
  depositDataFile: DepositDataFile;

  @Input()
  uploadStatus: UploadFileStatus;

  @Input()
  isLoadingHistory: boolean;

  @Input()
  readonly: boolean;

  private readonly _cancelBS: BehaviorSubject<UploadFileStatus | undefined> = new BehaviorSubject<UploadFileStatus | undefined>(undefined);
  @Output("cancelChange")
  readonly cancelObs: Observable<UploadFileStatus | undefined> = ObservableUtil.asObservable(this._cancelBS);

  private readonly _retryBS: BehaviorSubject<UploadFileStatus | undefined> = new BehaviorSubject<UploadFileStatus | undefined>(undefined);
  @Output("retryChange")
  readonly retryObs: Observable<UploadFileStatus | undefined> = ObservableUtil.asObservable(this._retryBS);

  private readonly _resumeBS: BehaviorSubject<DepositDataFile | undefined> = new BehaviorSubject<DepositDataFile | undefined>(undefined);
  @Output("resumeChange")
  readonly resumeObs: Observable<DepositDataFile | undefined> = ObservableUtil.asObservable(this._resumeBS);

  private readonly _deleteBS: BehaviorSubject<DepositDataFile | undefined> = new BehaviorSubject<DepositDataFile | undefined>(undefined);
  @Output("deleteChange")
  readonly deleteObs: Observable<DepositDataFile | undefined> = ObservableUtil.asObservable(this._deleteBS);

  private readonly _downloadBS: BehaviorSubject<DepositDataFile | undefined> = new BehaviorSubject<DepositDataFile | undefined>(undefined);
  @Output("downloadChange")
  readonly downloadObs: Observable<DepositDataFile | undefined> = ObservableUtil.asObservable(this._downloadBS);

  private readonly _showHistoryBS: BehaviorSubject<DepositDataFile | undefined> = new BehaviorSubject<DepositDataFile | undefined>(undefined);
  @Output("showHistoryChange")
  readonly showHistoryObs: Observable<DepositDataFile | undefined> = ObservableUtil.asObservable(this._showHistoryBS);

  get fileUploadStatusEnum(): typeof FileUploadStatusEnum {
    return FileUploadStatusEnum;
  }

  private readonly _IN_ERROR_STATUS: string = "IN_ERROR";

  constructor(public dialog: MatDialog) {
    super();
  }

  cancel(): void {
    this._cancelBS.next(this.uploadStatus);
  }

  isInError(): boolean {
    return this.depositDataFile.status === this._IN_ERROR_STATUS;
  }

  delete(): void {
    this._deleteBS.next(this.depositDataFile);
  }

  resume(): void {
    this._resumeBS.next(this.depositDataFile);
  }

  showHistory(): void {
    this._showHistoryBS.next(this.depositDataFile);
  }

  isUploaded(): boolean {
    return !isNullOrUndefined(this.depositDataFile) && isNullOrUndefined(this.uploadStatus);
  }

  isUploading(): boolean {
    return !this.isUploaded();
  }

  getColor(): string {
    if (this.uploadStatus.status === FileUploadStatusEnum.inError) {
      return "warn";
    }
    if (this.uploadStatus.status === FileUploadStatusEnum.completed) {
      return "accent";
    }
  }

  retry(): void {
    this._retryBS.next(this.uploadStatus);
  }

  download(depositDataFile: DepositDataFile): void {
    this._downloadBS.next(depositDataFile);
  }
}
