import {
  ChangeDetectionStrategy,
  Component,
  ViewChild,
} from "@angular/core";
import {
  MatDialog,
  MatDialogRef,
} from "@angular/material/dialog";
import {
  AlternativeDepositPersonSelected,
  AlternativeDepositPersonWrapper,
  DepositPersonAlternativeDialog,
} from "@deposit/components/dialogs/deposit-person-alternative/deposit-person-alternative.dialog";
import {Person} from "@models";
import {
  Actions,
  ofActionCompleted,
  Store,
} from "@ngxs/store";
import {SharedAbstractContainer} from "@shared/components/containers/shared-abstract/shared-abstract.container";
import {SharedPersonFormPresentational} from "@shared/components/presentationals/shared-person-form/shared-person-form.presentational";
import {LocalStateModel} from "@shared/models/local-state.model";
import {SharedPersonAction} from "@shared/stores/person/shared-person.action";
import {SharedPersonState} from "@shared/stores/person/shared-person.state";
import {Observable} from "rxjs";
import {tap} from "rxjs/operators";
import {
  isEmptyArray,
  isNullOrUndefined,
  isTrue,
  MemoizedUtil,
  ModelFormControlEvent,
  NotificationService,
} from "solidify-frontend";

@Component({
  selector: "dlcm-deposit-person-dialog",
  templateUrl: "./deposit-person.dialog.html",
  styleUrls: ["./deposit-person.dialog.scss"],
  changeDetection: ChangeDetectionStrategy.OnPush,
})
export class DepositPersonDialog extends SharedAbstractContainer {
  isLoadingPersonObs: Observable<boolean> = MemoizedUtil.isLoading(this.store, SharedPersonState);

  @ViewChild("formPresentational")
  readonly formPresentational: SharedPersonFormPresentational;

  constructor(protected readonly store: Store,
              protected readonly dialogRef: MatDialogRef<DepositPersonDialog>,
              private readonly notificationService: NotificationService,
              private readonly dialog: MatDialog,
              private readonly actions$: Actions) {
    super();
  }

  savePerson($event: ModelFormControlEvent<Person>): void {
    this.checkSimilarPerson($event);
  }

  private checkSimilarPerson($event: ModelFormControlEvent<Person>): void {
    this.subscribe(this.store.dispatch(new SharedPersonAction.Search($event.model)).pipe(
      tap((state: LocalStateModel) => {
        const listPersonMatching = state.shared.shared_person.listPersonMatching;
        if (isNullOrUndefined(listPersonMatching) || isEmptyArray(listPersonMatching)) {
          this.createPerson($event);
        } else {
          this.openAlternativeDialog($event, listPersonMatching);
        }
      }),
    ));
  }

  private createPerson($event: ModelFormControlEvent<Person>): void {
    this.store.dispatch(new SharedPersonAction.Create($event));
    this.subscribe(this.actions$.pipe(
      ofActionCompleted(SharedPersonAction.CreateSuccess),
      tap((result) => {
        if (isTrue(result.result.successful)) {
          this.dialogRef.close(result.action.model);
        }
      }),
    ));
  }

  private openAlternativeDialog(newPerson: ModelFormControlEvent<Person>, listMatchingPerson: Person[]): void {
    this.subscribe(this.dialog.open(DepositPersonAlternativeDialog, {
      data: {
        listExistingPerson: listMatchingPerson,
        newPerson: newPerson.model,
      } as AlternativeDepositPersonWrapper,
    }).afterClosed().pipe(
      tap((depositPersonSelected: AlternativeDepositPersonSelected) => {
        if (isNullOrUndefined(depositPersonSelected)) {
          return;
        }
        if (depositPersonSelected.isNew) {
          this.createPerson(newPerson);
        } else {
          this.dialogRef.close(depositPersonSelected.selected);
        }
      }),
    ));
  }
}
