import {
  ChangeDetectionStrategy,
  Component,
  Inject,
  OnInit,
} from "@angular/core";
import {FormBuilder} from "@angular/forms";
import {
  MAT_DIALOG_DATA,
  MatDialogRef,
} from "@angular/material/dialog";
import {FileUploadWrapper} from "@app/features/deposit/models/file-upload-wrapper.model";
import {DataCategoryHelper} from "@app/shared/helpers/data-category.helper";
import {
  AbstractDepositFileUploadDialog,
  AbstractDepositFileUploadFormComponentFormDefinition,
} from "@deposit/components/dialogs/abstract-deposit-file-upload/abstract-deposit-file-upload.dialog";
import {Enums} from "@enums";
import {
  isNotNullNorUndefined,
  isNullOrUndefined,
  PropertyName,
} from "solidify-frontend";

@Component({
  selector: "dlcm-deposit-file-upload-dialog",
  templateUrl: "./deposit-file-upload.dialog.html",
  styleUrls: ["./deposit-file-upload.dialog.scss"],
  changeDetection: ChangeDetectionStrategy.OnPush,
})
export class DepositFileUploadDialog extends AbstractDepositFileUploadDialog implements OnInit {
  formDefinition: FormComponentFormDefinition = new FormComponentFormDefinition();
  filesToUpload: FileWrapper[] = [];
  isFormValid: boolean = true;

  get dataCategoryHelper(): typeof DataCategoryHelper {
    return DataCategoryHelper;
  }

  constructor(protected _dialogRef: MatDialogRef<DepositFileUploadDialog>,
              protected _fb: FormBuilder,
              @Inject(MAT_DIALOG_DATA) public data: DepositFileUploadDialogData) {
    super(_dialogRef, _fb);
  }

  ngOnInit(): void {
    super.ngOnInit();
    this.form.addControl(this.formDefinition.subDirectory, this._fb.control(this.data.subDirectory));

    if (isNotNullNorUndefined(this.data.dataCategoryEnum)) {
      const dataCategoryFormControl = this.form.get(this.formDefinition.dataCategory);
      dataCategoryFormControl.setValue(this.data.dataCategoryEnum);
      if (this.data.dataCategoryEnum === Enums.DataFile.DataCategoryEnum.Secondary) {
        this.isFormValid = false;
      }
    }
  }

  onFileChange(event: Event & any): void {
    if (event.target.files.length > 0) {
      const files: FileList = event.target.files;
      this.addNewFileToExistingList(this.convertFileListToArray(files));
    }
  }

  convertFileListToArray(files: FileList): File[] {
    const arrayFile: File[] = [];
    for (let i = 0; i < files.length; i++) {
      arrayFile.push(files.item(i));
    }
    return arrayFile;
  }

  addNewFileToExistingList(files: File[]): void {
    files = files.filter(fileToFilter => {
      const isIncluded = this.getIndexOfFile(fileToFilter) !== -1;
      return !isIncluded;
    });
    const listFileWrapper: FileWrapper[] = [];
    files.forEach(f => {
      listFileWrapper.push({
        file: f,
        isInError: this.isFileNameInvalid(f),
      });
    });
    this.filesToUpload.push(...listFileWrapper);
  }

  private getIndexOfFile(file: File): number {
    return this.filesToUpload.findIndex(fileAlreadyToUpload => fileAlreadyToUpload.file.name === file.name && fileAlreadyToUpload.file.type === file.type && fileAlreadyToUpload.file.size === file.size && fileAlreadyToUpload.file.lastModified === file.lastModified);
  }

  onSubmit(): void {
    const base = this.form.value as FileUploadWrapper;
    const listFilesUploadWrapper: FileUploadWrapper[] = [];
    this.filesToUpload.forEach((f: FileWrapper) => {
      const fileUploadWrapper = {} as FileUploadWrapper;
      Object.assign(fileUploadWrapper, base);
      fileUploadWrapper.file = f.file;
      listFilesUploadWrapper.push(fileUploadWrapper);
    });
    this._dialogRef.close(listFilesUploadWrapper);
  }

  delete(file: File): void {
    this.filesToUpload.splice(this.getIndexOfFile(file), 1);
  }

  isFileNameInvalid(file: File): boolean {
    const fileName = file.name;
    return this.REGEX_INVALID.test(fileName);
  }

  isCurrentFileNameInError(): boolean {
    return !isNullOrUndefined(this.filesToUpload.find(f => f.isInError));
  }

  selectCategory(): void {
    this.isFormValid = false;
  }

  selectDataType(): void {
    this.isFormValid = true;
  }
}

class FormComponentFormDefinition extends AbstractDepositFileUploadFormComponentFormDefinition {
  @PropertyName() subDirectory: string;
}

class FileWrapper {
  file: File;
  isInError: boolean;
}

export interface DepositFileUploadDialogData {
  dataCategoryEnum?: Enums.DataFile.DataCategoryEnum;
  subDirectory?: string;
}
