import {
  ChangeDetectionStrategy,
  Component,
  Inject,
  OnInit,
} from "@angular/core";
import {
  AbstractControl,
  FormBuilder,
  FormGroup,
  Validators,
} from "@angular/forms";
import {
  MAT_DIALOG_DATA,
  MatDialogRef,
} from "@angular/material/dialog";
import {DepositDataFile} from "@deposit/models/deposit-data-file.model";
import {DepositDataFileAction} from "@deposit/stores/data-file/deposit-data-file.action";
import {DepositDataFileState} from "@deposit/stores/data-file/deposit-data-file.state";
import {
  Actions,
  ofActionCompleted,
  Store,
} from "@ngxs/store";
import {SharedAbstractContainer} from "@shared/components/containers/shared-abstract/shared-abstract.container";
import {BaseFormDefinition} from "@shared/models/base-form-definition.model";
import {RegexpUtil} from "@shared/utils/regexp.util";
import {Observable} from "rxjs";
import {
  take,
  tap,
} from "rxjs/operators";
import {
  FormValidationHelper,
  isTrue,
  MemoizedUtil,
  PropertyName,
  SolidifyValidator,
} from "solidify-frontend";

@Component({
  selector: "dlcm-deposit-file-move-dialog",
  templateUrl: "./deposit-file-move.dialog.html",
  styleUrls: ["./deposit-file-move.dialog.scss"],
  changeDetection: ChangeDetectionStrategy.OnPush,
})
export class DepositFileMoveDialog extends SharedAbstractContainer implements OnInit {
  isLoadingObs: Observable<boolean> = MemoizedUtil.isLoading(this._store, DepositDataFileState);

  form: FormGroup;
  formDefinition: FormComponentFormDefinition = new FormComponentFormDefinition();

  constructor(private readonly _store: Store,
              private readonly _actions$: Actions,
              private readonly _dialogRef: MatDialogRef<DepositFileMoveDialog>,
              @Inject(MAT_DIALOG_DATA) public data: DepositFileMoveDialogData,
              private readonly _fb: FormBuilder) {
    super();
  }

  ngOnInit(): void {
    super.ngOnInit();
    this.form = this._fb.group({
      [this.formDefinition.subDirectory]: [this.data.depositDataFile.relativeLocation, [Validators.required, Validators.pattern(RegexpUtil.subDirectory), SolidifyValidator]],
    });
  }

  close(): void {
    this._dialogRef.close();
  }

  onSubmit(): void {
    this._store.dispatch(new DepositDataFileAction.Move(this.data.parentResId, {
      resId: this.data.depositDataFile.resId,
      relativeLocation: this.form.get(this.formDefinition.subDirectory).value,
    }));

    this.subscribe(this._actions$.pipe(
      ofActionCompleted(DepositDataFileAction.MoveSuccess),
      take(1),
      tap((result) => {
        if (isTrue(result.result.successful)) {
          this._dialogRef.close(true);
        }
      }),
    ));
  }

  getFormControl(key: string): AbstractControl {
    return FormValidationHelper.getFormControl(this.form, key);
  }

  get formValidationHelper(): typeof FormValidationHelper {
    return FormValidationHelper;
  }
}

class FormComponentFormDefinition extends BaseFormDefinition {
  @PropertyName() subDirectory: string;
}

export interface DepositFileMoveDialogData {
  parentResId: string;
  depositDataFile: DepositDataFile;
}
