import {
  ChangeDetectionStrategy,
  Component,
  Inject,
  OnInit,
} from "@angular/core";
import {
  AbstractControl,
  FormBuilder,
  FormGroup,
  Validators,
} from "@angular/forms";
import {
  MAT_DIALOG_DATA,
  MatDialogRef,
} from "@angular/material/dialog";
import {DepositDataFile} from "@deposit/models/deposit-data-file.model";
import {DepositDataFileAction} from "@deposit/stores/data-file/deposit-data-file.action";
import {DepositDataFileState} from "@deposit/stores/data-file/deposit-data-file.state";
import {depositDataFileMetadataTypeActionNameSpace} from "@deposit/stores/data-file/metadata-type/deposit-data-file-metadata-type.action";
import {DepositDataFileMetadataTypeState} from "@deposit/stores/data-file/metadata-type/deposit-data-file-metadata-type.state";
import {Enums} from "@enums";
import {MetadataType} from "@models";
import {
  Actions,
  ofActionCompleted,
  Store,
} from "@ngxs/store";
import {SharedAbstractContainer} from "@shared/components/containers/shared-abstract/shared-abstract.container";
import {DataCategoryHelper} from "@shared/helpers/data-category.helper";
import {BaseFormDefinition} from "@shared/models/base-form-definition.model";
import {Observable} from "rxjs";
import {
  take,
  tap,
} from "rxjs/operators";
import {
  FormValidationHelper,
  isNonEmptyString,
  isNullOrUndefined,
  isTrue,
  MemoizedUtil,
  PropertyName,
  SolidifyValidator,
} from "solidify-frontend";
import {ResourceNameSpace} from "solidify-frontend/lib/stores";

@Component({
  selector: "dlcm-deposit-file-change-data-category-dialog",
  templateUrl: "./deposit-file-change-data-category.dialog.html",
  styleUrls: ["./deposit-file-change-data-category.dialog.scss"],
  changeDetection: ChangeDetectionStrategy.OnPush,
})
export class DepositFileChangeDataCategoryDialog extends SharedAbstractContainer implements OnInit {
  isLoadingObs: Observable<boolean> = MemoizedUtil.isLoading(this._store, DepositDataFileState);

  form: FormGroup;
  formDefinition: FormComponentFormDefinition = new FormComponentFormDefinition();

  metadataTypeLabelCallback: (value: MetadataType) => string = value => value.fullName;

  depositDataFileMetadataTypeActionNameSpace: ResourceNameSpace = depositDataFileMetadataTypeActionNameSpace;
  depositDataFileMetadataTypeState: typeof DepositDataFileMetadataTypeState = DepositDataFileMetadataTypeState;

  isFormValid: boolean = true;

  get dataTypeEnum(): typeof Enums.DataFile.DataTypeEnum {
    return Enums.DataFile.DataTypeEnum;
  }

  get formValidationHelper(): typeof FormValidationHelper {
    return FormValidationHelper;
  }

  constructor(private readonly _store: Store,
              private readonly _actions$: Actions,
              private readonly _dialogRef: MatDialogRef<DepositFileChangeDataCategoryDialog>,
              @Inject(MAT_DIALOG_DATA) public data: DepositFileChangeDataCategoryDialogData,
              private readonly _fb: FormBuilder) {
    super();
  }

  ngOnInit(): void {
    super.ngOnInit();
    this.form = this._fb.group({
      [this.formDefinition.dataCategory]: [this.data.depositDataFile.dataCategory, [Validators.required, SolidifyValidator]],
      [this.formDefinition.dataType]: [this.data.depositDataFile.dataType, [Validators.required, SolidifyValidator]],
      [this.formDefinition.metadataType]: [isNullOrUndefined(this.data.depositDataFile.metadataType) ? "" : this.data.depositDataFile.metadataType.resId],
    });

    this.subscribe(this.form.get(this.formDefinition.dataType).valueChanges.pipe(
      tap(value => {
        const metadataFormControl = this.form.get(this.formDefinition.metadataType);
        if (value === Enums.DataFile.DataTypeEnum.CustomMetadata) {
          metadataFormControl.setValidators([Validators.required]);
        } else {
          metadataFormControl.setValidators([]);
        }
        metadataFormControl.updateValueAndValidity();
      }),
    ));
  }

  close(): void {
    this._dialogRef.close();
  }

  onSubmit(): void {
    let metadata: MetadataType = null;
    if (!isNullOrUndefined(this.form.get(this.formDefinition.metadataType).value) && isNonEmptyString(this.form.get(this.formDefinition.metadataType).value)) {
      metadata = {
        resId: this.form.get(this.formDefinition.metadataType).value,
      } as MetadataType;
    }

    this._store.dispatch(new DepositDataFileAction.ChangeDataCategory(this.data.parentResId, {
      resId: this.data.depositDataFile.resId,
      dataCategory: this.form.get(this.formDefinition.dataCategory).value,
      dataType: this.form.get(this.formDefinition.dataType).value,
      metadataType: !isNullOrUndefined(metadata) ? metadata : undefined,
    }));

    this.subscribe(this._actions$.pipe(
      ofActionCompleted(DepositDataFileAction.ChangeDataCategorySuccess),
      take(1),
      tap((result) => {
        if (isTrue(result.result.successful)) {
          this._dialogRef.close();
        }
      }),
    ));
  }

  getFormControl(key: string): AbstractControl {
    return FormValidationHelper.getFormControl(this.form, key);
  }

  get dataCategoryHelper(): typeof DataCategoryHelper {
    return DataCategoryHelper;
  }

  getDataCategory(): Enums.DataFile.DataCategoryEnum {
    return this.form.get(this.formDefinition.dataCategory).value;
  }

  selectCategory(): void {
    this.isFormValid = false;
  }

  selectDataType(): void {
    this.isFormValid = true;
  }
}

class FormComponentFormDefinition extends BaseFormDefinition {
  @PropertyName() dataCategory: string;
  @PropertyName() dataType: string;
  @PropertyName() metadataType: string;
}

export interface DepositFileChangeDataCategoryDialogData {
  parentResId: string;
  depositDataFile: DepositDataFile;
}
