import {
  ChangeDetectionStrategy,
  Component,
  Inject,
  OnInit,
} from "@angular/core";
import {
  FormBuilder,
  FormGroup,
} from "@angular/forms";
import {
  MAT_DIALOG_DATA,
  MatDialogRef,
} from "@angular/material/dialog";
import {DepositAipAction} from "@deposit/stores/aip/deposit-aip.action";
import {DepositAipState} from "@deposit/stores/aip/deposit-aip.state";
import {Aip} from "@models";
import {Store} from "@ngxs/store";
import {SharedAbstractContainer} from "@shared/components/containers/shared-abstract/shared-abstract.container";
import {AppRoutesEnum} from "@shared/enums/routes.enum";
import {BaseFormDefinition} from "@shared/models/base-form-definition.model";
import {Observable} from "rxjs";
import {
  isEmptyArray,
  isNullOrUndefined,
  MemoizedUtil,
  PropertyName,
  QueryParameters,
} from "solidify-frontend";

@Component({
  selector: "dlcm-aip-upload-dialog",
  templateUrl: "./deposit-aip-upload.dialog.html",
  styleUrls: ["./deposit-aip-upload.dialog.scss"],
  changeDetection: ChangeDetectionStrategy.OnPush,
})
export class DepositAipUploadDialog extends SharedAbstractContainer implements OnInit {
  form: FormGroup;
  formDefinition: DepositAipUploadFormComponentFormDefinition = new DepositAipUploadFormComponentFormDefinition();

  listAipObs: Observable<Aip[]> = MemoizedUtil.list(this._store, DepositAipState);
  isLoadingObs: Observable<boolean> = MemoizedUtil.isLoading(this._store, DepositAipState);

  aipsToUpload: string[] = [];

  searchObs: Observable<string>;

  constructor(protected readonly _store: Store,
              protected readonly _dialogRef: MatDialogRef<DepositAipUploadDialog>,
              @Inject(MAT_DIALOG_DATA) public data: DepositAipUploadDialogData,
              protected _fb: FormBuilder) {
    super();
  }

  ngOnInit(): void {
    super.ngOnInit();
    this.form = this._fb.group({
      [this.formDefinition.search]: [""],
    });

    this._store.dispatch(new DepositAipAction.Clean(false, undefined));
  }

  search(searchTerm: string, queryParameters?: QueryParameters): void {
    if (searchTerm !== "") {
      const map = new Map<string, string>();
      map.set("info.name", searchTerm);
      const orgUnitId = this._store.selectSnapshot(state => state.router.state.root.children[0].children[0].params[AppRoutesEnum.paramIdOrgUnitWithoutPrefixParam]);
      map.set("info.organizationalUnitId", orgUnitId);
      const newQueryParameters = new QueryParameters();
      newQueryParameters.search = {
        searchItems: map,
      };
      this._store.dispatch(new DepositAipAction.ChangeQueryParameters(newQueryParameters, true));
    } else {
      this._store.dispatch(new DepositAipAction.Clean(false, undefined));
    }
  }

  onSubmit(): void {
    this._dialogRef.close(this.aipsToUpload);
  }

  checkIfSelected(aipId: string): boolean {
    return this.aipsToUpload.includes(aipId);
  }

  addRemoveSelection(aipId: string): void {
    if (this.aipsToUpload.includes(aipId)) {
      const index: number = this.aipsToUpload.indexOf(aipId);
      this.aipsToUpload.splice(index, 1);
    } else {
      this.aipsToUpload.push(aipId);
    }
  }

  getListAipToDisplay(list: Aip[]): Aip[] {
    const listIdAipAlreadyLinked = this.data.listIdAipAlreadyLinked;
    if (isNullOrUndefined(list) || isEmptyArray(list) || isNullOrUndefined(listIdAipAlreadyLinked) || isEmptyArray(listIdAipAlreadyLinked)) {
      return list;
    }
    return list.filter(aip => !listIdAipAlreadyLinked.includes(aip.resId));
  }
}

export class DepositAipUploadFormComponentFormDefinition extends BaseFormDefinition {
  @PropertyName() search: string;
}

export interface DepositAipUploadDialogData {
  listIdAipAlreadyLinked: string[];
}
