import {PersonRole} from "@admin/models/person-role.model";
import {
  ChangeDetectorRef,
  Component,
  Input,
  OnInit,
  ViewChild,
} from "@angular/core";
import {MatDialog} from "@angular/material/dialog";
import {ActivatedRoute} from "@angular/router";
import {AppState} from "@app/stores/app.state";
import {
  DepositAipUploadDialog,
  DepositAipUploadDialogData,
} from "@deposit/components/dialogs/deposit-aip-upload/deposit-aip-upload.dialog";
import {DepositFileUploadArchiveDialog} from "@deposit/components/dialogs/deposit-file-upload-archive/deposit-file-upload-archive.dialog";
import {
  DepositFileUploadDialog,
  DepositFileUploadDialogData,
} from "@deposit/components/dialogs/deposit-file-upload/deposit-file-upload.dialog";
import {DepositFormPresentational} from "@deposit/components/presentationals/deposit-form/deposit-form.presentational";
import {ModeDepositTabEnum} from "@deposit/enums/mode-deposit-tab.enum";
import {DepositDataFile} from "@deposit/models/deposit-data-file.model";
import {FileUploadWrapper} from "@deposit/models/file-upload-wrapper.model";
import {UploadFileStatus} from "@deposit/models/upload-file-status.model";
import {DepositCollectionAction} from "@deposit/stores/collection/deposit-collection.action";
import {DepositCollectionState} from "@deposit/stores/collection/deposit-collection.state";
import {DepositDataFileAction} from "@deposit/stores/data-file/deposit-data-file.action";
import {DepositDataFileState} from "@deposit/stores/data-file/deposit-data-file.state";
import {
  DepositAction,
  depositActionNameSpace,
} from "@deposit/stores/deposit.action";
import {
  DepositState,
  DepositStateModel,
} from "@deposit/stores/deposit.state";
import {Enums} from "@enums";
import {Deposit} from "@models";
import {
  Actions,
  ofActionCompleted,
  Select,
  Store,
} from "@ngxs/store";
import {SharedAbstractDetailEditRoutable} from "@shared/components/routables/shared-abstract-detail-edit/shared-abstract-detail-edit.routable";
import {LocalStateEnum} from "@shared/enums/local-state.enum";
import {AppRoutesEnum} from "@shared/enums/routes.enum";
import {TourEnum} from "@shared/enums/tour.enum";
import {SecurityService} from "@shared/services/security.service";
import {Observable} from "rxjs";
import {
  filter,
  tap,
} from "rxjs/operators";
import {
  AssociationRemoteState,
  isNonEmptyArray,
  isNullOrUndefined,
  MemoizedUtil,
  QueryParameters,
} from "solidify-frontend";

@Component({
  selector: "dlcm-deposit-upload-container",
  templateUrl: "./deposit-upload.container.html",
  styleUrls: ["./deposit-upload.container.scss"],
})
export class DepositUploadContainer extends SharedAbstractDetailEditRoutable<Deposit, DepositStateModel> implements OnInit {
  queryParametersObs: Observable<QueryParameters> = MemoizedUtil.queryParameters(this._store, DepositDataFileState);
  uploadStatusObs: Observable<UploadFileStatus[]> = MemoizedUtil.select(this._store, DepositState, state => state.uploadStatus);

  @Select(AppState.currentOrgUnitPerson) currentPersonObs: Observable<PersonRole>;

  @ViewChild("formPresentational")
  readonly formPresentational: DepositFormPresentational;

  readonly KEY_PARAM_NAME: keyof Deposit & string = undefined;

  get dataCategoryEnum(): typeof Enums.DataFile.DataCategoryEnum {
    return Enums.DataFile.DataCategoryEnum;
  }

  get modeDepositTabEnum(): typeof ModeDepositTabEnum {
    return ModeDepositTabEnum;
  }

  get tourEnum(): typeof TourEnum {
    return TourEnum;
  }

  @Input()
  mode: ModeDepositTabEnum;

  constructor(protected readonly _store: Store,
              protected readonly _route: ActivatedRoute,
              protected readonly _actions$: Actions,
              protected readonly _changeDetector: ChangeDetectorRef,
              protected readonly _dialog: MatDialog,
              protected readonly _securityService: SecurityService) {
    super(_store, _route, _actions$, _changeDetector, _dialog, LocalStateEnum.deposit, depositActionNameSpace);
  }

  ngOnInit(): void {
    super.ngOnInit();
    this.getCurrentModelOnParent();
  }

  protected getSubResourceWithParentId(id: string): void {
  }

  private getCurrentModelOnParent(): void {
    this._resId = this._route.snapshot.parent.paramMap.get(AppRoutesEnum.paramIdWithoutPrefixParam);
  }

  upload($event: FileUploadWrapper): void {
    this._store.dispatch(new DepositAction.UploadDataFile(this._resId, $event));
  }

  uploadArchive($event: FileUploadWrapper): void {
    this._store.dispatch(new DepositAction.UploadDataFile(this._resId, $event, true));
  }

  retry($event: UploadFileStatus): void {
    this._store.dispatch(new DepositAction.RetrySendDataFile(this._resId, $event));
  }

  cancel($event: UploadFileStatus): void {
    this._store.dispatch(new DepositAction.MarkAsCancelDataFileSending(this._resId, $event));
  }

  resume($event: DepositDataFile): void {
    this._store.dispatch(new DepositDataFileAction.Resume(this._resId, $event));
  }

  refresh(): void {
    this._store.dispatch(new DepositDataFileAction.Refresh(this._resId));
  }

  download($event: DepositDataFile): void {
    this._store.dispatch(new DepositDataFileAction.Download(this._resId, $event));
  }

  openModalUpload(dataCategoryEnum?: Enums.DataFile.DataCategoryEnum): void {
    const dialogRef = this._dialog.open(DepositFileUploadDialog, {
      width: "500px",
      data: {
        subDirectory: MemoizedUtil.selectSnapshot(this._store, DepositDataFileState, state => state.currentFolder),
        dataCategoryEnum: dataCategoryEnum,
      } as DepositFileUploadDialogData,
    });
    this.subscribe(dialogRef.afterClosed().pipe(
      filter((listFilesUploadWrapper: FileUploadWrapper[]) => isNonEmptyArray(listFilesUploadWrapper)),
      tap((listFilesUploadWrapper: FileUploadWrapper[]) => {
        listFilesUploadWrapper.forEach(f => this.upload(f));
      }),
    ));
  }

  openModalUploadArchive(): void {
    const dialogRef = this._dialog.open(DepositFileUploadArchiveDialog, {
      width: "500px",
    });
    this.subscribe(dialogRef.afterClosed().pipe(
      filter((fileUploadWrapper: FileUploadWrapper) => !isNullOrUndefined(fileUploadWrapper)),
      tap((fileUploadWrapper: FileUploadWrapper) => {
        this.uploadArchive(fileUploadWrapper);
      }),
    ));
  }

  openModalAip(): void {
    const selected = AssociationRemoteState.selectedSnapshot(this._store, DepositCollectionState);
    const dialogRef = this._dialog.open(DepositAipUploadDialog, {
      width: "90%",
      data: {
        listIdAipAlreadyLinked: selected.map(aip => aip.resId),
      } as DepositAipUploadDialogData,
    });
    this.subscribe(dialogRef.afterClosed().pipe(
      filter((listAipIds: string[]) => isNonEmptyArray(listAipIds)),
      tap(((listAipIds: string[]) => {
          this.addAip(listAipIds);
        }),
      )));
  }

  addAip(aipIds: string[]): void {
    this._store.dispatch(new DepositCollectionAction.Create(this._resId, aipIds));
    this.subscribe(this._actions$.pipe(
      ofActionCompleted(DepositCollectionAction.CreateSuccess),
      tap(() => {
        this._store.dispatch(new DepositCollectionAction.Refresh(this._resId));
      }),
    ));
  }
}
