import {PersonRole} from "@admin/models/person-role.model";
import {
  ChangeDetectorRef,
  Component,
  Inject,
  OnInit,
  ViewChild,
} from "@angular/core";
import {
  MatDialog,
  MatDialogRef,
} from "@angular/material/dialog";
import {ActivatedRoute} from "@angular/router";
import {AppState} from "@app/stores/app.state";
import {
  DepositFileUploadDialog,
  DepositFileUploadDialogData,
} from "@deposit/components/dialogs/deposit-file-upload/deposit-file-upload.dialog";
import {DepositDataFileHelper} from "@deposit/helpers/deposit-data-file.helper";
import {DepositDataFile} from "@deposit/models/deposit-data-file.model";
import {FileUploadWrapper} from "@deposit/models/file-upload-wrapper.model";
import {DepositDataFileAction} from "@deposit/stores/data-file/deposit-data-file.action";
import {DepositDataFileState} from "@deposit/stores/data-file/deposit-data-file.state";
import {
  DepositAction,
  depositActionNameSpace,
} from "@deposit/stores/deposit.action";
import {
  DepositState,
  DepositStateModel,
} from "@deposit/stores/deposit.state";
import {DepositOrganizationalUnitState} from "@deposit/stores/organizational-unit/deposit-organizational-unit.state";
import {Enums} from "@enums";
import {Deposit} from "@models";
import {Navigate} from "@ngxs/router-plugin";
import {
  Actions,
  Select,
  Store,
} from "@ngxs/store";
import {SharedFileAndAipInformationContainer} from "@shared/components/containers/shared-file-and-aip-information/shared-file-and-aip-information.container";
import {
  SharedDeleteDialog,
  SharedDeleteDialogData,
} from "@shared/components/dialogs/shared-abstract-delete/shared-delete.dialog";
import {
  SharedConfirmDialog,
  SharedConfirmDialogData,
} from "@shared/components/dialogs/shared-confirm/shared-confirm.dialog";
import {SharedPreviewDialogData} from "@shared/components/dialogs/shared-preview/shared-preview-dialog-data.model";
import {SharedPreviewDialog} from "@shared/components/dialogs/shared-preview/shared-preview.dialog";
import {SharedDataTablePresentational} from "@shared/components/presentationals/shared-data-table/shared-data-table.presentational";
import {SharedAbstractDetailEditRoutable} from "@shared/components/routables/shared-abstract-detail-edit/shared-abstract-detail-edit.routable";
import {DataTableComponentEnum} from "@shared/enums/data-table-component.enum";
import {FieldTypeEnum} from "@shared/enums/field-type.enum";
import {FileViewModeEnum} from "@shared/enums/file-view-mode.enum";
import {IconNameEnum} from "@shared/enums/icon-name.enum";
import {LabelTranslateEnum} from "@shared/enums/label-translate.enum";
import {LocalStateEnum} from "@shared/enums/local-state.enum";
import {
  AppRoutesEnum,
  DepositRoutesEnum,
} from "@shared/enums/routes.enum";
import {FileVisualizerHelper} from "@shared/filevisualizer/helpers/file-visualizer.helper";
import {
  AbstractFileVisualizer,
  FILE_VISUALIZERS,
} from "@shared/filevisualizer/services/abstract-file-visualizer.service";
import {DataTableActions} from "@shared/models/data-table-actions.model";
import {DataTableBulkActions} from "@shared/models/data-table-bulk-actions.model";
import {DataTableColumns} from "@shared/models/data-table-columns.model";
import {SecurityService} from "@shared/services/security.service";
import {Observable} from "rxjs";
import {
  distinctUntilChanged,
  filter,
  map,
  take,
  tap,
} from "rxjs/operators";
import {
  isNonEmptyArray,
  isNullOrUndefined,
  isTrue,
  isUndefined,
  MappingObjectUtil,
  MARK_AS_TRANSLATABLE,
  MemoizedUtil,
  OrderEnum,
  QueryParameters,
  QueryParametersUtil,
  ResourceActionHelper,
  StringUtil,
} from "solidify-frontend";

@Component({
  selector: "dlcm-deposit-file-container",
  templateUrl: "./deposit-file.container.html",
  styleUrls: ["./deposit-file.container.scss"],
})
export class DepositFileContainer extends SharedAbstractDetailEditRoutable<Deposit, DepositStateModel> implements OnInit {
  private readonly _KEY_QUERY_PARAMETERS: keyof DepositDataFile = "status";
  isLoadingDataFileObs: Observable<boolean> = MemoizedUtil.isLoading(this._store, DepositDataFileState);
  listDataFileObs: Observable<DepositDataFile[]> = MemoizedUtil.list(this._store, DepositDataFileState);
  queryParametersObs: Observable<QueryParameters> = MemoizedUtil.queryParameters(this._store, DepositDataFileState).pipe(
    tap(queryParameters => this.isInErrorStatusFilter = MappingObjectUtil.get(QueryParametersUtil.getSearchItems(queryParameters), this._KEY_QUERY_PARAMETERS) === Enums.DataFile.StatusEnum.IN_ERROR),
  );
  draggingDepositDataFile: DepositDataFile | undefined = undefined;
  intermediateFoldersObs: Observable<string[]> = MemoizedUtil.select(this._store, DepositDataFileState, state => state.intermediateFolders);
  foldersWithIntermediateFoldersObs: Observable<string[]> = MemoizedUtil.select(this._store, DepositDataFileState, state => state.foldersWithIntermediateFolders);
  currentFolderObs: Observable<string> = MemoizedUtil.select(this._store, DepositDataFileState, state => state.currentFolder);
  canEditObs: Observable<boolean> = MemoizedUtil.select(this._store, DepositState, state => state.canEdit).pipe(
    distinctUntilChanged(),
    tap(canEdit => this.actions = [...this.actions]), // Force method computeContext on datatable
  );

  @Select(AppState.currentOrgUnitPerson) currentPersonObs: Observable<PersonRole>;

  @ViewChild("dataTablePresentational")
  readonly dataTablePresentational: SharedDataTablePresentational<DepositDataFile>;

  currentFileViewMode: FileViewModeEnum = FileViewModeEnum.FolderView;

  columns: DataTableColumns<DepositDataFile>[];

  actions: DataTableActions<DepositDataFile>[] = [
    {
      logo: IconNameEnum.notIgnore,
      callback: (depositDataFile: DepositDataFile) => this.validate(this._resId, depositDataFile),
      placeholder: current => LabelTranslateEnum.doNotIgnore,
      displayOnCondition: (depositDataFile: DepositDataFile) => depositDataFile.status === Enums.DataFile.StatusEnum.IGNORED_FILE,
    },
    {
      logo: IconNameEnum.download,
      callback: (depositDataFile: DepositDataFile) => this.downloadDataFile(this._resId, depositDataFile),
      placeholder: current => LabelTranslateEnum.download,
      displayOnCondition: current => current.status === Enums.DataFile.StatusEnum.PROCESSED || current.status === Enums.DataFile.StatusEnum.READY || current.status === Enums.DataFile.StatusEnum.VIRUS_CHECKED,
      isWrapped: false,
    },
    {
      logo: IconNameEnum.resume,
      callback: (depositDataFile: DepositDataFile) => this.resumeDataFile(this._resId, depositDataFile),
      placeholder: current => LabelTranslateEnum.resume,
      displayOnCondition: (depositDataFile: DepositDataFile) => depositDataFile.status === Enums.DataFile.StatusEnum.IN_ERROR,
    },
    {
      logo: IconNameEnum.delete,
      callback: (depositDataFile: DepositDataFile) => this._deleteDataFile(this._resId, depositDataFile),
      placeholder: current => LabelTranslateEnum.delete,
      displayOnCondition: (depositDataFile: DepositDataFile) => MemoizedUtil.selectSnapshot(this._store, DepositState, state => state.canEdit),
    },
    {
      logo: IconNameEnum.preview,
      callback: (depositDataFile: DepositDataFile) => this._showPreview(depositDataFile),
      placeholder: current => LabelTranslateEnum.showPreview,
      displayOnCondition: (depositDataFile: DepositDataFile) => FileVisualizerHelper.canHandle(this._fileVisualizers, {
        dataFile: depositDataFile,
        fileExtension: FileVisualizerHelper.getFileExtension(depositDataFile.fileName),
      }),
    },
  ];

  bulkActions: DataTableBulkActions<DepositDataFile>[] = [
    {
      logo: IconNameEnum.delete,
      callback: (list: DepositDataFile[]) => this._bulkDeleteDataFile(this._resId, list.map(s => s.resId)),
      label: LabelTranslateEnum.deleteSelection,
      displayOnCondition: (list: DepositDataFile[]) => this.canEditObs,
    },
    {
      logo: IconNameEnum.resume,
      callback: (list: DepositDataFile[]) => this._bulkResumeDataFile(this._resId, list.map(s => s.resId)),
      label: LabelTranslateEnum.resumeSelection,
      displayOnCondition: (list: DepositDataFile[]) => this.canEditObs && list.findIndex(d => d.status === Enums.DataFile.StatusEnum.IN_ERROR) !== -1,
    },
    {
      logo: IconNameEnum.notIgnore,
      callback: (list: DepositDataFile[]) => this._bulkValidateDataFile(this._resId, list.map(s => s.resId)),
      label: LabelTranslateEnum.doNotIgnoreSelection,
      displayOnCondition: (list: DepositDataFile[]) => this.canEditObs && list.findIndex(d => d.status === Enums.DataFile.StatusEnum.IGNORED_FILE) !== -1,
    },
  ];

  columnsToSkippedFilter: keyof DepositDataFile[] | string[] = [
    DepositDataFileHelper.RELATIVE_LOCATION,
    "status",
  ];

  dialogDataFileRef: MatDialogRef<SharedFileAndAipInformationContainer<DepositDataFile>>;

  get fileViewModeEnum(): typeof FileViewModeEnum {
    return FileViewModeEnum;
  }

  get depositEnum(): typeof Enums.Deposit.StatusEnum {
    return Enums.Deposit.StatusEnum;
  }

  readonly KEY_PARAM_NAME: keyof Deposit & string = undefined;
  isInErrorStatusFilter: boolean = false;

  constructor(protected readonly _store: Store,
              protected readonly _route: ActivatedRoute,
              protected readonly _actions$: Actions,
              protected readonly _changeDetector: ChangeDetectorRef,
              protected readonly _dialog: MatDialog,
              protected readonly _securityService: SecurityService,
              @Inject(FILE_VISUALIZERS) private readonly _fileVisualizers: AbstractFileVisualizer[]) {
    super(_store, _route, _actions$, _changeDetector, _dialog, LocalStateEnum.deposit, depositActionNameSpace);

    this.columns = [
      {
        field: "fileName",
        header: LabelTranslateEnum.fileName,
        type: FieldTypeEnum.string,
        order: OrderEnum.ascending,
        filterableField: "dataFile.sourceData" as any,
        sortableField: "dataFile.sourceData" as any,
        isSortable: true,
        isFilterable: true,
      },
      {
        field: "creation.when" as any,
        header: LabelTranslateEnum.created,
        type: FieldTypeEnum.datetime,
        order: OrderEnum.none,
        isFilterable: false,
        isSortable: true,
      },
      {
        field: "status",
        header: StringUtil.stringEmpty,
        type: FieldTypeEnum.singleSelect,
        order: OrderEnum.none,
        sortableField: "dataFile.status" as any,
        filterableField: "dataFile.status" as any,
        isSortable: false,
        isFilterable: false,
        translate: false,
        component: DataTableComponentEnum.dataFileQuickStatus,
        width: "35px",
        // filterEnum: ComplianceLevelEnumHelper.getListKeyValue(),
      },
      {
        field: "status",
        header: LabelTranslateEnum.status,
        type: FieldTypeEnum.singleSelect,
        order: OrderEnum.none,
        sortableField: "dataFile.status" as any,
        isSortable: true,
        isFilterable: true,
        translate: true,
        tooltip: (value) => Enums.DataFile.getExplanation(value as Enums.DataFile.StatusEnum),
        filterEnum: Enums.DataFile.StatusEnumTranslate,
        component: DataTableComponentEnum.status,
      },
      {
        field: "complianceLevel",
        header: LabelTranslateEnum.complianceLevel,
        type: FieldTypeEnum.singleSelect,
        order: OrderEnum.none,
        sortableField: "dataFile.complianceLevel" as any,
        filterableField: "dataFile.complianceLevel" as any,
        isSortable: true,
        isFilterable: false,
        translate: true,
        width: "185px",
        component: DataTableComponentEnum.conformityLevelStar,
        filterEnum: Enums.DataFile.ComplianceLevelEnumTranslate,
      },
      {
        field: "smartSize",
        header: LabelTranslateEnum.size,
        type: FieldTypeEnum.string,
        order: OrderEnum.none,
        sortableField: "dataFile.fileSize" as any,
        filterableField: "dataFile.fileSize" as any,
        isSortable: true,
        isFilterable: false,
        translate: true,
        width: "100px",
      },
    ];
  }

  ngOnInit(): void {
    super.ngOnInit();
    this.getCurrentModelOnParent();
  }

  private getCurrentModelOnParent(): void {
    this._resId = this._route.snapshot.parent.paramMap.get(AppRoutesEnum.paramIdWithoutPrefixParam);
    this.getDepositById(this._resId);
  }

  private getDepositById(id: string): void {
    const depositInState = MemoizedUtil.currentSnapshot(this._store, DepositState);
    if (isNullOrUndefined(depositInState) || id !== depositInState.resId) {
      this._store.dispatch(ResourceActionHelper.getById(depositActionNameSpace, id, true));
    }
    this.getSubResourceWithParentId(id);
  }

  getSubResourceWithParentId(id: string): void {
    this._store.dispatch(new DepositDataFileAction.GetAll(id, undefined, true));
    this._store.dispatch(new DepositDataFileAction.GetListFolder(id));
  }

  onQueryParametersEvent(queryParameters: QueryParameters, withRefresh: boolean = true): void {
    this._store.dispatch(new DepositDataFileAction.ChangeQueryParameters(this._resId, queryParameters, true, withRefresh));
    // this._changeDetector.detectChanges(); // Allow to display spinner the first time
  }

  refresh(): void {
    this._store.dispatch(new DepositDataFileAction.Refresh(this._resId));
  }

  download($event: DepositDataFile): void {
    this._store.dispatch(new DepositDataFileAction.Download(this._resId, $event));
  }

  selectFolder(folderFullName: string | undefined): void {
    if (isUndefined(folderFullName)) {
      this.currentFileViewMode = FileViewModeEnum.FlatView;
    } else {
      this.currentFileViewMode = FileViewModeEnum.FolderView;
    }
    this._store.dispatch(new DepositDataFileAction.ChangeCurrentFolder(folderFullName, true));
  }

  downloadFolder(folderName: string): void {
    this._store.dispatch(new DepositAction.Download(this._resId, folderName));
  }

  deleteFolder(folderName: string): void {
    this.subscribe(this._dialog.open(SharedDeleteDialog, {
      minWidth: "500px",
      data: {
        name: folderName,
        resourceNameSpace: depositActionNameSpace,
        message: MARK_AS_TRANSLATABLE("deposit.dialog.deleteFolder.message"),
      } as SharedDeleteDialogData,
    }).afterClosed().pipe(
      tap((isConfirmed: boolean | undefined) => {
        if (isNullOrUndefined(isConfirmed)) {
          return;
        }
        if (isTrue(isConfirmed)) {
          this._store.dispatch(new DepositDataFileAction.DeleteFolder(this._resId, folderName));
        }
      }),
    ));
  }

  showDetail(depositDataFile: DepositDataFile): void {
    const orgUnitId = MemoizedUtil.currentSnapshot(this._store, DepositOrganizationalUnitState)?.resId;
    this._store.dispatch(new Navigate([AppRoutesEnum.deposit, orgUnitId, DepositRoutesEnum.detail, this._resId, DepositRoutesEnum.files, depositDataFile.resId]));
  }

  private _bulkDeleteDataFile(parentId: string, listId: string[]): Observable<boolean> {
    return this._dialog.open(SharedDeleteDialog, {
      minWidth: "500px",
      data: {
        name: undefined,
        resourceNameSpace: depositActionNameSpace,
        message: MARK_AS_TRANSLATABLE("deposit.dialog.deleteList.message"),
      } as SharedDeleteDialogData,
    }).afterClosed().pipe(
      take(1),
      map((isConfirmed: boolean | undefined) => {
        if (isNullOrUndefined(isConfirmed)) {
          return false;
        }
        if (isTrue(isConfirmed)) {
          this._bulkDeleteDataFileWithoutConfirmation(parentId, listId);
        }
        return true;
      }),
    );
  }

  private _bulkDeleteDataFileWithoutConfirmation(parentId: string, listId: string[]): void {
    this._store.dispatch(new DepositDataFileAction.DeleteAll(parentId, listId));
  }

  private _bulkResumeDataFile(parentId: string, listId: string[]): void {
    this._store.dispatch(new DepositDataFileAction.ResumeAll(parentId, listId));
  }

  private _bulkValidateDataFile(parentId: string, listId: string[]): void {
    this._store.dispatch(new DepositDataFileAction.ValidateAll(parentId, listId));
  }

  private _deleteDataFile(parentId: string, dataFile: DepositDataFile): void {
    this._store.dispatch(new DepositDataFileAction.Delete(parentId, dataFile.resId));
  }

  private _showPreview(dataFile: DepositDataFile): void {
    this._dialog.open(SharedPreviewDialog, {
      width: "max-content",
      maxWidth: "90vw",
      height: "min-content",
      data: {
        fileInput: {
          dataFile: dataFile,
        },

      } as SharedPreviewDialogData,
    });
  }

  moveDataFile(dataFile: DepositDataFile): void {
    this._store.dispatch(new DepositDataFileAction.Move(this._resId, {
      resId: dataFile.resId,
      relativeLocation: dataFile.relativeLocation,
    }));
  }

  validate(parentId: string, dataFile: DepositDataFile): void {
    this._store.dispatch(new DepositDataFileAction.Validate(parentId, dataFile));
  }

  downloadDataFile(parentId: string, dataFile: DepositDataFile): void {
    this._store.dispatch(new DepositDataFileAction.Download(parentId, dataFile));
  }

  resumeDataFile(parentId: string, dataFile: DepositDataFile): void {
    this._store.dispatch(new DepositDataFileAction.Resume(parentId, dataFile));
  }

  selectAndDelete(): void {
    this.subscribe(this._dialog.open(SharedConfirmDialog, {
      data: {
        titleToTranslate: MARK_AS_TRANSLATABLE("deposit.file.bulkDelete.title"),
        messageToTranslate: MARK_AS_TRANSLATABLE("deposit.file.bulkDelete.message"),
        confirmButtonToTranslate: LabelTranslateEnum.yesImSure,
        cancelButtonToTranslate: LabelTranslateEnum.cancel,
        colorConfirm: "warn",
      } as SharedConfirmDialogData,
    }).afterClosed().pipe(
      tap((isConfirmed: boolean | undefined) => {
        if (isNullOrUndefined(isConfirmed)) {
          return;
        }
        if (isTrue(isConfirmed)) {
          this.subscribe(this.dataTablePresentational.selectAllResult().pipe(
            take(1),
            tap(selection => {
              this._bulkDeleteDataFileWithoutConfirmation(this._resId, selection);
              this.dataTablePresentational.cleanSelection();
            }),
          ));
        }
      }),
    ));
  }

  uploadInFolder(fullFolderPath: string): void {
    const dialogRef = this._dialog.open(DepositFileUploadDialog, {
      width: "500px",
      data: {
        subDirectory: fullFolderPath,
      } as DepositFileUploadDialogData,
    });
    this.subscribe(dialogRef.afterClosed().pipe(
      filter((listFilesUploadWrapper: FileUploadWrapper[]) => isNonEmptyArray(listFilesUploadWrapper)),
      tap((listFilesUploadWrapper: FileUploadWrapper[]) => {
        listFilesUploadWrapper.forEach(f => this.upload(f));
      }),
    ));
  }

  upload($event: FileUploadWrapper): void {
    this._store.dispatch(new DepositAction.UploadDataFile(this._resId, $event));
  }

  selectAndResumeAll(): void {
    this.subscribe(this.dataTablePresentational.selectAllResult().pipe(
      take(1),
      tap(selectionList => {
        this._bulkResumeDataFile(this._resId, selectionList);
        this.dataTablePresentational.cleanSelection();
      }),
    ));
  }

  dragStart($event: DepositDataFile): void {
    this.draggingDepositDataFile = $event;
  }

  dragEnd($event: DepositDataFile): void {
    this.draggingDepositDataFile = undefined;
  }
}
