import {
  ChangeDetectorRef,
  Component,
  OnInit,
} from "@angular/core";
import {
  MatDialog,
  MatDialogRef,
} from "@angular/material/dialog";
import {ActivatedRoute} from "@angular/router";
import {DepositCollectionAction} from "@deposit/stores/collection/deposit-collection.action";
import {DepositCollectionState} from "@deposit/stores/collection/deposit-collection.state";
import {depositActionNameSpace} from "@deposit/stores/deposit.action";
import {
  DepositState,
  DepositStateModel,
} from "@deposit/stores/deposit.state";
import {DepositOrganizationalUnitState} from "@deposit/stores/organizational-unit/deposit-organizational-unit.state";
import {Enums} from "@enums";
import {
  Aip,
  Deposit,
} from "@models";
import {Navigate} from "@ngxs/router-plugin";
import {
  Actions,
  ofActionCompleted,
  Store,
} from "@ngxs/store";
import {SharedFileAndAipInformationContainer} from "@shared/components/containers/shared-file-and-aip-information/shared-file-and-aip-information.container";
import {SharedAbstractDetailEditRoutable} from "@shared/components/routables/shared-abstract-detail-edit/shared-abstract-detail-edit.routable";
import {DataTableComponentEnum} from "@shared/enums/data-table-component.enum";
import {FieldTypeEnum} from "@shared/enums/field-type.enum";
import {IconNameEnum} from "@shared/enums/icon-name.enum";
import {LabelTranslateEnum} from "@shared/enums/label-translate.enum";
import {LocalStateEnum} from "@shared/enums/local-state.enum";
import {
  AppRoutesEnum,
  DepositRoutesEnum,
} from "@shared/enums/routes.enum";
import {DataTableActions} from "@shared/models/data-table-actions.model";
import {DataTableColumns} from "@shared/models/data-table-columns.model";
import {SecurityService} from "@shared/services/security.service";
import {Observable} from "rxjs";
import {tap} from "rxjs/operators";
import {
  AssociationRemoteState,
  isNullOrUndefined,
  MemoizedUtil,
  OrderEnum,
  QueryParameters,
  ResourceActionHelper,
} from "solidify-frontend";

@Component({
  selector: "dlcm-deposit-collection-container",
  templateUrl: "./deposit-collection.container.html",
  styleUrls: ["./deposit-collection.container.scss"],
})
export class DepositCollectionContainer extends SharedAbstractDetailEditRoutable<Deposit, DepositStateModel> implements OnInit {
  readonly KEY_PARAM_NAME: keyof Deposit & string = undefined;

  isLoadingCollectionObs: Observable<boolean> = MemoizedUtil.isLoading(this._store, DepositCollectionState);
  listCollectionObs: Observable<Aip[]> = AssociationRemoteState.selected(this._store, DepositCollectionState);
  queryParametersObs: Observable<QueryParameters> = AssociationRemoteState.queryParameters(this._store, DepositCollectionState);

  canEditObs: Observable<boolean> = MemoizedUtil.select(this._store, DepositState, state => state.canEdit);

  dialogCollectionFileRef: MatDialogRef<SharedFileAndAipInformationContainer<Aip>>;

  columns: DataTableColumns<Aip>[];
  actions: DataTableActions<Aip>[] = [
    {
      logo: IconNameEnum.delete,
      callback: (aip: Aip) => this.deleteAip(this._resId, aip),
      placeholder: current => LabelTranslateEnum.delete,
      displayOnCondition: (aip: Aip) => MemoizedUtil.selectSnapshot(this._store, DepositState, state => state.canEdit),
    },
    {
      logo: IconNameEnum.internalLink,
      callback: (aip: Aip) => this.goToAip(aip),
      placeholder: current => LabelTranslateEnum.goToAip,
      displayOnCondition: (aip: Aip) => this._securityService.isRootOrAdmin(),
    },
  ];

  constructor(protected readonly _store: Store,
              protected readonly _route: ActivatedRoute,
              protected readonly _actions$: Actions,
              protected readonly _changeDetector: ChangeDetectorRef,
              protected readonly _dialog: MatDialog,
              protected readonly _securityService: SecurityService) {
    super(_store, _route, _actions$, _changeDetector, _dialog, LocalStateEnum.deposit, depositActionNameSpace);

    this.columns = [
      {
        field: "info.name" as any,
        header: LabelTranslateEnum.nameLabel,
        type: FieldTypeEnum.string,
        order: OrderEnum.none,
        isSortable: false,
        isFilterable: false,
      },
      {
        field: "info.dataSensitivity" as any,
        header: LabelTranslateEnum.sensitivity,
        type: FieldTypeEnum.singleSelect,
        order: OrderEnum.none,
        isSortable: false,
        isFilterable: false,
        translate: true,
        filterEnum: Enums.Deposit.DataSensitivityEnumTranslate,
        component: DataTableComponentEnum.sensitivity,
      },
      {
        field: "info.access" as any,
        header: LabelTranslateEnum.accessLevel,
        type: FieldTypeEnum.singleSelect,
        order: OrderEnum.none,
        isSortable: false,
        isFilterable: false,
        translate: true,
        filterEnum: Enums.Deposit.AccessEnumTranslate,
        component: DataTableComponentEnum.accessLevel,
      },
      {
        field: "smartSize",
        header: LabelTranslateEnum.size,
        type: FieldTypeEnum.string,
        order: OrderEnum.none,
        filterableField: "archiveSize",
        isFilterable: false,
        isSortable: false,
        width: "100px",
      },
      {
        field: "archiveFileNumber",
        header: LabelTranslateEnum.files,
        type: FieldTypeEnum.string,
        order: OrderEnum.none,
        filterableField: "archiveFileNumber",
        isFilterable: false,
        isSortable: false,
        width: "80px",
      },
      {
        field: "info.status" as any,
        header: LabelTranslateEnum.status,
        type: FieldTypeEnum.singleSelect,
        order: OrderEnum.none,
        isSortable: false,
        isFilterable: false,
        translate: true,
        filterEnum: Enums.Package.StatusEnumTranslate,
        component: DataTableComponentEnum.status,
      },

      {
        field: "info.complianceLevel" as any,
        header: LabelTranslateEnum.complianceLevel,
        type: FieldTypeEnum.singleSelect,
        order: OrderEnum.none,
        isSortable: false,
        isFilterable: false,
        translate: true,
        component: DataTableComponentEnum.conformityLevelStar,
        filterEnum: Enums.DataFile.ComplianceLevelEnumTranslate,
      },
    ];
  }

  ngOnInit(): void {
    super.ngOnInit();
    this.getCurrentModelOnParent();
  }

  private getCurrentModelOnParent(): void {
    this._resId = this._route.snapshot.parent.paramMap.get(AppRoutesEnum.paramIdWithoutPrefixParam);
    this.getDepositById(this._resId);
  }

  private getDepositById(id: string): void {
    const depositInState = MemoizedUtil.currentSnapshot(this._store, DepositState);
    if (isNullOrUndefined(depositInState) || id !== depositInState.resId) {
      this._store.dispatch(ResourceActionHelper.getById(depositActionNameSpace, id, true));
    }
    this.getSubResourceWithParentId(id);
  }

  getSubResourceWithParentId(id: string): void {
    this._store.dispatch(new DepositCollectionAction.GetAll(id, null, true));
  }

  refresh(): void {
    this._store.dispatch(new DepositCollectionAction.Refresh(this._resId));
  }

  onQueryParametersEvent(queryParameters: QueryParameters): void {
    this._store.dispatch(new DepositCollectionAction.ChangeQueryParameters(this._resId, queryParameters, true));
  }

  showDetail(aip: Aip): void {
    const orgUnitId = MemoizedUtil.currentSnapshot(this._store, DepositOrganizationalUnitState)?.resId;
    this._store.dispatch(new Navigate([AppRoutesEnum.deposit, orgUnitId, DepositRoutesEnum.detail, this._resId, DepositRoutesEnum.collections, aip.resId]));
  }

  private goToAip(aip: Aip): void {
    this.subscribe(this._actions$.pipe(
      ofActionCompleted(DepositCollectionAction.GoToAip),
      tap(result => {
        if (!result.result.successful) {
          return;
        }
        if (this.dialogCollectionFileRef) {
          this.dialogCollectionFileRef.close();
        }
      }),
    ));
    this._store.dispatch(new DepositCollectionAction.GoToAip(aip));
  }

  deleteAip(_resId: string, aip: Aip): void {
    this.subscribe(this._actions$.pipe(
      ofActionCompleted(DepositCollectionAction.DeleteSuccess),
      tap((result) => {
        if (!result.result.successful) {
          return;
        }
        if (this.dialogCollectionFileRef) {
          this.dialogCollectionFileRef.close();
        }
        this._store.dispatch(new DepositCollectionAction.Refresh(this._resId));
      }),
    ));
    this._store.dispatch(new DepositCollectionAction.Delete(this._resId, aip.resId));
  }
}
