import {
  AdminUserAction,
  adminUserActionNameSpace,
} from "@admin/user/stores/admin-user.action";
import {HttpClient} from "@angular/common/http";
import {Injectable} from "@angular/core";
import {User} from "@models";
import {
  Action,
  Actions,
  ofActionCompleted,
  Selector,
  State,
  StateContext,
  Store,
} from "@ngxs/store";
import {AdminResourceApiEnum} from "@shared/enums/api.enum";
import {LocalStateEnum} from "@shared/enums/local-state.enum";
import {RoutesEnum} from "@shared/enums/routes.enum";
import {SharedPersonAction} from "@shared/stores/person/shared-person.action";
import {ResourceLogoStateModel} from "@shared/stores/resource-logo/resource-logo-state.model";
import {
  defaultResourceLogoStateInitValue,
  ResourceLogoState,
  ResourceLogoStateModeEnum,
} from "@shared/stores/resource-logo/resource-logo.state";
import {Observable} from "rxjs";
import {tap} from "rxjs/operators";
import {
  ApiService,
  isNotNullNorUndefined,
  isNullOrUndefined,
  MARK_AS_TRANSLATABLE,
  NotificationService,
  OverrideDefaultAction,
  StoreUtil,
  urlSeparator,
} from "solidify-frontend";

export interface AdminUserStateModel extends ResourceLogoStateModel<User> {
}

@Injectable()
@State<AdminUserStateModel>({
  name: LocalStateEnum.admin_user,
  defaults: {
    ...defaultResourceLogoStateInitValue(),
  },
})
export class AdminUserState extends ResourceLogoState<AdminUserStateModel, User> {
  constructor(protected apiService: ApiService,
              protected store: Store,
              protected notificationService: NotificationService,
              protected actions$: Actions,
              protected readonly _httpClient: HttpClient) {
    super(apiService, store, notificationService, actions$, {
      nameSpace: adminUserActionNameSpace,
      routeRedirectUrlAfterSuccessCreateAction: (resId: string) => RoutesEnum.adminUserDetail + urlSeparator + resId,
      routeRedirectUrlAfterSuccessUpdateAction: (resId: string) => RoutesEnum.adminUserDetail + urlSeparator + resId,
      routeRedirectUrlAfterSuccessDeleteAction: RoutesEnum.adminUser,
      notificationResourceCreateSuccessTextToTranslate: MARK_AS_TRANSLATABLE("admin.user.notification.resource.create"),
      notificationResourceDeleteSuccessTextToTranslate: MARK_AS_TRANSLATABLE("admin.user.notification.resource.delete"),
      notificationResourceUpdateSuccessTextToTranslate: MARK_AS_TRANSLATABLE("admin.user.notification.resource.update"),
    }, _httpClient, ResourceLogoStateModeEnum.avatar);
  }

  protected get _urlResource(): string {
    return AdminResourceApiEnum.users;
  }

  protected get _urlLogoResource(): string {
    return AdminResourceApiEnum.people;
  }

  @Selector()
  static isLoading(state: AdminUserStateModel): boolean {
    return StoreUtil.isLoadingState(state);
  }

  @Selector()
  static isLoadingWithDependency(state: AdminUserStateModel): boolean {
    return this.isLoading(state);
  }

  @Selector()
  static currentTitle(state: AdminUserStateModel): string | undefined {
    if (isNullOrUndefined(state.current)) {
      return undefined;
    }
    return state.current.externalUid;
  }

  @Selector()
  static isReadyToBeDisplayed(state: AdminUserStateModel): boolean {
    return this.isReadyToBeDisplayedInCreateMode
      && !isNullOrUndefined(state.current);
  }

  @Selector()
  static isReadyToBeDisplayedInCreateMode(state: AdminUserStateModel): boolean {
    return true;
  }

  @Action(AdminUserAction.LoadResource)
  loadResource(ctx: StateContext<AdminUserStateModel>, action: AdminUserAction.LoadResource): Observable<any> {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
    });

    return StoreUtil.dispatchParallelActionAndWaitForSubActionsCompletion(ctx, [
      {
        action: new SharedPersonAction.GetAll(undefined, false, false),
        subActionCompletions: [
          this.actions$.pipe(ofActionCompleted(SharedPersonAction.GetAllSuccess)),
          this.actions$.pipe(ofActionCompleted(SharedPersonAction.GetAllFail)),
        ],
      },
    ]).pipe(
      tap(success => {
        if (success) {
          ctx.dispatch(new AdminUserAction.LoadResourceSuccess(action));
        } else {
          ctx.dispatch(new AdminUserAction.LoadResourceFail(action));
        }
      }),
    );
  }

  @OverrideDefaultAction()
  @Action(AdminUserAction.GetByIdSuccess)
  getByIdSuccess(ctx: StateContext<AdminUserStateModel>, action: AdminUserAction.GetByIdSuccess): void {
    super.getByIdSuccess(ctx, action);
    if (isNotNullNorUndefined(action.model.avatar)) {
      ctx.dispatch(new AdminUserAction.GetPhoto(action.model.person.resId));
    }
  }
}
