import {AdminUserAction} from "@admin/user/stores/admin-user.action";
import {AdminUserState} from "@admin/user/stores/admin-user.state";
import {
  ChangeDetectionStrategy,
  ChangeDetectorRef,
  Component,
  ElementRef,
  Input,
} from "@angular/core";
import {
  FormBuilder,
  Validators,
} from "@angular/forms";
import {Enums} from "@enums";
import {
  AccessOrganizationalUnit,
  Person,
  User,
} from "@models";
import {SharedAbstractFormPresentational} from "@shared/components/presentationals/shared-abstract-form/shared-abstract-form.presentational";
import {RoutesEnum} from "@shared/enums/routes.enum";
import {BaseFormDefinition} from "@shared/models/base-form-definition.model";
import {ResourceLogoNameSpace} from "@shared/stores/resource-logo/resource-logo-namespace.model";
import {
  DateUtil,
  isEmptyString,
  isNotNullNorUndefined,
  isNullOrUndefined,
  KeyValue,
  PropertyName,
  SolidifyValidator,
} from "solidify-frontend";

@Component({
  selector: "dlcm-admin-user-form",
  templateUrl: "./admin-user-form.presentational.html",
  styleUrls: ["./admin-user-form.presentational.scss"],
  changeDetection: ChangeDetectionStrategy.OnPush,
})
export class AdminUserFormPresentational extends SharedAbstractFormPresentational<User> {
  formDefinition: FormComponentFormDefinition = new FormComponentFormDefinition();
  applicationRolesNames: KeyValue[] = Enums.UserApplicationRole.UserApplicationRoleEnumTranslate;

  userAvatarActionNameSpace: ResourceLogoNameSpace = AdminUserAction;
  userState: typeof AdminUserState = AdminUserState;

  @Input()
  listPersons: Person[];

  @Input()
  listOrganizationalUnit: AccessOrganizationalUnit[];

  @Input()
  currentUser: User;

  constructor(protected readonly _changeDetectorRef: ChangeDetectorRef,
              protected readonly _elementRef: ElementRef,
              private readonly _fb: FormBuilder) {
    super(_changeDetectorRef, _elementRef);
  }

  protected bindFormTo(user: User): void {
    this.form = this._fb.group({
      [this.formDefinition.externalUid]: [user.externalUid, [Validators.required, SolidifyValidator]],
      [this.formDefinition.firstName]: [user.firstName, [Validators.required, SolidifyValidator]],
      [this.formDefinition.lastName]: [user.lastName, [Validators.required, SolidifyValidator]],
      [this.formDefinition.homeOrganization]: [user.homeOrganization, [Validators.required, SolidifyValidator]],
      [this.formDefinition.email]: [user.email, [Validators.required, SolidifyValidator]],
      [this.formDefinition.person]: isNullOrUndefined(user.person) ? [] : [user.person.resId],
      [this.formDefinition.applicationRole]: [user.applicationRole.resId, [Validators.required, SolidifyValidator]],
      [this.formDefinition.lastLoginIpAddress]: [user.lastLoginIpAddress, [SolidifyValidator]],
      [this.formDefinition.lastLoginTime]: [DateUtil.convertDateToDateTimeString(new Date(user.lastLoginTime)), [SolidifyValidator]],
    });
  }

  protected initNewForm(): void {
    this.form = this._fb.group({
      [this.formDefinition.externalUid]: ["", [Validators.required, SolidifyValidator]],
      [this.formDefinition.firstName]: ["", [Validators.required, SolidifyValidator]],
      [this.formDefinition.lastName]: ["", [Validators.required, SolidifyValidator]],
      [this.formDefinition.homeOrganization]: ["", [Validators.required, SolidifyValidator]],
      [this.formDefinition.email]: ["", [Validators.required, SolidifyValidator]],
      [this.formDefinition.person]: [""],
      [this.formDefinition.applicationRole]: ["", [Validators.required, SolidifyValidator]],
    });
  }

  protected disableSpecificField(): void {
  }

  protected treatmentBeforeSubmit(user: User): User {
    delete user.lastLoginTime;
    delete user.lastLoginIpAddress;
    user.applicationRole = {resId: this.form.get(this.formDefinition.applicationRole).value};
    // create a new person with the corresponding resId, if needed.
    const personResId: string = user.person as any;
    if (isNullOrUndefined(personResId) || isEmptyString(personResId)) {
      user.person = null;
    } else {
      user.person = {resId: personResId};
    }
    return user;
  }

  goToPerson(event: Event, personId: string): void {
    if (isNotNullNorUndefined(event)) {
      event.stopPropagation();
      event.preventDefault();
    }
    this._navigateBS.next([RoutesEnum.adminPersonDetail, personId]);
  }

  getUser(): User {
    if (isNullOrUndefined(this.form)) {
      return this.model;
    }
    return this.form.value as User;
  }

  getPersonResId(): string {
    if (isNullOrUndefined(this.model.person)) {
      return null;
    }
    return this.model.person.resId;
  }
}

class FormComponentFormDefinition extends BaseFormDefinition {
  @PropertyName() externalUid: string;
  @PropertyName() firstName: string;
  @PropertyName() lastName: string;
  @PropertyName() homeOrganization: string;
  @PropertyName() email: string;
  @PropertyName() person: string;
  @PropertyName() accessToken: string;
  @PropertyName() refreshToken: string;
  @PropertyName() applicationRole: string;
  @PropertyName() lastLoginIpAddress: string;
  @PropertyName() lastLoginTime: string;
}
