import {
  ChangeDetectionStrategy,
  ChangeDetectorRef,
  Component,
  ElementRef,
  Input,
} from "@angular/core";
import {
  FormBuilder,
  Validators,
} from "@angular/forms";
import {
  Language,
  ResearchDomain,
} from "@models";
import {SharedAbstractFormPresentational} from "@shared/components/presentationals/shared-abstract-form/shared-abstract-form.presentational";
import {BaseFormDefinition} from "@shared/models/base-form-definition.model";
import {ResearchDomainLabel} from "@shared/models/business/research-domain-label.model";
import {
  isNullOrUndefined,
  PropertyName,
  SolidifyValidator,
} from "solidify-frontend";

@Component({
  selector: "dlcm-admin-research-domain-form",
  templateUrl: "./admin-research-domain-form.presentational.html",
  styleUrls: ["../../../../../../shared/components/presentationals/shared-abstract-form/shared-abstract-form.presentational.scss"],
  changeDetection: ChangeDetectionStrategy.OnPush,
})
export class AdminResearchDomainFormPresentational extends SharedAbstractFormPresentational<ResearchDomain> {
  private _languages: Language[];

  @Input()
  set languages(value: Language[]) {
    this._languages = value;
    if (isNullOrUndefined(this.languages)) {
      return;
    }
    this.languages.forEach(language => {
      if (isNullOrUndefined(this.model) || isNullOrUndefined(this.model.labels)) {
        this.addControl(language.resId);
        return;
      }
      const existingValue = (this.model.labels as ResearchDomainLabel[]).find(label => label.language.resId === language.resId);
      if (isNullOrUndefined(existingValue)) {
        this.addControl(language.resId);
        return;
      }
      this.addControl(language.resId, existingValue.description);
    });
  }

  private addControl(languageResId: string, value: string | undefined = undefined): void {
    this.form.addControl(languageResId, this._fb.control(value));
  }

  get languages(): Language[] {
    return this._languages;
  }

  public formDefinition: FormComponentFormDefinition = new FormComponentFormDefinition();

  constructor(protected readonly _changeDetectorRef: ChangeDetectorRef,
              protected readonly _elementRef: ElementRef,
              private readonly _fb: FormBuilder) {
    super(_changeDetectorRef, _elementRef);
  }

  protected initNewForm(): void {
    this.form = this._fb.group({
      [this.formDefinition.name]: ["", [Validators.required, SolidifyValidator]],
      [this.formDefinition.source]: ["", [Validators.required, SolidifyValidator]],
      [this.formDefinition.code]: ["", [Validators.required, SolidifyValidator]],
    });
  }

  protected bindFormTo(researchDomains: ResearchDomain): void {
    this.form = this._fb.group({
      [this.formDefinition.name]: [researchDomains.name, [Validators.required, SolidifyValidator]],
      [this.formDefinition.source]: [researchDomains.source, [Validators.required, SolidifyValidator]],
      [this.formDefinition.code]: [researchDomains.code, [Validators.required, SolidifyValidator]],
    });
  }

  protected treatmentBeforeSubmit(researchDomains: ResearchDomain): ResearchDomain {
    const newLabels: ResearchDomainLabel[] = [];
    this.languages.forEach(language => {
      newLabels.push({
        description: researchDomains[language.resId],
        language: language,
      });
    });
    return {
      code: researchDomains.code,
      name: researchDomains.name,
      source: researchDomains.source,
      labels: newLabels,
    };
  }
}

class FormComponentFormDefinition extends BaseFormDefinition {
  @PropertyName() name: string;
  @PropertyName() source: string;
  @PropertyName() code: string;
}
