import {
  AdminPersonAction,
  adminPersonActionNameSpace,
} from "@admin/person/stores/admin-person.action";
import {AdminPersonInstitutionsAction} from "@admin/person/stores/institutions/admin-people-institutions.action";
import {
  AdminPersonInstitutionsState,
  AdminPersonInstitutionsStateModel,
} from "@admin/person/stores/institutions/admin-people-institutions.state";
import {AdminPersonOrgUnitRoleAction} from "@admin/person/stores/person-role/admin-person-orgunit-role.action";
import {
  AdminPersonOrgUnitRoleState,
  AdminPersonOrgUnitRoleStateModel,
  defaultAdminPersonOrgUnitRoleStateModel,
} from "@admin/person/stores/person-role/admin-person-orgunit-role.state";
import {Injectable} from "@angular/core";
import {DepositStateModel} from "@deposit/stores/deposit.state";
import {Person} from "@models";
import {
  Action,
  Actions,
  ofActionCompleted,
  Selector,
  State,
  StateContext,
  Store,
} from "@ngxs/store";
import {AdminResourceApiEnum} from "@shared/enums/api.enum";
import {LocalStateEnum} from "@shared/enums/local-state.enum";
import {RoutesEnum} from "@shared/enums/routes.enum";
import {SharedRoleAction} from "@shared/stores/role/shared-role.action";
import {Observable} from "rxjs";
import {tap} from "rxjs/operators";
import {
  ApiService,
  defaultAssociationStateInitValue,
  defaultRelation3TiersStateInitValue,
  defaultResourceStateInitValue,
  isArray,
  isNullOrUndefined,
  MARK_AS_TRANSLATABLE,
  NotificationService,
  OverrideDefaultAction,
  Relation3TiersForm,
  ResourceState,
  ResourceStateModel,
  StoreUtil,
  urlSeparator,
} from "solidify-frontend";

export interface AdminPersonStateModel extends ResourceStateModel<Person> {
  admin_person_organizationalUnitRole?: AdminPersonOrgUnitRoleStateModel;
  admin_person_institutions?: AdminPersonInstitutionsStateModel;
}

@Injectable()
@State<AdminPersonStateModel>({
  name: LocalStateEnum.admin_person,
  defaults: {
    ...defaultResourceStateInitValue(),
    admin_person_organizationalUnitRole: {...defaultAdminPersonOrgUnitRoleStateModel()},
    admin_person_institutions: {...defaultAssociationStateInitValue()},
  },
  children: [
    AdminPersonOrgUnitRoleState,
    AdminPersonInstitutionsState,
  ],
})
export class AdminPersonState extends ResourceState<AdminPersonStateModel, Person> {
  constructor(protected apiService: ApiService,
              protected store: Store,
              protected notificationService: NotificationService,
              protected actions$: Actions) {
    super(apiService, store, notificationService, actions$, {
      nameSpace: adminPersonActionNameSpace,
      routeRedirectUrlAfterSuccessCreateAction: (resId: string) => RoutesEnum.adminPersonDetail + urlSeparator + resId,
      routeRedirectUrlAfterSuccessUpdateAction: (resId: string) => RoutesEnum.adminPersonDetail + urlSeparator + resId,
      routeRedirectUrlAfterSuccessDeleteAction: RoutesEnum.adminPerson,
      notificationResourceCreateSuccessTextToTranslate: MARK_AS_TRANSLATABLE("admin.person.notification.resource.create"),
      notificationResourceDeleteSuccessTextToTranslate: MARK_AS_TRANSLATABLE("admin.person.notification.resource.delete"),
      notificationResourceUpdateSuccessTextToTranslate: MARK_AS_TRANSLATABLE("admin.person.notification.resource.update"),
    });
  }

  protected get _urlResource(): string {
    return AdminResourceApiEnum.people;
  }

  @Selector()
  static isLoading(state: AdminPersonStateModel): boolean {
    return StoreUtil.isLoadingState(state);
  }

  @Selector()
  static isLoadingWithDependency(state: AdminPersonStateModel): boolean {
    return this.isLoading(state)
      || StoreUtil.isLoadingState(state.admin_person_organizationalUnitRole)
      || StoreUtil.isLoadingState(state.admin_person_institutions);
  }

  @Selector()
  static currentTitle(state: AdminPersonStateModel): string | undefined {
    if (isNullOrUndefined(state.current)) {
      return undefined;
    }
    return state.current.fullName;
  }

  @Selector()
  static isReadyToBeDisplayed(state: AdminPersonStateModel): boolean {
    return this.isReadyToBeDisplayedInCreateMode
      && !isNullOrUndefined(state.current)
      && !isNullOrUndefined(state.admin_person_organizationalUnitRole.selected)
      && !isNullOrUndefined(state.admin_person_institutions.selected);
  }

  @Selector()
  static isReadyToBeDisplayedInCreateMode(state: AdminPersonStateModel): boolean {
    return true;
  }

  @Action(AdminPersonAction.LoadResource)
  loadResource(ctx: StateContext<DepositStateModel>, action: AdminPersonAction.LoadResource): Observable<any> {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
    });

    return StoreUtil.dispatchParallelActionAndWaitForSubActionsCompletion(ctx, [
      {
        action: new SharedRoleAction.GetAll(undefined, false, false),
        subActionCompletions: [
          this.actions$.pipe(ofActionCompleted(SharedRoleAction.GetAllSuccess)),
          this.actions$.pipe(ofActionCompleted(SharedRoleAction.GetAllFail)),
        ],
      },
    ]).pipe(
      tap(success => {
        if (success) {
          ctx.dispatch(new AdminPersonAction.LoadResourceSuccess(action));
        } else {
          ctx.dispatch(new AdminPersonAction.LoadResourceFail(action));
        }
      }),
    );
  }

  @OverrideDefaultAction()
  @Action(AdminPersonAction.Create)
  create(ctx: StateContext<AdminPersonStateModel>, action: AdminPersonAction.Create): Observable<Person> {
    return super.internalCreate(ctx, action)
      .pipe(
        tap(model => {
          this.updateSubResource(model, action, ctx)
            .subscribe(success => {
              if (success) {
                ctx.dispatch(new AdminPersonAction.CreateSuccess(action, model));
              } else {
                ctx.dispatch(new AdminPersonAction.CreateFail(action));
              }
            });
        }),
      );
  }

  @OverrideDefaultAction()
  @Action(AdminPersonAction.Update)
  update(ctx: StateContext<AdminPersonStateModel>, action: AdminPersonAction.Update): Observable<Person> {
    return super.internalUpdate(ctx, action)
      .pipe(
        tap(model => {
          this.updateSubResource(model, action, ctx)
            .subscribe(success => {
              if (success) {
                ctx.dispatch(new AdminPersonAction.UpdateSuccess(action, model));
              } else {
                ctx.dispatch(new AdminPersonAction.UpdateFail(action));
              }
            });
        }),
      );
  }

  private updateSubResource(model: Person, action: AdminPersonAction.Create | AdminPersonAction.Update, ctx: StateContext<AdminPersonStateModel>): Observable<boolean> {
    const personId = model.resId;
    const newOrgUnitRole = action.modelFormControlEvent.formControl.get("orgUnitRole").value as Relation3TiersForm[];
    newOrgUnitRole.forEach((elem) => {
      if (!isNullOrUndefined(elem.listId) && !isArray(elem.listId)) {
        elem.listId = [elem.listId];
      }
    });
    const newInstitutions = action.modelFormControlEvent.model.institutions.map(o => o.resId);

    return StoreUtil.dispatchParallelActionAndWaitForSubActionsCompletion(ctx, [
      {
        action: new AdminPersonOrgUnitRoleAction.Update(personId, newOrgUnitRole),
        subActionCompletions: [
          this.actions$.pipe(ofActionCompleted(AdminPersonOrgUnitRoleAction.UpdateSuccess)),
          this.actions$.pipe(ofActionCompleted(AdminPersonOrgUnitRoleAction.UpdateFail)),
        ],
      },
      {
        action: new AdminPersonInstitutionsAction.Update(personId, newInstitutions),
        subActionCompletions: [
          this.actions$.pipe(ofActionCompleted(AdminPersonInstitutionsAction.UpdateSuccess)),
          this.actions$.pipe(ofActionCompleted(AdminPersonInstitutionsAction.UpdateFail)),
        ],
      },
    ]);
  }
}
