import {OrganizationalUnitRole} from "@admin/models/organizational-unit-role.model";
import {
  ChangeDetectionStrategy,
  ChangeDetectorRef,
  Component,
  ElementRef,
  Input,
} from "@angular/core";
import {
  FormBuilder,
  Validators,
} from "@angular/forms";
import {sharedInstitutionActionNameSpace} from "@app/shared/stores/institution/shared-institution.action";
import {
  Institution,
  Person,
  Role,
} from "@models";
import {PersonOrgUnitRoleMode} from "@shared/components/containers/shared-table-person-orgunit-role/shared-table-person-orgunit-role.container";
import {SharedAbstractFormPresentational} from "@shared/components/presentationals/shared-abstract-form/shared-abstract-form.presentational";
import {RoutesEnum} from "@shared/enums/routes.enum";
import {BaseFormDefinition} from "@shared/models/base-form-definition.model";
import {SharedInstitutionState} from "@shared/stores/institution/shared-institution.state";
import {
  isNullOrUndefined,
  OrderEnum,
  PropertyName,
  ResourceNameSpace,
  SolidifyValidator,
  Sort,
} from "solidify-frontend";

@Component({
  selector: "dlcm-admin-person-form",
  templateUrl: "./admin-person-form.presentational.html",
  styleUrls: ["../../../../../../shared/components/presentationals/shared-abstract-form/shared-abstract-form.presentational.scss"],
  changeDetection: ChangeDetectionStrategy.OnPush,
})
export class AdminPersonFormPresentational extends SharedAbstractFormPresentational<Person> {
  formDefinition: FormComponentFormDefinition = new FormComponentFormDefinition();

  @Input()
  selectedOrgUnitRole: OrganizationalUnitRole[];

  @Input()
  selectedInstitutions: Institution[];

  @Input()
  listRole: Role[];

  sharedInstitutionSort: Sort<Institution> = {
    field: "name",
    order: OrderEnum.ascending,
  };
  sharedInstitutionActionNameSpace: ResourceNameSpace = sharedInstitutionActionNameSpace;
  sharedInstitutionState: typeof SharedInstitutionState = SharedInstitutionState;

  get personOrgUnitRoleMode(): typeof PersonOrgUnitRoleMode {
    return PersonOrgUnitRoleMode;
  }

  constructor(protected readonly _changeDetectorRef: ChangeDetectorRef,
              protected readonly _elementRef: ElementRef,
              private readonly _fb: FormBuilder) {
    super(_changeDetectorRef, _elementRef);
  }

  protected bindFormTo(person: Person): void {
    this.form = this._fb.group({
      [this.formDefinition.firstName]: [person.firstName, [Validators.required, SolidifyValidator]],
      [this.formDefinition.lastName]: [person.lastName, [Validators.required, SolidifyValidator]],
      [this.formDefinition.orcid]: [person.orcid, [SolidifyValidator]],
      [this.formDefinition.orgUnitRole]: ["", [SolidifyValidator]],
      [this.formDefinition.institutions]: [this.selectedInstitutions.map(i => i.resId)],
    });
  }

  protected initNewForm(): void {
    this.form = this._fb.group({
      [this.formDefinition.firstName]: ["", [Validators.required, SolidifyValidator]],
      [this.formDefinition.lastName]: ["", [Validators.required, SolidifyValidator]],
      [this.formDefinition.orcid]: ["", [SolidifyValidator]],
      [this.formDefinition.institutions]: [],
      [this.formDefinition.orgUnitRole]: ["", [SolidifyValidator]],
    });
  }

  protected treatmentBeforeSubmit(person: Person): Person {
    person.institutions = [];
    const listInstitutions = this.form.get(this.formDefinition.institutions).value;
    if (!isNullOrUndefined(listInstitutions)) {
      listInstitutions.forEach(resId => {
        person.institutions.push({resId: resId});
      });
    }
    return person;
  }

  navigateToInstitution(institution: Institution): void {
    this.navigate([RoutesEnum.adminInstitutionDetail, institution.resId]);
  }

  navigateToContributor(): void {
    this.navigate([RoutesEnum.preservationSpaceContributorDetail, this.model.resId]);
  }
}

class FormComponentFormDefinition extends BaseFormDefinition {
  @PropertyName() firstName: string;
  @PropertyName() lastName: string;
  @PropertyName() orcid: string;
  @PropertyName() orgUnitRole: string;
  @PropertyName() institutions: string;
}
