import {PersonRole} from "@admin/models/person-role.model";
import {AdminOrganizationalUnitAction} from "@admin/organizational-unit/stores/admin-organizational-unit.action";
import {AdminOrganizationalUnitState} from "@admin/organizational-unit/stores/admin-organizational-unit.state";
import {
  ChangeDetectionStrategy,
  ChangeDetectorRef,
  Component,
  ElementRef,
  Input,
  ViewChild,
} from "@angular/core";
import {
  FormBuilder,
  Validators,
} from "@angular/forms";
import {NotificationHelper} from "@app/features/preservation-space/notification/helper/notification.helper";
import {BreakpointService} from "@app/shared/services/breakpoint.service";
import {sharedResearchDomainActionNameSpace} from "@app/shared/stores/research-domain/shared-research-domain.action";
import {environment} from "@environments/environment";
import {
  DisseminationPolicy,
  FundingAgency,
  Institution,
  License,
  OrganizationalUnit,
  PreservationPolicy,
  ResearchDomain,
  Role,
  SubmissionPolicy,
} from "@models";
import {PersonOrgUnitRoleMode} from "@shared/components/containers/shared-table-person-orgunit-role/shared-table-person-orgunit-role.container";
import {SharedAbstractFormPresentational} from "@shared/components/presentationals/shared-abstract-form/shared-abstract-form.presentational";
import {SharedSearchableSingleSelectPresentational} from "@shared/components/presentationals/shared-searchable-single-select/shared-searchable-single-select.presentational";
import {RoutesEnum} from "@shared/enums/routes.enum";
import {BaseFormDefinition} from "@shared/models/base-form-definition.model";
import {sharedFundingAgencyActionNameSpace} from "@shared/stores/funding-agency/shared-funding-agency.action";
import {SharedFundingAgencyState} from "@shared/stores/funding-agency/shared-funding-agency.state";
import {sharedInstitutionActionNameSpace} from "@shared/stores/institution/shared-institution.action";
import {SharedInstitutionState} from "@shared/stores/institution/shared-institution.state";
import {sharedLicenseActionNameSpace} from "@shared/stores/license/shared-license.action";
import {
  SharedLicenseState,
  SharedLicenseStateModel,
} from "@shared/stores/license/shared-license.state";
import {SharedResearchDomainState} from "@shared/stores/research-domain/shared-research-domain.state";
import {ResourceLogoNameSpace} from "@shared/stores/resource-logo/resource-logo-namespace.model";
import {RegexpUtil} from "@shared/utils/regexp.util";
import {
  DateUtil,
  isEmptyString,
  isNullOrUndefined,
  MappingObject,
  MappingObjectUtil,
  OrderEnum,
  PropertyName,
  ResourceNameSpace,
  SolidifyValidator,
  Sort,
  urlSeparator,
} from "solidify-frontend";

@Component({
  selector: "dlcm-admin-organizational-unit-form",
  templateUrl: "./admin-organizational-unit-form.presentational.html",
  styleUrls: ["./admin-organizational-unit-form.presentational.scss"],
  changeDetection: ChangeDetectionStrategy.OnPush,
})
export class AdminOrganizationalUnitFormPresentational extends SharedAbstractFormPresentational<OrganizationalUnit> {
  private readonly KEY_SOURCE: keyof ResearchDomain = "source";
  readonly TIME_BEFORE_DISPLAY_TOOLTIP: number = environment.timeBeforeDisplayTooltipOnInput;
  formDefinition: FormComponentFormDefinition = new FormComponentFormDefinition();

  organizationalUnitActionNameSpace: ResourceLogoNameSpace = AdminOrganizationalUnitAction;
  organizationalUnitState: typeof AdminOrganizationalUnitState = AdminOrganizationalUnitState;

  @Input()
  selectedSubmissionPolicies: SubmissionPolicy[];

  @Input()
  selectedPreservationPolicies: PreservationPolicy[];

  @Input()
  selectedFundingAgencies: FundingAgency[];

  @Input()
  selectedDisseminationPolicies: DisseminationPolicy[];

  @Input()
  selectedPersonRole: PersonRole[];

  @Input()
  selectedInstitutions: Institution[];

  @Input()
  listSubmissionPolicies: SubmissionPolicy[];

  @Input()
  listPreservationPolicies: PreservationPolicy[];

  @Input()
  listDisseminationPolicies: DisseminationPolicy[];

  @Input()
  researchDomainSources: string[];

  @Input()
  listRole: Role[];

  @Input()
  defaultPlatformLicenseId: string;

  @Input()
  urlQueryParameters: MappingObject<string | undefined>;

  @ViewChild("licenseSingleSearchableSelect")
  licenseSingleSearchableSelect: SharedSearchableSingleSelectPresentational<SharedLicenseStateModel, License>;

  sharedInstitutionSort: Sort<Institution> = {
    field: "name",
    order: OrderEnum.ascending,
  };
  sharedInstitutionActionNameSpace: ResourceNameSpace = sharedInstitutionActionNameSpace;
  sharedInstitutionState: typeof SharedInstitutionState = SharedInstitutionState;

  sharedFundingAgencySort: Sort<FundingAgency> = {
    field: "name",
    order: OrderEnum.ascending,
  };
  sharedFundingAgencyActionNameSpace: ResourceNameSpace = sharedFundingAgencyActionNameSpace;
  sharedFundingAgencyState: typeof SharedFundingAgencyState = SharedFundingAgencyState;

  sharedResearchDomainSort: Sort<ResearchDomain> = {
    field: "name",
    order: OrderEnum.ascending,
  };
  sharedResearchDomainActionNameSpace: ResourceNameSpace = sharedResearchDomainActionNameSpace;
  sharedResearchDomainState: typeof SharedResearchDomainState = SharedResearchDomainState;

  sharedLicenseSort: Sort<License> = {
    field: "title",
    order: OrderEnum.ascending,
  };
  sharedLicenseActionNameSpace: ResourceNameSpace = sharedLicenseActionNameSpace;
  sharedLicenseState: typeof SharedLicenseState = SharedLicenseState;

  extraSearchParameterSource: MappingObject = {};

  licenceCallback: (value: License) => string = (value: License) => value.openLicenseId + " (" + value.title + ")";
  researchDomainLabelCallback: (value: ResearchDomain) => string = (value: ResearchDomain) => `[${value.source}] ${value.name}`;

  get personOrgUnitRoleMode(): typeof PersonOrgUnitRoleMode {
    return PersonOrgUnitRoleMode;
  }

  constructor(protected readonly _changeDetectorRef: ChangeDetectorRef,
              protected readonly _elementRef: ElementRef,
              private readonly _fb: FormBuilder,
              public readonly breakpointService: BreakpointService) {
    super(_changeDetectorRef, _elementRef);
  }

  protected initNewForm(): void {
    this.form = this._fb.group({
      [this.formDefinition.name]: ["", [Validators.required, SolidifyValidator]],
      [this.formDefinition.description]: ["", [SolidifyValidator]],
      [this.formDefinition.openingDate]: [""],
      [this.formDefinition.closingDate]: [""],
      [this.formDefinition.submissionPolicies]: [[], [Validators.required, SolidifyValidator]],
      [this.formDefinition.preservationPolicies]: [[], [Validators.required, SolidifyValidator]],
      [this.formDefinition.disseminationPolicies]: [[], [SolidifyValidator]],
      [this.formDefinition.defaultSubmissionPolicy]: [undefined, [Validators.required, SolidifyValidator]],
      [this.formDefinition.defaultPreservationPolicy]: [undefined, [Validators.required, SolidifyValidator]],
      [this.formDefinition.defaultDisseminationPolicy]: [undefined, [SolidifyValidator]],
      [this.formDefinition.url]: ["", [SolidifyValidator, Validators.pattern(RegexpUtil.url)]],
      [this.formDefinition.personRole]: ["", [SolidifyValidator]],
      [this.formDefinition.fundingAgencies]: ["", [SolidifyValidator]],
      [this.formDefinition.institutions]: ["", [SolidifyValidator]],
      [this.formDefinition.filterResearchDomains]: ["", [SolidifyValidator]],
      [this.formDefinition.researchDomains]: ["", [SolidifyValidator]],
      [this.formDefinition.keywords]: [[], [SolidifyValidator]],
      [this.formDefinition.defaultLicense]: ["", [SolidifyValidator]],
    });

    this._creationForUserAskingInNotificationRequest();
  }

  protected bindFormTo(organizationalUnit: OrganizationalUnit): void {
    this.form = this._fb.group({
      [this.formDefinition.name]: [organizationalUnit.name, [Validators.required, SolidifyValidator]],
      [this.formDefinition.description]: [organizationalUnit.description, [SolidifyValidator]],
      [this.formDefinition.openingDate]: [organizationalUnit.openingDate],
      [this.formDefinition.closingDate]: [organizationalUnit.closingDate],
      [this.formDefinition.submissionPolicies]: [this.selectedSubmissionPolicies.map(s => s.resId), [Validators.required, SolidifyValidator]],
      [this.formDefinition.preservationPolicies]: [this.selectedPreservationPolicies.map(p => p.resId), [Validators.required, SolidifyValidator]],
      [this.formDefinition.disseminationPolicies]: [this.selectedDisseminationPolicies.map(d => d.resId), [SolidifyValidator]],
      [this.formDefinition.defaultSubmissionPolicy]: [(organizationalUnit.defaultSubmissionPolicy ? organizationalUnit.defaultSubmissionPolicy.resId : undefined), [Validators.required, SolidifyValidator]],
      [this.formDefinition.defaultPreservationPolicy]: [(organizationalUnit.defaultPreservationPolicy ? organizationalUnit.defaultPreservationPolicy.resId : undefined), [Validators.required, SolidifyValidator]],
      [this.formDefinition.defaultDisseminationPolicy]: [(organizationalUnit.defaultDisseminationPolicy ? organizationalUnit.defaultDisseminationPolicy.resId : undefined), [SolidifyValidator]],
      [this.formDefinition.url]: [organizationalUnit.url, [SolidifyValidator, Validators.pattern(RegexpUtil.url)]],
      [this.formDefinition.personRole]: ["", [SolidifyValidator]],
      [this.formDefinition.fundingAgencies]: [this.selectedFundingAgencies.map(f => f.resId), [SolidifyValidator]],
      [this.formDefinition.institutions]: [this.selectedInstitutions.map(i => i.resId), [SolidifyValidator]],
      [this.formDefinition.filterResearchDomains]: ["", [SolidifyValidator]],
      [this.formDefinition.researchDomains]: [organizationalUnit.researchDomains?.map(r => r.resId), [SolidifyValidator]],
      [this.formDefinition.keywords]: [[...organizationalUnit.keywords], [SolidifyValidator]],
      [this.formDefinition.defaultLicense]: [organizationalUnit?.defaultLicense?.resId, [SolidifyValidator]],
    });
  }

  protected treatmentBeforeSubmit(organizationalUnit: OrganizationalUnit | any): OrganizationalUnit {
    delete organizationalUnit[this.formDefinition.filterResearchDomains];
    organizationalUnit.openingDate = DateUtil.convertToLocalDateDateSimple(organizationalUnit.openingDate);
    organizationalUnit.closingDate = DateUtil.convertToLocalDateDateSimple(organizationalUnit.closingDate);
    organizationalUnit.defaultSubmissionPolicy = {resId: organizationalUnit[this.formDefinition.defaultSubmissionPolicy]};
    organizationalUnit.defaultPreservationPolicy = {resId: organizationalUnit[this.formDefinition.defaultPreservationPolicy]};
    organizationalUnit.defaultDisseminationPolicy = {resId: organizationalUnit[this.formDefinition.defaultDisseminationPolicy]};
    organizationalUnit.preservationPolicies = [];
    this.form.get(this.formDefinition.preservationPolicies).value.forEach(resId => {
      organizationalUnit.preservationPolicies.push({resId: resId});
    });
    organizationalUnit.submissionPolicies = [];
    this.form.get(this.formDefinition.submissionPolicies).value.forEach(resId => {
      organizationalUnit.submissionPolicies.push({resId: resId});
    });
    organizationalUnit.disseminationPolicies = [];
    this.form.get(this.formDefinition.disseminationPolicies).value.forEach(resId => {
      organizationalUnit.disseminationPolicies.push({resId: resId});
    });
    organizationalUnit.fundingAgencies = [];
    if (this.form.get(this.formDefinition.fundingAgencies).value !== "") {
      this.form.get(this.formDefinition.fundingAgencies).value.forEach(resId => {
        organizationalUnit.fundingAgencies.push({resId: resId});
      });
    }
    organizationalUnit.institutions = [];
    if (this.form.get(this.formDefinition.institutions).value !== "") {
      this.form.get(this.formDefinition.institutions).value.forEach(resId => {
        organizationalUnit.institutions.push({resId: resId});
      });
    }
    organizationalUnit.researchDomains = [];
    if (this.form.get(this.formDefinition.researchDomains).value !== "") {
      this.form.get(this.formDefinition.researchDomains).value.forEach(resId => {
        organizationalUnit.researchDomains.push({resId: resId});
      });
    }

    const defaultLicenseResId = this.form.get(this.formDefinition.defaultLicense).value;
    if (this.licenseSingleSearchableSelect.isDefaultValueSelected || isNullOrUndefined(defaultLicenseResId) || isEmptyString(defaultLicenseResId)) {
      organizationalUnit.defaultLicense = null;
    } else {
      organizationalUnit.defaultLicense = {resId: defaultLicenseResId};
    }
    return organizationalUnit;
  }

  navigateToDeposit(orgUnitId: string): void {
    this.navigate([RoutesEnum.deposit + urlSeparator + orgUnitId]);
  }

  navigateToArchive(orgUnitId: string): void {
    this.navigate([RoutesEnum.homeDetail + urlSeparator + orgUnitId]);
  }

  navigateToSubmissionPolicy(submissionPolicy: SubmissionPolicy): void {
    this.navigate([RoutesEnum.adminSubmissionPolicyDetail, submissionPolicy.resId]);
  }

  navigateToPreservationPolicy(preservationPolicy: PreservationPolicy): void {
    this.navigate([RoutesEnum.adminPreservationPolicyDetail, preservationPolicy.resId]);
  }

  navigateToDisseminationPolicy(disseminationPolicy: DisseminationPolicy): void {
    this.navigate([RoutesEnum.adminDisseminationPolicyDetail, disseminationPolicy.resId]);
  }

  navigateToFundingAgency(fundingAgency: FundingAgency): void {
    this.navigate([RoutesEnum.adminFundingAgenciesDetail, fundingAgency.resId]);
  }

  navigateToInstitution(institution: Institution): void {
    this.navigate([RoutesEnum.adminInstitutionDetail, institution.resId]);
  }

  navigateToResearchDomain(researchDomain: ResearchDomain): void {
    this.navigate([RoutesEnum.adminResearchDomain]);
  }

  navigateToLicense(licenseId: string): void {
    this.navigate([RoutesEnum.adminLicenseDetail, licenseId]);
  }

  filterResearchDomain(source: string): void {
    this._computeExtraSearchParameterCategory(source);
  }

  private _creationForUserAskingInNotificationRequest(): void {
    if (isNullOrUndefined(this.urlQueryParameters) || MappingObjectUtil.size(this.urlQueryParameters) === 0) {
      return;
    }
    const roleId = MappingObjectUtil.get(this.urlQueryParameters, NotificationHelper.KEY_ROLE_ID);
    const personId = MappingObjectUtil.get(this.urlQueryParameters, NotificationHelper.KEY_PERSON_ID);
    const orgUnitName = decodeURIComponent(MappingObjectUtil.get(this.urlQueryParameters, NotificationHelper.KEY_ORGUNIT_NAME));
    if (isNullOrUndefined(roleId) || isNullOrUndefined(personId) || isNullOrUndefined(orgUnitName)) {
      return;
    }
    const existingPerson = this.selectedPersonRole.find(p => p.resId === personId);
    if (isNullOrUndefined(existingPerson)) {
      this.selectedPersonRole = [{
        resId: personId,
        roles: [{
          resId: roleId,
        }],
      }];
    }
    this.form.get(this.formDefinition.name).setValue(orgUnitName);
  }

  private _computeExtraSearchParameterCategory(value: string): void {
    if (isNullOrUndefined(value) || isEmptyString(value)) {
      MappingObjectUtil.delete(this.extraSearchParameterSource, this.KEY_SOURCE as string);
      return;
    }
    MappingObjectUtil.set(this.extraSearchParameterSource, this.KEY_SOURCE as string, value);
  }
}

class FormComponentFormDefinition extends BaseFormDefinition {
  @PropertyName() name: string;
  @PropertyName() description: string;
  @PropertyName() closingDate: string;
  @PropertyName() openingDate: string;
  @PropertyName() submissionPolicies: string;
  @PropertyName() preservationPolicies: string;
  @PropertyName() disseminationPolicies: string;
  @PropertyName() defaultSubmissionPolicy: string;
  @PropertyName() defaultPreservationPolicy: string;
  @PropertyName() defaultDisseminationPolicy: string;
  @PropertyName() url: string;
  @PropertyName() personRole: string;
  @PropertyName() fundingAgencies: string;
  @PropertyName() institutions: string;
  @PropertyName() filterResearchDomains: string;
  @PropertyName() researchDomains: string;
  @PropertyName() keywords: string;
  @PropertyName() defaultLicense: string;
}
