import {
  AdminOaiSetAction,
  adminOaiSetActionNameSpace,
} from "@admin/oai-set/stores/admin-oai-set.action";
import {Injectable} from "@angular/core";
import {OaiSet} from "@models";
import {
  Action,
  Actions,
  ofActionCompleted,
  Selector,
  State,
  StateContext,
  Store,
} from "@ngxs/store";
import {AccessResourceApiEnum} from "@shared/enums/api.enum";
import {LocalStateEnum} from "@shared/enums/local-state.enum";
import {
  RoutesEnum,
  urlSeparator,
} from "@shared/enums/routes.enum";
import {SharedOrgUnitAction} from "@shared/stores/organizational-unit/shared-organizational-unit.action";
import {Observable} from "rxjs";
import {tap} from "rxjs/operators";
import {
  ApiService,
  defaultResourceStateInitValue,
  isNullOrUndefined,
  MARK_AS_TRANSLATABLE,
  NotificationService,
  ResourceState,
  ResourceStateModel,
  StoreUtil,
} from "solidify-frontend";
import {ActionSubActionCompletionsWrapper} from "solidify-frontend/lib/models/stores/base.action";

export interface AdminOaiSetStateModel extends ResourceStateModel<OaiSet> {
}

@Injectable()
@State<AdminOaiSetStateModel>({
  name: LocalStateEnum.admin_oaiSet,
  defaults: {
    ...defaultResourceStateInitValue(),
  },
})
export class AdminOaiSetState extends ResourceState<AdminOaiSetStateModel, OaiSet> {
  constructor(protected apiService: ApiService,
              protected store: Store,
              protected notificationService: NotificationService,
              protected actions$: Actions) {
    super(apiService, store, notificationService, actions$, {
      nameSpace: adminOaiSetActionNameSpace,
      routeRedirectUrlAfterSuccessCreateAction: (resId: string) => RoutesEnum.adminOaiSetDetail + urlSeparator + resId,
      routeRedirectUrlAfterSuccessUpdateAction: (resId: string) => RoutesEnum.adminOaiSetDetail + urlSeparator + resId,
      routeRedirectUrlAfterSuccessDeleteAction: RoutesEnum.adminOaiSet,
      notificationResourceCreateSuccessTextToTranslate: MARK_AS_TRANSLATABLE("admin.oai-set.notification.resource.create"),
      notificationResourceDeleteSuccessTextToTranslate: MARK_AS_TRANSLATABLE("admin.oai-set.notification.resource.delete"),
      notificationResourceUpdateSuccessTextToTranslate: MARK_AS_TRANSLATABLE("admin.oai-set.notification.resource.update"),
    });
  }

  protected get _urlResource(): string {
    return AccessResourceApiEnum.oaiSets;
  }

  @Selector()
  static isLoading(state: AdminOaiSetStateModel): boolean {
    return StoreUtil.isLoadingState(state);
  }

  @Selector()
  static isLoadingWithDependency(state: AdminOaiSetStateModel): boolean {
    return this.isLoading(state);
  }

  @Selector()
  static isReadyToBeDisplayed(state: AdminOaiSetStateModel): boolean {
    return this.isReadyToBeDisplayedInCreateMode
      && !isNullOrUndefined(state.current);
  }

  @Selector()
  static isReadyToBeDisplayedInCreateMode(state: AdminOaiSetStateModel): boolean {
    return true;
  }

  @Selector()
  static currentTitle(state: AdminOaiSetStateModel): string | undefined {
    if (isNullOrUndefined(state.current)) {
      return undefined;
    }
    return state.current.name;
  }

  @Action(AdminOaiSetAction.LoadResource)
  loadResource(ctx: StateContext<AdminOaiSetStateModel>, action: AdminOaiSetAction.LoadResource): Observable<any> {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
    });

    return StoreUtil.dispatchParallelActionAndWaitForSubActionsCompletion(ctx, [
      {
        action: new SharedOrgUnitAction.GetAll(undefined, false, false),
        subActionCompletions: [
          this.actions$.pipe(ofActionCompleted(SharedOrgUnitAction.GetAllSuccess)),
          this.actions$.pipe(ofActionCompleted(SharedOrgUnitAction.GetAllFail)),
        ],
      },
    ]).pipe(
      tap(success => {
        if (success) {
          ctx.dispatch(new AdminOaiSetAction.LoadResourceSuccess(action));
        } else {
          ctx.dispatch(new AdminOaiSetAction.LoadResourceFail(action));
        }
      }),
    );
  }

  @Action(AdminOaiSetAction.BulkCreate)
  bulkCreate(ctx: StateContext<AdminOaiSetStateModel>, action: AdminOaiSetAction.BulkCreate): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
    });

    const listActionSubActionCompletionsWrapper: ActionSubActionCompletionsWrapper[] = [];
    const listOrgUnit = action.listOrgUnit;
    listOrgUnit.forEach(orgUnit => {
      const orgUnitName = orgUnit.name;
      const oaiSet = {
        spec: `orgunit:${orgUnitName.toLowerCase()}`,
        name: orgUnitName,
        description: `Set for ${orgUnitName}`,
        query: `aip-organizational-unit:${orgUnit.resId}`,
        enabled: false,
      } as OaiSet;
      listActionSubActionCompletionsWrapper.push({
        action: new AdminOaiSetAction.CreateDuringBulk({
          model: oaiSet,
        }),
        subActionCompletions: [
          this.actions$.pipe(ofActionCompleted(AdminOaiSetAction.CreateDuringBulkSuccess)),
          this.actions$.pipe(ofActionCompleted(AdminOaiSetAction.CreateFail)),
        ],
      });
    });
    StoreUtil.dispatchSequentialActionAndWaitForSubActionsCompletion(ctx, listActionSubActionCompletionsWrapper).subscribe(success => {
      if (success) {
        ctx.dispatch(new AdminOaiSetAction.BulkCreateSuccess(action));
      } else {
        ctx.dispatch(new AdminOaiSetAction.BulkCreateFail(action));
      }
    });
  }

  @Action(AdminOaiSetAction.BulkCreateSuccess)
  bulkCreateSuccess(ctx: StateContext<AdminOaiSetStateModel>, action: AdminOaiSetAction.BulkCreateSuccess): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });

    StoreUtil.navigateIfDefined(ctx, RoutesEnum.adminOaiSet, null);
    this.notificationService.showSuccess(MARK_AS_TRANSLATABLE("admin.oai-set.notification.bulkCreateSuccess"));
  }

  @Action(AdminOaiSetAction.BulkCreateFail)
  bulkCreateFail(ctx: StateContext<AdminOaiSetStateModel>, action: AdminOaiSetAction.BulkCreateFail): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
  }

  @Action(AdminOaiSetAction.CreateDuringBulk)
  createDuringBulk(ctx: StateContext<AdminOaiSetStateModel>, action: AdminOaiSetAction.CreateDuringBulk): Observable<OaiSet> {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
    });

    return super.internalCreate(ctx, action as AdminOaiSetAction.Create).pipe(
      tap(model => {
        this.store.dispatch(new AdminOaiSetAction.CreateDuringBulkSuccess(action));
      }),
    );
  }
}
