import {
  AdminNotificationAction,
  adminNotificationActionNameSpace,
} from "@admin/notification/stores/admin-notification.action";
import {
  AdminNotificationStatusHistoryState,
  AdminNotificationStatusHistoryStateModel,
} from "@admin/notification/stores/status-history/admin-notification-status-history.state";
import {Injectable} from "@angular/core";
import {NotificationDlcm} from "@models";
import {
  Action,
  Actions,
  ofActionCompleted,
  Selector,
  State,
  StateContext,
  Store,
} from "@ngxs/store";
import {AdminResourceApiEnum} from "@shared/enums/api.enum";
import {LocalStateEnum} from "@shared/enums/local-state.enum";
import {RoutesEnum} from "@shared/enums/routes.enum";
import {SharedRoleAction} from "@shared/stores/role/shared-role.action";
import {defaultStatusHistoryInitValue} from "@shared/stores/status-history/status-history.state";
import {Observable} from "rxjs";
import {tap} from "rxjs/operators";
import {
  ApiService,
  defaultResourceStateInitValue,
  isNullOrUndefined,
  MARK_AS_TRANSLATABLE,
  NotificationService,
  OverrideDefaultAction,
  ResourceState,
  ResourceStateModel,
  StoreUtil,
  urlSeparator,
} from "solidify-frontend";

export interface AdminNotificationStateModel extends ResourceStateModel<NotificationDlcm> {
  admin_notification_statusHistory: AdminNotificationStatusHistoryStateModel;
}

@Injectable()
@State<AdminNotificationStateModel>({
  name: LocalStateEnum.admin_notification,
  defaults: {
    ...defaultResourceStateInitValue(),
    admin_notification_statusHistory: {...defaultStatusHistoryInitValue()},
  },
  children: [
    AdminNotificationStatusHistoryState,
  ],
})
export class AdminNotificationState extends ResourceState<AdminNotificationStateModel, NotificationDlcm> {
  constructor(protected apiService: ApiService,
              protected store: Store,
              protected notificationService: NotificationService,
              protected actions$: Actions) {
    super(apiService, store, notificationService, actions$, {
      nameSpace: adminNotificationActionNameSpace,
      routeRedirectUrlAfterSuccessCreateAction: (resId: string) => RoutesEnum.adminNotificationDetail + urlSeparator + resId,
      routeRedirectUrlAfterSuccessUpdateAction: (resId: string) => RoutesEnum.adminNotificationDetail + urlSeparator + resId,
      routeRedirectUrlAfterSuccessDeleteAction: RoutesEnum.adminNotification,
      notificationResourceCreateSuccessTextToTranslate: MARK_AS_TRANSLATABLE("admin.notifications.notification.resource.create"),
      notificationResourceDeleteSuccessTextToTranslate: MARK_AS_TRANSLATABLE("admin.notifications.notification.resource.delete"),
      notificationResourceUpdateSuccessTextToTranslate: MARK_AS_TRANSLATABLE("admin.notifications.notification.resource.update"),
    });
  }

  protected get _urlResource(): string {
    return AdminResourceApiEnum.notifications;
  }

  @Selector()
  static isLoading(state: AdminNotificationStateModel): boolean {
    return StoreUtil.isLoadingState(state);
  }

  @Selector()
  static isLoadingWithDependency(state: AdminNotificationStateModel): boolean {
    return this.isLoading(state);
  }

  @Selector()
  static currentTitle(state: AdminNotificationStateModel): string | undefined {
    if (isNullOrUndefined(state.current)) {
      return undefined;
    }
    return state.current.emitter["person"].fullName;
  }

  @Selector()
  static isReadyToBeDisplayed(state: AdminNotificationStateModel): boolean {
    return this.isReadyToBeDisplayedInCreateMode
      && !isNullOrUndefined(state.current);
  }

  @Selector()
  static isReadyToBeDisplayedInCreateMode(state: AdminNotificationStateModel): boolean {
    return true;
  }

  @OverrideDefaultAction()
  @Action(AdminNotificationAction.LoadResource)
  loadResource(ctx: StateContext<AdminNotificationStateModel>, action: AdminNotificationAction.LoadResource): Observable<any> {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
    });

    return StoreUtil.dispatchParallelActionAndWaitForSubActionsCompletion(ctx, [
      {
        action: new SharedRoleAction.GetAll(undefined, false, false),
        subActionCompletions: [
          this.actions$.pipe(ofActionCompleted(SharedRoleAction.GetAllSuccess)),
          this.actions$.pipe(ofActionCompleted(SharedRoleAction.GetAllFail)),
        ],
      },
    ]).pipe(
      tap(success => {
        if (success) {
          ctx.dispatch(new AdminNotificationAction.LoadResourceSuccess(action));
        } else {
          ctx.dispatch(new AdminNotificationAction.LoadResourceFail(action));
        }
      }),
    );
  }
}
