import {
  AdminMetadataTypeAction,
  adminMetadataTypeActionNameSpace,
} from "@admin/metadata-type/stores/admin-metadata-type.action";
import {
  HttpClient,
  HttpErrorResponse,
  HttpEventType,
} from "@angular/common/http";
import {Injectable} from "@angular/core";
import {UploadEventModel} from "@deposit/models/upload-event.model";
import {Enums} from "@enums";
import {MetadataType} from "@models";
import {
  Action,
  Actions,
  Selector,
  State,
  StateContext,
  Store,
} from "@ngxs/store";
import {PreservationPlanningSipStateModel} from "@preservation-planning/sip/stores/preservation-planning-sip.state";
import {ApiActionEnum} from "@shared/enums/api-action.enum";
import {ApiResourceNameEnum} from "@shared/enums/api-resource-name.enum";
import {AdminResourceApiEnum} from "@shared/enums/api.enum";
import {LocalStateEnum} from "@shared/enums/local-state.enum";
import {RoutesEnum} from "@shared/enums/routes.enum";
import {DownloadService} from "@shared/services/download.service";
import {Observable} from "rxjs";
import {
  catchError,
  map,
} from "rxjs/operators";
import {
  ApiService,
  defaultResourceStateInitValue,
  HttpStatus,
  isNullOrUndefined,
  isTrue,
  MARK_AS_TRANSLATABLE,
  NotificationService,
  ResourceState,
  ResourceStateModel,
  SolidifyStateError,
  StoreUtil,
  StringUtil,
  urlSeparator,
} from "solidify-frontend";

export interface AdminMetadataTypeStateModel extends ResourceStateModel<MetadataType> {
}

@Injectable()
@State<AdminMetadataTypeStateModel>({
  name: LocalStateEnum.admin_metadataType,
  defaults: {
    ...defaultResourceStateInitValue(),
  },
})
export class AdminMetadataTypeState extends ResourceState<AdminMetadataTypeStateModel, MetadataType> {
  private readonly _FILE_KEY: string = "file";

  constructor(protected apiService: ApiService,
              protected store: Store,
              protected notificationService: NotificationService,
              protected actions$: Actions,
              protected downloadService: DownloadService,
              protected httpClient: HttpClient) {
    super(apiService, store, notificationService, actions$, {
      nameSpace: adminMetadataTypeActionNameSpace,
      routeRedirectUrlAfterSuccessCreateAction: (resId: string) => RoutesEnum.adminMetadataTypeDetail + urlSeparator + resId,
      routeRedirectUrlAfterSuccessUpdateAction: (resId: string) => RoutesEnum.adminMetadataTypeDetail + urlSeparator + resId,
      routeRedirectUrlAfterSuccessDeleteAction: RoutesEnum.adminMetadataType,
      notificationResourceCreateSuccessTextToTranslate: MARK_AS_TRANSLATABLE("admin.metadataType.notification.resource.create"),
      notificationResourceDeleteSuccessTextToTranslate: MARK_AS_TRANSLATABLE("admin.metadataType.notification.resource.delete"),
      notificationResourceUpdateSuccessTextToTranslate: MARK_AS_TRANSLATABLE("admin.metadataType.notification.resource.update"),
    });
  }

  protected get _urlResource(): string {
    return AdminResourceApiEnum.metadataTypes;
  }

  @Selector()
  static isLoading(state: AdminMetadataTypeStateModel): boolean {
    return StoreUtil.isLoadingState(state);
  }

  @Selector()
  static isLoadingWithDependency(state: AdminMetadataTypeStateModel): boolean {
    return this.isLoading(state);
  }

  @Selector()
  static currentTitle(state: AdminMetadataTypeStateModel): string | undefined {
    if (isNullOrUndefined(state.current)) {
      return undefined;
    }
    return state.current.name;
  }

  @Selector()
  static isReadyToBeDisplayed(state: AdminMetadataTypeStateModel): boolean {
    return this.isReadyToBeDisplayedInCreateMode
      && !isNullOrUndefined(state.current);
  }

  @Selector()
  static isReadyToBeDisplayedInCreateMode(state: AdminMetadataTypeStateModel): boolean {
    return true;
  }

  @Action(AdminMetadataTypeAction.Download)
  download(ctx: StateContext<PreservationPlanningSipStateModel>, action: AdminMetadataTypeAction.Download): void {
    let ext = "txt";
    if (action.metadataType.metadataFormat === Enums.MetadataType.MetadataFormatEnum.XML) {
      ext = "xml";
    } else if (action.metadataType.metadataFormat === Enums.MetadataType.MetadataFormatEnum.JSON) {
      ext = "json";
    }
    const fileName = "metadata_type_" + StringUtil.convertToSnakeCase(action.metadataType.name) + "." + ext;
    this.downloadService.download(false, `${this._urlResource}/${action.metadataType.resId}/${ApiResourceNameEnum.SCHEMA}`, fileName);
  }

  @Action(AdminMetadataTypeAction.TestFile)
  testFile(ctx: StateContext<PreservationPlanningSipStateModel>, action: AdminMetadataTypeAction.TestFile): Observable<void> {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
    });
    const formData = new FormData();
    formData.append(this._FILE_KEY, action.file, action.file.name);

    return this.apiService.upload(`${this._urlResource}/${action.id}/${ApiActionEnum.VALIDATE}`, formData)
      .pipe(
        map((event: UploadEventModel) => {
          switch (event.type) {
            case HttpEventType.UploadProgress:
              return;
            case HttpEventType.Response:
              if (event.status === HttpStatus.OK && isTrue(event.body)) {
                ctx.dispatch(new AdminMetadataTypeAction.TestFileSuccess(action));
              } else {
                ctx.dispatch(new AdminMetadataTypeAction.TestFileFail(action));
              }
              return;
            default:
              return;
          }
        }),
        catchError((error: Error | HttpErrorResponse) => {
          ctx.dispatch(new AdminMetadataTypeAction.TestFileFail(action));
          throw new SolidifyStateError(this, error);
        }),
      );
  }

  @Action(AdminMetadataTypeAction.TestFileSuccess)
  testFileSuccess(ctx: StateContext<PreservationPlanningSipStateModel>, action: AdminMetadataTypeAction.TestFileSuccess): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
    this.notificationService.showInformation(MARK_AS_TRANSLATABLE("notification.sip.action.testFile.success"));
  }

  @Action(AdminMetadataTypeAction.TestFileFail)
  testFileFail(ctx: StateContext<PreservationPlanningSipStateModel>, action: AdminMetadataTypeAction.TestFileFail): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
    this.notificationService.showError(MARK_AS_TRANSLATABLE("notification.sip.action.testFile.fail"));
  }
}
