import {
  ChangeDetectionStrategy,
  ChangeDetectorRef,
  Component,
  ElementRef,
  Output,
} from "@angular/core";
import {
  FormBuilder,
  Validators,
} from "@angular/forms";
import {Enums} from "@enums";
import {MetadataType} from "@models";
import {SharedAbstractFormPresentational} from "@shared/components/presentationals/shared-abstract-form/shared-abstract-form.presentational";
import {BaseFormDefinition} from "@shared/models/base-form-definition.model";
import {RegexpUtil} from "@shared/utils/regexp.util";
import {
  BehaviorSubject,
  Observable,
} from "rxjs";
import {
  KeyValue,
  MARK_AS_TRANSLATABLE,
  NotificationService,
  ObservableUtil,
  PropertyName,
  SolidifyValidator,
} from "solidify-frontend";

@Component({
  selector: "dlcm-admin-metadata-type-form",
  templateUrl: "./admin-metadata-type-form.presentational.html",
  styleUrls: ["./admin-metadata-type-form.presentational.scss"],
  changeDetection: ChangeDetectionStrategy.OnPush,
})
export class AdminMetadataTypeFormPresentational extends SharedAbstractFormPresentational<MetadataType> {
  formDefinition: FormComponentFormDefinition = new FormComponentFormDefinition();
  readonly files: string = "files";
  readonly maxSizeFile: number = 25000000;
  uploadedOrChangedFile: boolean = false;

  metadataFormatEnum: KeyValue[] = Enums.MetadataType.MetadataFormatEnumTranslate;

  private readonly _fileUploadBS: BehaviorSubject<File | undefined> = new BehaviorSubject<File | undefined>(undefined);
  @Output("fileUploadChange")
  readonly fileUploadObs: Observable<File | undefined> = ObservableUtil.asObservable(this._fileUploadBS);

  constructor(protected readonly _changeDetectorRef: ChangeDetectorRef,
              protected readonly _elementRef: ElementRef,
              private readonly _fb: FormBuilder,
              private readonly _notificationService: NotificationService) {
    super(_changeDetectorRef, _elementRef);
  }

  protected initNewForm(): void {
    this.form = this._fb.group({
      [this.formDefinition.name]: ["", [Validators.required, SolidifyValidator]],
      [this.formDefinition.description]: ["", [Validators.required, SolidifyValidator]],
      [this.formDefinition.url]: ["", [SolidifyValidator, Validators.pattern(RegexpUtil.url)]],
      [this.formDefinition.metadataFormat]: ["", [SolidifyValidator, Validators.required]],
      [this.formDefinition.version]: ["", [SolidifyValidator, Validators.required]],
      [this.formDefinition.metadataSchema]: ["", [SolidifyValidator]],
    });
  }

  protected bindFormTo(metadataTypes: MetadataType): void {
    this.form = this._fb.group({
      [this.formDefinition.name]: [metadataTypes.name, [Validators.required, SolidifyValidator]],
      [this.formDefinition.description]: [metadataTypes.description, [Validators.required, SolidifyValidator]],
      [this.formDefinition.url]: [metadataTypes.url, [SolidifyValidator, Validators.pattern(RegexpUtil.url)]],
      [this.formDefinition.metadataFormat]: [metadataTypes.metadataFormat, [SolidifyValidator, Validators.required]],
      [this.formDefinition.version]: [metadataTypes.version, [SolidifyValidator, Validators.required]],
      [this.formDefinition.metadataSchema]: [metadataTypes.metadataSchema, [SolidifyValidator]],
    });
  }

  protected treatmentBeforeSubmit(metadataTypes: MetadataType): MetadataType {
    return metadataTypes;
  }

  onFileChange(event: Event & any): void {
    if (event.target.files.length === 0) {
      return;
    }
    const file: File = event.target.files[0];
    if (!this.isFileAccepted(file)) {
      this._notificationService.showWarning(MARK_AS_TRANSLATABLE("admin.metadataType.fileNotSupported"));
      return;
    }
    if (file.size > this.maxSizeFile) {
      this._notificationService.showWarning(MARK_AS_TRANSLATABLE("admin.metadataType.maximumSize"));
      return;
    }
    const fileReader = new FileReader();
    fileReader.onload = (e) => {
      this.form.get(this.formDefinition.metadataSchema).setValue(fileReader.result);
      this._notificationService.showInformation(MARK_AS_TRANSLATABLE("admin.metadataType.readUploadedFileSuccessfully"));
    };
    fileReader.readAsText(file);
    this.uploadedOrChangedFile = true;
  }

  onTestFileUpload(event: Event & any): void {
    if (event.target.files.length === 0) {
      return;
    }
    const file: File = event.target.files[0];
    this._fileUploadBS.next(file);
  }

  isFileAccepted(file: File): boolean {
    if (file.type.startsWith("text") || file.type === "application/json" || file.name.endsWith(".xsd")) {
      return true;
    }
    return false;
  }
}

class FormComponentFormDefinition extends BaseFormDefinition {
  @PropertyName() name: string;
  @PropertyName() description: string;
  @PropertyName() url: string;
  @PropertyName() metadataFormat: string;
  @PropertyName() version: string;
  @PropertyName() metadataSchema: string;
}
