import {
  AdminLanguageAction,
  adminLanguageActionNameSpace,
} from "@admin/language/stores/admin-language.action";
import {Injectable} from "@angular/core";
import {DepositStateModel} from "@deposit/stores/deposit.state";
import {Language} from "@models";
import {
  Action,
  Actions,
  ofActionCompleted,
  Selector,
  State,
  StateContext,
  Store,
} from "@ngxs/store";
import {AdminResourceApiEnum} from "@shared/enums/api.enum";
import {LocalStateEnum} from "@shared/enums/local-state.enum";
import {RoutesEnum} from "@shared/enums/routes.enum";
import {SharedLanguageAction} from "@shared/stores/language/shared-language.action";
import {Observable} from "rxjs";
import {tap} from "rxjs/operators";
import {
  ApiService,
  defaultResourceStateInitValue,
  isNullOrUndefined,
  MARK_AS_TRANSLATABLE,
  NotificationService,
  ResourceState,
  ResourceStateModel,
  StoreUtil,
} from "solidify-frontend";

export interface AdminLanguageStateModel extends ResourceStateModel<Language> {

}

@Injectable()
@State<AdminLanguageStateModel>({
  name: LocalStateEnum.admin_language,
  defaults: {
    ...defaultResourceStateInitValue(),
  },
})
export class AdminLanguageState extends ResourceState<AdminLanguageStateModel, Language> {
  constructor(protected apiService: ApiService,
              protected store: Store,
              protected notificationService: NotificationService,
              protected actions$: Actions) {
    super(apiService, store, notificationService, actions$, {
      nameSpace: adminLanguageActionNameSpace,
      routeRedirectUrlAfterSuccessDeleteAction: RoutesEnum.adminLanguage,
      notificationResourceCreateSuccessTextToTranslate: MARK_AS_TRANSLATABLE("admin.language.notification.resource.create"),
      notificationResourceDeleteSuccessTextToTranslate: MARK_AS_TRANSLATABLE("admin.language.notification.resource.delete"),
      notificationResourceUpdateSuccessTextToTranslate: MARK_AS_TRANSLATABLE("admin.language.notification.resource.update"),
    });
  }

  protected get _urlResource(): string {
    return AdminResourceApiEnum.languages;
  }

  @Selector()
  static isLoading(state: AdminLanguageStateModel): boolean {
    return StoreUtil.isLoadingState(state);
  }

  @Selector()
  static isLoadingWithDependency(state: AdminLanguageStateModel): boolean {
    return this.isLoading(state);
  }

  @Selector()
  static currentTitle(state: AdminLanguageStateModel): string | undefined {
    if (isNullOrUndefined(state.current)) {
      return undefined;
    }
    return state.current.code;
  }

  @Selector()
  static isReadyToBeDisplayed(state: AdminLanguageStateModel): boolean {
    return this.isReadyToBeDisplayedInCreateMode
      && !isNullOrUndefined(state.current);
  }

  @Selector()
  static isReadyToBeDisplayedInCreateMode(state: AdminLanguageStateModel): boolean {
    return true;
  }

  @Action(AdminLanguageAction.LoadResource)
  loadResource(ctx: StateContext<DepositStateModel>, action: AdminLanguageAction.LoadResource): Observable<any> {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
    });

    return StoreUtil.dispatchParallelActionAndWaitForSubActionsCompletion(ctx, [
      {
        action: new SharedLanguageAction.GetAll(null, false, false),
        subActionCompletions: [
          this.actions$.pipe(ofActionCompleted(SharedLanguageAction.GetAllSuccess)),
          this.actions$.pipe(ofActionCompleted(SharedLanguageAction.GetAllFail)),
        ],
      },
    ]).pipe(
      tap(success => {
        if (success) {
          ctx.dispatch(new AdminLanguageAction.LoadResourceSuccess(action));
        } else {
          ctx.dispatch(new AdminLanguageAction.LoadResourceFail(action));
        }
      }),
    );
  }
}
