import {
  ChangeDetectionStrategy,
  ChangeDetectorRef,
  Component,
  ElementRef,
  Input,
  OnInit,
} from "@angular/core";
import {
  FormBuilder,
  Validators,
} from "@angular/forms";
import {
  IndexFieldAlias,
  Label,
  Language,
} from "@models";
import {Store} from "@ngxs/store";
import {SharedAbstractFormPresentational} from "@shared/components/presentationals/shared-abstract-form/shared-abstract-form.presentational";
import {BaseFormDefinition} from "@shared/models/base-form-definition.model";
import {SharedLanguageAction} from "@shared/stores/language/shared-language.action";
import {
  distinctUntilChanged,
  tap,
} from "rxjs/operators";
import {
  isNullOrUndefined,
  PropertyName,
  SolidifyValidator,
} from "solidify-frontend";

@Component({
  selector: "dlcm-admin-index-field-alias-form",
  templateUrl: "./admin-index-field-alias-form.presentational.html",
  styleUrls: ["./admin-index-field-alias-form.presentational.scss"],
  changeDetection: ChangeDetectionStrategy.OnPush,
})
export class AdminIndexFieldAliasFormPresentational extends SharedAbstractFormPresentational<IndexFieldAlias> implements OnInit {
  formDefinition: FormComponentFormDefinition = new FormComponentFormDefinition();

  private _languages: Language[];

  @Input()
  set languages(value: Language[]) {
    this._languages = value;
    if (isNullOrUndefined(this.languages)) {
      return;
    }
    this.languages.forEach(language => {
      if (isNullOrUndefined(this.model) || isNullOrUndefined(this.model.labels)) {
        this.addControl(language.resId);
        return;
      }
      const existingValue = (this.model.labels as Label[]).find(label => label.language.resId === language.resId);
      if (isNullOrUndefined(existingValue)) {
        this.addControl(language.resId);
        return;
      }
      this.addControl(language.resId, existingValue.text);
    });
  }

  constructor(protected readonly _changeDetectorRef: ChangeDetectorRef,
              protected readonly _elementRef: ElementRef,
              private readonly _fb: FormBuilder,
              protected readonly _store: Store) {
    super(_changeDetectorRef, _elementRef);
  }

  protected initNewForm(): void {
    this.form = this._fb.group({
      [this.formDefinition.indexName]: ["", [Validators.required, SolidifyValidator]],
      [this.formDefinition.alias]: ["", [Validators.required, SolidifyValidator]],
      [this.formDefinition.field]: ["", [Validators.required, SolidifyValidator]],
      [this.formDefinition.facet]: [false, [SolidifyValidator]],
      [this.formDefinition.facetMinCount]: [1, []],
      [this.formDefinition.facetLimit]: ["", []],
      [this.formDefinition.facetOrder]: ["", []],
      [this.formDefinition.facetDefaultVisibleValues]: ["", []],
    });
  }

  protected bindFormTo(indexFieldAlias: IndexFieldAlias): void {
    this.form = this._fb.group({
      [this.formDefinition.indexName]: [indexFieldAlias.indexName, [Validators.required, SolidifyValidator]],
      [this.formDefinition.alias]: [indexFieldAlias.alias, [Validators.required, SolidifyValidator]],
      [this.formDefinition.field]: [indexFieldAlias.field, [Validators.required, SolidifyValidator]],
      [this.formDefinition.facet]: [indexFieldAlias.facet, [SolidifyValidator]],
      [this.formDefinition.facetMinCount]: [indexFieldAlias.facetMinCount, []],
      [this.formDefinition.facetLimit]: [indexFieldAlias.facetLimit, []],
      [this.formDefinition.facetOrder]: [indexFieldAlias.facetOrder, []],
      [this.formDefinition.facetDefaultVisibleValues]: [indexFieldAlias.facetDefaultVisibleValues, []],
    });
  }

  ngOnInit(): void {
    super.ngOnInit();
    this._store.dispatch(new SharedLanguageAction.GetAll());
    this.addValidators();
  }

  private addValidators(): void {
    const facetFormControl = this.form.get(this.formDefinition.facet);
    this.subscribe(facetFormControl.valueChanges.pipe(
      distinctUntilChanged(),
      tap(isFacet => {
        this.setValidator(isFacet);
      }),
    ));
    this.setValidator(facetFormControl.value);
  }

  private setValidator(isFacet: boolean): void {
    const facetMinCountFormControl = this.form.get(this.formDefinition.facetMinCount);
    const facetLimitFormControl = this.form.get(this.formDefinition.facetLimit);
    const facetOrderControl = this.form.get(this.formDefinition.facetOrder);

    if (isFacet) {
      facetMinCountFormControl.setValidators([Validators.required, Validators.min(1), SolidifyValidator]);
      facetLimitFormControl.setValidators([Validators.required, Validators.min(1), SolidifyValidator]);
      facetOrderControl.setValidators([Validators.required, Validators.min(0), SolidifyValidator]);
    } else {
      facetMinCountFormControl.setValidators([]);
      facetMinCountFormControl.setValue(undefined);
      facetLimitFormControl.setValidators([]);
      facetLimitFormControl.setValue(undefined);
      facetOrderControl.setValidators([]);
      facetOrderControl.setValue(undefined);
    }
    this.form.updateValueAndValidity();
    this._changeDetectorRef.detectChanges();
  }

  protected treatmentBeforeSubmit(indexFieldAlias: IndexFieldAlias): IndexFieldAlias {

    const newLabels: Label[] = [];
    this.languages.forEach(language => {
      newLabels.push({
        text: indexFieldAlias[language.resId],
        language: language,
      });
    });
    indexFieldAlias.labels = newLabels;

    return indexFieldAlias;
  }

  private addControl(languageResId: string, value: string | undefined = undefined): void {
    if (!isNullOrUndefined(this.form)) {
      this.form.addControl(languageResId, this._fb.control(value));
    }
  }

  get languages(): Language[] {
    return this._languages;
  }
}

class FormComponentFormDefinition extends BaseFormDefinition {
  @PropertyName() indexName: string;
  @PropertyName() alias: string;
  @PropertyName() field: string;
  @PropertyName() facet: string;
  @PropertyName() facetMinCount: string;
  @PropertyName() facetLimit: string;
  @PropertyName() facetOrder: string;
  @PropertyName() facetDefaultVisibleValues: string;
}
