import {
  AdminFundingAgenciesAction,
  adminFundingAgenciesActionNameSpace,
} from "@admin/funding-agencies/stores/admin-funding-agencies.action";
import {
  AdminFundingAgenciesOrgUnitAction,
  adminFundingAgenciesOrgUnitActionNameSpace,
} from "@admin/funding-agencies/stores/organizational-unit/admin-organizational-unit-preservation-policy.action";
import {
  AdminFundingAgenciesOrganizationalUnitState,
  AdminFundingAgenciesOrganizationalUnitStateModel,
} from "@admin/funding-agencies/stores/organizational-unit/admin-organizational-unit-preservation-policy.state";
import {Injectable} from "@angular/core";
import {FundingAgency} from "@models";
import {
  Action,
  Actions,
  ofActionCompleted,
  Selector,
  State,
  StateContext,
  Store,
} from "@ngxs/store";
import {AdminResourceApiEnum} from "@shared/enums/api.enum";
import {LocalStateEnum} from "@shared/enums/local-state.enum";
import {RoutesEnum} from "@shared/enums/routes.enum";
import {SharedOrgUnitAction} from "@shared/stores/organizational-unit/shared-organizational-unit.action";
import {Observable} from "rxjs";
import {tap} from "rxjs/operators";
import {
  ApiService,
  defaultAssociationStateInitValue,
  defaultResourceStateInitValue,
  isNullOrUndefined,
  MARK_AS_TRANSLATABLE,
  NotificationService,
  OverrideDefaultAction,
  ResourceState,
  ResourceStateModel,
  StoreUtil,
  urlSeparator,
} from "solidify-frontend";

export interface AdminFundingAgenciesStateModel extends ResourceStateModel<FundingAgency> {
  admin_fundingAgencies_organizationalUnit: AdminFundingAgenciesOrganizationalUnitStateModel;
}

@Injectable()
@State<AdminFundingAgenciesStateModel>({
  name: LocalStateEnum.admin_fundingAgencies,
  defaults: {
    ...defaultResourceStateInitValue(),
    admin_fundingAgencies_organizationalUnit: {...defaultAssociationStateInitValue()},
  },
  children: [
    AdminFundingAgenciesOrganizationalUnitState,
  ],
})
export class AdminFundingAgenciesState extends ResourceState<AdminFundingAgenciesStateModel, FundingAgency> {
  constructor(protected apiService: ApiService,
              protected store: Store,
              protected notificationService: NotificationService,
              protected actions$: Actions) {
    super(apiService, store, notificationService, actions$, {
      nameSpace: adminFundingAgenciesActionNameSpace,
      routeRedirectUrlAfterSuccessCreateAction: (resId: string) => RoutesEnum.adminFundingAgenciesDetail + urlSeparator + resId,
      routeRedirectUrlAfterSuccessUpdateAction: (resId: string) => RoutesEnum.adminFundingAgenciesDetail + urlSeparator + resId,
      routeRedirectUrlAfterSuccessDeleteAction: RoutesEnum.adminFundingAgencies,
      notificationResourceCreateSuccessTextToTranslate: MARK_AS_TRANSLATABLE("admin.funding-agencies.notification.resource.create"),
      notificationResourceDeleteSuccessTextToTranslate: MARK_AS_TRANSLATABLE("admin.funding-agencies.notification.resource.delete"),
      notificationResourceUpdateSuccessTextToTranslate: MARK_AS_TRANSLATABLE("admin.funding-agencies.notification.resource.update"),
    });
  }

  protected get _urlResource(): string {
    return AdminResourceApiEnum.fundingAgencies;
  }

  @Selector()
  static isLoading(state: AdminFundingAgenciesStateModel): boolean {
    return StoreUtil.isLoadingState(state);
  }

  @Selector()
  static isLoadingWithDependency(state: AdminFundingAgenciesStateModel): boolean {
    return this.isLoading(state)
      || StoreUtil.isLoadingState(state.admin_fundingAgencies_organizationalUnit);
  }

  @Selector()
  static currentTitle(state: AdminFundingAgenciesStateModel): string | undefined {
    if (isNullOrUndefined(state.current)) {
      return undefined;
    }
    return state.current.name;
  }

  @Selector()
  static isReadyToBeDisplayed(state: AdminFundingAgenciesStateModel): boolean {
    return this.isReadyToBeDisplayedInCreateMode
      && !isNullOrUndefined(state.current)
      && !isNullOrUndefined(state.admin_fundingAgencies_organizationalUnit.selected);
  }

  @Selector()
  static isReadyToBeDisplayedInCreateMode(state: AdminFundingAgenciesStateModel): boolean {
    return true;
  }

  @Action(AdminFundingAgenciesAction.LoadResource)
  loadResource(ctx: StateContext<AdminFundingAgenciesStateModel>, action: AdminFundingAgenciesAction.LoadResource): Observable<any> {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
    });

    return StoreUtil.dispatchParallelActionAndWaitForSubActionsCompletion(ctx, [
      {
        action: new SharedOrgUnitAction.GetAll(undefined, false, false),
        subActionCompletions: [
          this.actions$.pipe(ofActionCompleted(SharedOrgUnitAction.GetAllSuccess)),
          this.actions$.pipe(ofActionCompleted(SharedOrgUnitAction.GetAllFail)),
        ],
      },
    ]).pipe(
      tap(success => {
        if (success) {
          ctx.dispatch(new AdminFundingAgenciesAction.LoadResourceSuccess(action));
        } else {
          ctx.dispatch(new AdminFundingAgenciesAction.LoadResourceFail(action));
        }
      }),
    );
  }

  @OverrideDefaultAction()
  @Action(AdminFundingAgenciesAction.Create)
  create(ctx: StateContext<AdminFundingAgenciesStateModel>, action: AdminFundingAgenciesAction.Create): Observable<FundingAgency> {
    return super.internalCreate(ctx, action)
      .pipe(
        tap(model => {
          this.updateSubResource(model, action, ctx)
            .subscribe(success => {
              if (success) {
                ctx.dispatch(new AdminFundingAgenciesAction.CreateSuccess(action, model));
              } else {
                ctx.dispatch(new AdminFundingAgenciesAction.CreateFail(action));
              }
            });
        }),
      );
  }

  @OverrideDefaultAction()
  @Action(AdminFundingAgenciesAction.Update)
  update(ctx: StateContext<AdminFundingAgenciesStateModel>, action: AdminFundingAgenciesAction.Update): Observable<FundingAgency> {
    return super.internalUpdate(ctx, action)
      .pipe(
        tap(model => {
          this.updateSubResource(model, action, ctx)
            .subscribe(success => {
              if (success) {
                ctx.dispatch(new AdminFundingAgenciesAction.UpdateSuccess(action, model));
              } else {
                ctx.dispatch(new AdminFundingAgenciesAction.UpdateFail(action));
              }
            });
        }),
      );
  }

  private updateSubResource(model: FundingAgency, action: AdminFundingAgenciesAction.Create | AdminFundingAgenciesAction.Update, ctx: StateContext<AdminFundingAgenciesStateModel>): Observable<boolean> {
    const fundingAgencyId = model.resId;
    const newOrgUnit = action.modelFormControlEvent.model.organizationalUnits.map(o => o.resId);

    return StoreUtil.dispatchParallelActionAndWaitForSubActionsCompletion(ctx, [{
      action: new AdminFundingAgenciesOrgUnitAction.Update(fundingAgencyId, newOrgUnit),
      subActionCompletions: [
        this.actions$.pipe(ofActionCompleted(AdminFundingAgenciesOrgUnitAction.UpdateSuccess)),
        this.actions$.pipe(ofActionCompleted(AdminFundingAgenciesOrgUnitAction.UpdateFail)),
      ],
    }]);
  }

  @OverrideDefaultAction()
  @Action(AdminFundingAgenciesAction.Delete)
  delete(ctx: StateContext<AdminFundingAgenciesStateModel>, action: AdminFundingAgenciesAction.Delete): Observable<any> {
    const oldOrgUnit = ctx.getState().admin_fundingAgencies_organizationalUnit.selected.map(o => o.resId);

    return StoreUtil.dispatchParallelActionAndWaitForSubActionsCompletion(
      ctx,
      [
        {
          action: new adminFundingAgenciesOrgUnitActionNameSpace.DeleteList(action.resId, oldOrgUnit),
          subActionCompletions: [
            this.actions$.pipe(ofActionCompleted(adminFundingAgenciesOrgUnitActionNameSpace.DeleteListSuccess)),
            this.actions$.pipe(ofActionCompleted(adminFundingAgenciesOrgUnitActionNameSpace.DeleteListFail)),
          ],
        },
      ],
    ).pipe(
      tap(deleteSubResourceWithSuccess => {
        if (deleteSubResourceWithSuccess) {
          super.delete(ctx, action).subscribe();
        } else {
          super.deleteFail(ctx, action);
        }
      }),
    );
  }
}
