import {
  ChangeDetectionStrategy,
  ChangeDetectorRef,
  Component,
  ElementRef,
  Input,
} from "@angular/core";
import {
  FormBuilder,
  Validators,
} from "@angular/forms";
import {
  FundingAgency,
  OrganizationalUnit,
} from "@models";
import {SharedAbstractFormPresentational} from "@shared/components/presentationals/shared-abstract-form/shared-abstract-form.presentational";
import {RoutesEnum} from "@shared/enums/routes.enum";
import {BaseFormDefinition} from "@shared/models/base-form-definition.model";
import {RegexpUtil} from "@shared/utils/regexp.util";
import {
  isNullOrUndefined,
  PropertyName,
  SolidifyValidator,
} from "solidify-frontend";

@Component({
  selector: "dlcm-admin-funding-agencies-form",
  templateUrl: "./admin-funding-agencies-form.presentational.html",
  styleUrls: ["../../../../../../shared/components/presentationals/shared-abstract-form/shared-abstract-form.presentational.scss"],
  changeDetection: ChangeDetectionStrategy.OnPush,
})
export class AdminFundingAgenciesFormPresentational extends SharedAbstractFormPresentational<FundingAgency> {
  formDefinition: FormComponentFormDefinition = new FormComponentFormDefinition();

  @Input()
  selectedOrgUnits: OrganizationalUnit[];

  @Input()
  listOrgUnits: OrganizationalUnit[];

  constructor(protected readonly _changeDetectorRef: ChangeDetectorRef,
              protected readonly _elementRef: ElementRef,
              private readonly _fb: FormBuilder) {
    super(_changeDetectorRef, _elementRef);
  }

  protected initNewForm(): void {
    this.form = this._fb.group({
      [this.formDefinition.acronym]: ["", [Validators.required, SolidifyValidator]],
      [this.formDefinition.name]: ["", [Validators.required, SolidifyValidator]],
      [this.formDefinition.description]: ["", [SolidifyValidator]],
      [this.formDefinition.url]: ["", [SolidifyValidator, Validators.pattern(RegexpUtil.url)]],
      [this.formDefinition.organizationalUnits]: [],
    });
  }

  protected bindFormTo(fundingAgency: FundingAgency): void {
    this.form = this._fb.group({
      [this.formDefinition.acronym]: [fundingAgency.acronym, [Validators.required, SolidifyValidator]],
      [this.formDefinition.name]: [fundingAgency.name, [Validators.required, SolidifyValidator]],
      [this.formDefinition.description]: [fundingAgency.description, [SolidifyValidator]],
      [this.formDefinition.url]: [fundingAgency.url, [SolidifyValidator, Validators.pattern(RegexpUtil.url)]],
      [this.formDefinition.organizationalUnits]: [this.selectedOrgUnits.map(o => o.resId)],
    });
  }

  protected treatmentBeforeSubmit(fundingAgency: FundingAgency): FundingAgency {
    fundingAgency.organizationalUnits = [];
    const listOrgUnitId = this.form.get(this.formDefinition.organizationalUnits).value;
    if (!isNullOrUndefined(listOrgUnitId)) {
      listOrgUnitId.forEach(resId => {
        fundingAgency.organizationalUnits.push({resId: resId});
      });
    }
    return fundingAgency;
  }

  navigateToOrgUnit(organizationalUnit: OrganizationalUnit): void {
    this.navigate([RoutesEnum.adminOrganizationalUnitDetail, organizationalUnit.resId]);
  }
}

class FormComponentFormDefinition extends BaseFormDefinition {
  @PropertyName() acronym: string;
  @PropertyName() name: string;
  @PropertyName() description: string;
  @PropertyName() organizationalUnits: string;
  @PropertyName() url: string;
}
