import {
  ChangeDetectionStrategy,
  ChangeDetectorRef,
  Component,
  ElementRef,
  OnInit,
} from "@angular/core";
import {
  AbstractControl,
  FormBuilder,
  Validators,
} from "@angular/forms";
import {Archive} from "@home/models/archive.model";
import {
  ArchiveACL,
  User,
} from "@models";
import {SharedAbstractFormPresentational} from "@shared/components/presentationals/shared-abstract-form/shared-abstract-form.presentational";
import {BaseFormDefinition} from "@shared/models/base-form-definition.model";
import {BreakpointService} from "@shared/services/breakpoint.service";
import {sharedArchiveActionNameSpace} from "@shared/stores/archive/shared-archive.action";
import {
  PARAM_QUERY_SEARCH,
  SharedArchiveState,
} from "@shared/stores/archive/shared-archive.state";
import {sharedUserActionNameSpace} from "@shared/stores/user/shared-user.action";
import {SharedUserState} from "@shared/stores/user/shared-user.state";
import {Observable} from "rxjs";
import {
  filter,
  tap,
} from "rxjs/operators";
import {
  DateUtil,
  isNullOrUndefined,
  isTruthyObject,
  OrderEnum,
  PropertyName,
  ResourceNameSpace,
  SolidifyValidator,
  Sort,
} from "solidify-frontend";

@Component({
  selector: "dlcm-admin-archive-acl-form",
  templateUrl: "./admin-archive-acl-form.presentational.html",
  styleUrls: ["./admin-archive-acl-form.presentational.scss"],
  changeDetection: ChangeDetectionStrategy.OnPush,
})
export class AdminArchiveAclFormPresentational extends SharedAbstractFormPresentational<ArchiveACL> implements OnInit {
  formDefinition: FormComponentFormDefinition = new FormComponentFormDefinition();

  searchKey: string = PARAM_QUERY_SEARCH;

  sharedArchiveNameSpace: ResourceNameSpace = sharedArchiveActionNameSpace;
  sharedArchiveState: typeof SharedArchiveState = SharedArchiveState;

  sharedUserSort: Sort<User> = {
    field: "lastName",
    order: OrderEnum.ascending,
  };
  sharedUserNameSpace: ResourceNameSpace = sharedUserActionNameSpace;
  sharedUserState: typeof SharedUserState = SharedUserState;
  labelUserCallback: (user: User) => string = user => user.lastName + ", " + user.firstName;

  aipFormControl: AbstractControl;
  userFormControl: AbstractControl;

  constructor(protected readonly _changeDetectorRef: ChangeDetectorRef,
              protected readonly _elementRef: ElementRef,
              private readonly _fb: FormBuilder,
              public readonly breakpointService: BreakpointService) {
    super(_changeDetectorRef, _elementRef);
  }

  protected initNewForm(): void {
    this.form = this._fb.group({
      [this.formDefinition.aipId]: ["", [Validators.required, SolidifyValidator]],
      [this.formDefinition.organizationalUnitId]: ["", [Validators.required, SolidifyValidator]],
      [this.formDefinition.user]: ["", [Validators.required, SolidifyValidator]],
      [this.formDefinition.expiration]: ["", [SolidifyValidator]],
    });
  }

  protected bindFormTo(archiveAcl: ArchiveACL): void {
    this.form = this._fb.group({
      [this.formDefinition.aipId]: [archiveAcl.aipId, [Validators.required, SolidifyValidator]],
      [this.formDefinition.organizationalUnitId]: [archiveAcl.organizationalUnit.resId, [Validators.required, SolidifyValidator]],
      [this.formDefinition.user]: [archiveAcl.user.resId, [Validators.required, SolidifyValidator]],
      [this.formDefinition.expiration]: [DateUtil.convertOffsetDateTimeIso8601ToDate(archiveAcl.expiration), [SolidifyValidator]],
    });
  }

  ngOnInit(): void {
    super.ngOnInit();
    this.aipFormControl = this.form.get(this.formDefinition.aipId);
    this.userFormControl = this.form.get(this.formDefinition.user);

    this.subscribe(this._observableRemoveErrorField(this.aipFormControl));
    this.subscribe(this._observableRemoveErrorField(this.userFormControl));
  }

  private _observableRemoveErrorField(formControl: AbstractControl): Observable<string> {
    return formControl.valueChanges.pipe(
      filter(() => isTruthyObject(formControl.errors)),
      tap(() => {
        this._removeErrorOnAipAndUserField();
      }),
    );
  }

  private _removeErrorOnAipAndUserField(): void {
    this._removeErrorOnFormControl(this.aipFormControl);
    this._removeErrorOnFormControl(this.userFormControl);

    this._changeDetectorRef.detectChanges();
  }

  private _removeErrorOnFormControl(formControl: AbstractControl): void {
    formControl.setErrors(null);
    formControl.markAsTouched();
    formControl.updateValueAndValidity();
  }

  protected treatmentBeforeSubmit(archiveAcl: ArchiveACL): ArchiveACL {
    const archiveAclToSubmit = {
      resId: archiveAcl.resId,
      aipId: this.form.get(this.formDefinition.aipId).value,
      organizationalUnit: {
        resId: this.form.get(this.formDefinition.organizationalUnitId).value,
      },
      user: {
        resId: this.form.get(this.formDefinition.user).value,
      },
      expiration: DateUtil.convertToOffsetDateTimeIso8601(this.form.get(this.formDefinition.expiration).value),
    } as ArchiveACL;
    return archiveAclToSubmit;
  }

  archiveSelected(archive: Archive): void {
    if (isNullOrUndefined(archive)) {
      this.form.get(this.formDefinition.organizationalUnitId).setValue("");
    } else {
      this.form.get(this.formDefinition.organizationalUnitId).setValue(archive.organizationalUnitId);
    }
  }
}

class FormComponentFormDefinition extends BaseFormDefinition {
  @PropertyName() aipId: string;
  @PropertyName() organizationalUnitId: string;
  @PropertyName() user: string;
  @PropertyName() expiration: string;
}
