import {
  ChangeDetectionStrategy,
  ChangeDetectorRef,
  Component,
  ElementRef,
  Input,
} from "@angular/core";
import {
  FormArray,
  FormBuilder,
  Validators,
} from "@angular/forms";
import {SharedAbstractFormPresentational} from "@app/shared/components/presentationals/shared-abstract-form/shared-abstract-form.presentational";
import {BaseFormDefinition} from "@app/shared/models/base-form-definition.model";
import {Enums} from "@enums";
import {
  AccessOrganizationalUnit,
  User,
} from "@models";
import {
  EnumUtil,
  isNullOrUndefined,
  Override,
  PropertyName,
  SolidifyValidator,
} from "solidify-frontend";

@Component({
  selector: "dlcm-user-form",
  templateUrl: "./user-form.presentational.html",
  styleUrls: ["../../../shared/components/presentationals/shared-abstract-form/shared-abstract-form.presentational.scss"],
  changeDetection: ChangeDetectionStrategy.OnPush,
})
export class UserFormPresentational extends SharedAbstractFormPresentational<User> {
  formDefinition: FormComponentFormDefinition = new FormComponentFormDefinition();
  applicationRolesNames: string[] = EnumUtil.convertToArray(Enums.UserApplicationRole.UserApplicationRoleEnum, [Enums.UserApplicationRole.UserApplicationRoleEnum.trusted_client]);

  @Input()
  listOrganizationalUnit: AccessOrganizationalUnit[];

  constructor(protected readonly _changeDetectorRef: ChangeDetectorRef,
              protected readonly _elementRef: ElementRef,
              private readonly _fb: FormBuilder) {
    super(_changeDetectorRef, _elementRef);
  }

  protected bindFormTo(user: User): void {
    this.form = this._fb.group({
      [this.formDefinition.externalUid]: [user.externalUid, [Validators.required, SolidifyValidator]],
      [this.formDefinition.firstName]: [user.firstName, [Validators.required, SolidifyValidator]],
      [this.formDefinition.lastName]: [user.lastName, [Validators.required, SolidifyValidator]],
      [this.formDefinition.homeOrganization]: [user.homeOrganization, [Validators.required, SolidifyValidator]],
      [this.formDefinition.email]: [user.email, [Validators.required, SolidifyValidator]],
      [this.formDefinition.person]: isNullOrUndefined(user.person) ? [] : [user.person.resId],
      [this.formDefinition.accessToken]: [user.accessToken, [Validators.required, SolidifyValidator]],
      [this.formDefinition.refreshToken]: [user.refreshToken, [Validators.required, SolidifyValidator]],
      [this.formDefinition.applicationRole]: [user.applicationRole.resId, [Validators.required, SolidifyValidator]],
    });
  }

  protected initNewForm(): void {
    this.form = this._fb.group({
      [this.formDefinition.externalUid]: ["", [Validators.required, SolidifyValidator]],
      [this.formDefinition.firstName]: ["", [Validators.required, SolidifyValidator]],
      [this.formDefinition.lastName]: ["", [Validators.required, SolidifyValidator]],
      [this.formDefinition.homeOrganization]: ["", [Validators.required, SolidifyValidator]],
      [this.formDefinition.email]: ["", [Validators.required, SolidifyValidator]],
      [this.formDefinition.person]: [""],
      [this.formDefinition.accessToken]: ["", [Validators.required, SolidifyValidator]],
      [this.formDefinition.refreshToken]: ["", [Validators.required, SolidifyValidator]],
      [this.formDefinition.applicationRole]: ["", [Validators.required, SolidifyValidator]],
    });
  }

  protected treatmentBeforeSubmit(user: User): User {
    return user;
  }

  get getApplicationRolesNamesArray(): FormArray {
    return this.form.controls.applicationRoles.value as FormArray;
  }

  @Override()
  protected disableSpecificField(): void {
    this.form.get(this.formDefinition.homeOrganization).disable();
  }
}

class FormComponentFormDefinition extends BaseFormDefinition {
  @PropertyName() externalUid: string;
  @PropertyName() firstName: string;
  @PropertyName() lastName: string;
  @PropertyName() homeOrganization: string;
  @PropertyName() email: string;
  @PropertyName() person: string;
  @PropertyName() accessToken: string;
  @PropertyName() refreshToken: string;
  @PropertyName() applicationRole: string;
}
